package com.digiwin.athena.semc.controller.portal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResponseBody;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.common.enums.EAIServiceNameEnum;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.portal.*;
import com.digiwin.athena.semc.dto.portal.todo.TodoFounderDto;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.portal.TodoList;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.service.portal.LabelSystemSourceService;
import com.digiwin.athena.semc.service.portal.TodoAppSortService;
import com.digiwin.athena.semc.service.portal.TodoListService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Resource;
import javax.validation.Valid;

import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;

/**
 * 待办列表(TodoList)表控制层
 *
 * @author sunyfa
 * @since 2022-12-06 13:48:51
 */
@Slf4j
@RestController
@RequestMapping("/semc/todoList")
public class TodoListController extends BasicController<TodoListService, TodoList> {
    @Resource
    private TodoListService todoListService;

    @Autowired
    LabelSystemSourceService labelSystemSourceService;

    @Autowired
    LabelSystemDataService labelSystemDataService;

    @Resource
    private ESPService espService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private EnvProperties envProperties;

    @Resource
    private TodoAppSortService todoAppSortService;

    /**
     * 拉取待办测试
     *
     * @return 待办数据
     */
    @PostMapping("/queryTodoListTest")
    public ResponseEntity<BaseResultDTO<Map<String, Object>>> queryTodoListTest(@RequestBody TodoListReq.EaiSys todoListReq) {
        // 查询条件
         String qryCondition="";
        Map<String, Object> resultMap = new HashMap<>();
        TodoListResp todoListResp = new TodoListResp();
        try {

            Map<String, String> extHeader = new HashMap<>();
            extHeader.put("digi-userToken", Utils.getUserToken());
            extHeader.put("digi-appToken", envProperties.getAppToken());
            Map<String,Object> parameter=new HashMap<>();
            parameter.put("page_no",1);
            parameter.put("page_size",100);
            parameter.put("tenant_id",Utils.getTenantId());
            Map<String, Object> result = espService.queryByEsp(todoListReq.getEaiSysName(),
                    todoListReq.getEaiSysUid(), "to.do.list.query", extHeader, parameter, qryCondition,
                    null);
            // 没有显示字段时，返回异常
            if (Objects.isNull(result.get("display_field_list"))) {
                log.error(messageUtils.getMessage(I18NKey.TODO_FIELD_MISSING_ERROR));
                todoListResp.setResultCode("3001");
            }
            resultMap=result;
            //todoListResp = todoListService.handleEspResult(user, eaiSys, result, todoListReq.getQryCondition(), null);
            // }
        } catch (RuntimeException bizException) {
            log.error("TodoList handleEspResult exception", bizException);
            todoListResp.setResultCode("3001");
        } catch (Exception e) {
            log.error("TodoList query exception", e);
            todoListResp.setResultCode("3001");
        }
        return ResponseEntityWrapperUtil.wrapperOk(resultMap);
    }


    /**
     * 查询系统的待办
     *
     * @param todoListReq 查询待办列表请求参数
     * @return 待办数据
     */
    @PostMapping("/queryTodoList")
    public ResponseEntity<BaseResultDTO<Map<String, Object>>> queryTodoList(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user,
                                        @Valid @RequestBody TodoListReq todoListReq) {
        log.info("Query todo list param: todoListReq:{}", JSON.toJSONString(todoListReq));

        Map<String, Object> resultMap = new HashMap<>();

        todoListReq.getEaiSysList().stream().forEach(eaiSys -> {
            TodoListResp todoListResp = new TodoListResp();

            try {
                Map<String, String> extHeader = new HashMap<>();
                extHeader.put("digi-userToken", user.getToken());
                extHeader.put("digi-appToken", StringUtils.isBlank(eaiSys.getAppToken()) ? envProperties.getAppToken() : eaiSys.getAppToken());

                // 通过ESP查询三方系统的待办数据
                Map<String, Object> result = espService.queryByEsp(eaiSys.getEaiSysName(),
                        eaiSys.getEaiSysUid(), EAIServiceNameEnum.TODOLIST.getServiceName(), extHeader, null, todoListReq.getQryCondition(),
                        null);

                // 没有显示字段时，返回异常
                if (Objects.isNull(result.get("display_field_list"))) {
                    log.error(messageUtils.getMessage(I18NKey.TODO_FIELD_MISSING_ERROR));
                    todoListResp.setResultCode("3001");
                }

                todoListResp = todoListService.handleEspResult(user, eaiSys, result, todoListReq.getQryCondition(), null);
                // }
            } catch (RuntimeException bizException) {
                log.error("TodoList handleEspResult exception", bizException);
                todoListResp.setResultCode("3001");
            } catch (Exception e) {
                log.error("TodoList query exception", e);
                todoListResp.setResultCode("3001");
            }

            resultMap.put(eaiSys.getEaiSysName(), todoListResp);
        });

        return ResponseEntityWrapperUtil.wrapperOk(resultMap);
    }


    /**
     * 查询三方待办
     *
     * @param todoListReq 查询待办列表请求参数
     * @return 待办数据
     */
    @PostMapping("/label/pageQueryTodo")
    public ResultPageBean pageQueryTodo(@Valid @RequestBody LabelSystemPreReq todoListReq) {
        log.info("pageQueryTodo:{}", JSON.toJSONString(todoListReq));
        TodoListResp todoListResp = new TodoListResp();
        try {
            todoListResp = todoListService.pageQueryTodo(todoListReq);
        } catch (Exception e) {
            log.error("pageQueryTodo query exception", e);
            todoListResp.setResultCode("3001");
        }
        return ResultPageBean.success(ResponseBody.getInstance(todoListResp, PageInfo.getPageInfo(todoListReq.getPageNum(), todoListReq.getPageSize(), todoListResp.getTotalResults())));
    }


    /**
     * 根据SSO配置中的AppId查询应用下，鼎捷云账号对应地端用户的三方待办信息
     *
     * @param todoListReq 查询待办列表请求参数
     * @return 待办数据
     */
    @PostMapping("/todo/selectTodoList")
    public ResultPageBean selectTodoList(@Valid @RequestBody PreSystemToDoReq todoListReq) {
        log.info("selectTodoList:{}", JSON.toJSONString(todoListReq));
        TodoListResp todoListResp = new TodoListResp();
        try {
            todoListResp = todoListService.selectTodoList(todoListReq);
        } catch (Exception e) {
            log.error("selectTodoList query exception", e);
            todoListResp.setResultCode("3001");
        }
        return ResultPageBean.success(ResponseBody.getInstance(todoListResp, PageInfo.getPageInfo(todoListReq.getPageNum(), todoListReq.getPageSize(), todoListResp.getTotalResults())));
    }

    /**
     * 获取跳转的url
     *
     * @param reimburseReq
     * @return
     */
    @Operation(method = "getTodoListUrl", description = "获取跳转url")
    @PostMapping("/getTodoListUrl")
    public ResponseEntity<?> getTodoListUrl(@RequestBody @Valid ReimburseReq reimburseReq) {
        return todoListService.getTodoListUrl(reimburseReq);
    }

    /**
     * 查询首页待办组件三方待办未处理数及未读数
     *
     * @return 查询结果
     */
    @GetMapping("/queryToDoCount")
    public ResponseEntity<BaseResultDTO<JSONObject>> queryToDoCount() {
        try {
            JSONObject jsonObject = new JSONObject();
            // 查询当前租户配置的数据源应用
            List<Long> dataIds = labelSystemSourceService.selectPreSystem(Constants.DataTypeEnum.TYPE_THIRD_TODO.getVal(), Constants.LabelTypeEnum.SYSTEM_PRE.getVal());
            List<LabelSystemData> systemDataList = labelSystemDataService.queryDataInfoBy(dataIds,false);
            if (CollectionUtils.isEmpty(systemDataList)) {
                jsonObject.put("toDoCount", 0);
                jsonObject.put("unReadCount", 0);
                return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
            }
            jsonObject = todoListService.queryToDoCount(systemDataList);
            return ResponseEntityWrapperUtil.wrapperOk(jsonObject);
        } catch (Exception e) {
            log.error("query to do count error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/todoList/queryToDoCount");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 同步三方待办
     *
     * @param req 查询待办列表请求参数
     * @return 待办数据
     */
    @PostMapping("/todo/syncThirdTodoData")
    public ResponseEntity<?> syncThirdTodoData(@Valid @RequestBody PreSystemToDoUpdateReq req) {
        return todoListService.syncThirdTodoData(req);
    }

    /**
     * 用户保存三方待办应用排序
     * @param req 应用id集合
     * @return
     */
    @PostMapping("/todo/saveAppSort")
    public ResponseEntity<BaseResultDTO<Boolean>> saveAppSort(@RequestBody List<String> req) {
        return ResponseEntityWrapperUtil.wrapperOk(todoAppSortService.save(req));
    }

    /**
     * 查询三方待办统计信息
     * @param req
     * @return
     */
    @PostMapping("/todo/selectTodoCount")
    public ResponseEntity<BaseResultDTO<TodoCountResp>> selectTodoCount(@Valid @RequestBody PreSystemToDoCountReq req) {
        return ResponseEntityWrapperUtil.wrapperOk(todoListService.selectTodoCount(req));
    }

    /**
     * 搜索三方待办应用
     */
    @PostMapping("/todo/searchApp")
    public ResponseEntity<BaseResultDTO<List<LabelSystemData>>> searchTodoApp(@Valid @RequestBody PreSystemToDoSearchAppReq req) {
        return ResponseEntityWrapperUtil.wrapperOk(todoListService.searchTodoApp(req));
    }

}
