package com.digiwin.athena.semc.controller.router;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContext;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.dto.erpsso.AuthorizeDTO;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.util.HttpRequestDeviceUtils;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@RestController
@RequestMapping(value = "/semc/router")
public class RoutingAddressController {

    @Resource
    private EnvProperties envProperties;

    @Resource
    private RestTemplate restTemplate;

    @Autowired
    private AppLinkService appLinkService;

    @Resource
    private IamService iamService;

    private static final String WE_COM = "weCom";
    private static final String SOURCE_TYPE = "enterprise_Wechat";
    private static final String RESPONSE_TYPE = "token";
    //移动端登录接口
    private static final String MOBILE_SSO_PATH = "/mobile/v1/sso/grantOauthAccess";
    //SSO单点登录接口
    private static final String PC_SSO_PATH = "/api/iam/v2/identity/oauth/token/access";
    //去往详情页
    private static final String REDIRECT_URL_TEMPLATE = "%s/sso-login?userToken=%s&routerLink=/entrance/third-login&proxyToken=%s&taskId=%s&dwLang=%s";
    //去往首页
    private static final String REDIRECT_URL_TEMPLATE_HOME = "%s/sso-login?userToken=%s&routerLink=/&dwLang=%s";
    private static final String HOME_PAGE_REDIRECT_TEMPLATE = "%s";

    /**
     * @param request:
     * @param code:
     * @param taskId:
     * @param response:
     * @Author: shuai
     * @return: org.springframework.http.ResponseEntity<?>
     **/
    @Operation(method = "routingDistribution", description = "路由分发")
    @GetMapping("/routingDistribution")
    public ResponseEntity<?> routingJudgment(HttpServletRequest request,
                                             @RequestParam("code") String code,
                                             @RequestParam("taskId") String taskId,
                                             @RequestParam("appId") String appId,
                                             @RequestParam("tenantId") String tenantId,
                                             HttpServletResponse response) throws IOException {

        if (HttpRequestDeviceUtils.isMobileDevice(request)) {
            String url = handleMobileTask(code, tenantId, appId, taskId);
            response.sendRedirect(url);
        } else {
            handlePcLogin(request, code, tenantId, appId, taskId, response);
        }
        return null;
    }

    @Operation(method = "enterpriseWeChatLogin", description = "企微跳转")
    @GetMapping("/enterpriseWeChatLogin")
    public ResponseEntity<?> routingToHomepage(HttpServletRequest request, @RequestParam("appId") String appId, @RequestParam("code") String code, @RequestParam("tenantId") String tenantId, HttpServletResponse response) throws IOException {
        if (HttpRequestDeviceUtils.isMobileDevice(request)) {
            String url = handleMobileLogin(code, tenantId, appId);
            response.sendRedirect(url);
        } else {
            handlePcLoginHomePage(request, code, tenantId, response, appId);
        }
        return null;
    }


    //移动端跳转任务
    private String handleMobileTask(String code, String tenantId, String appId, String taskId) {
        String url = String.format("%s%s?code=%s&tenantId=%s&appId=%s&taskId=%s&targetTenantId=%s&sourceType=%s&isHideAppBar=false&canBack=false",
                envProperties.getMobileUrl(), MOBILE_SSO_PATH, code, tenantId, appId, taskId, tenantId, SOURCE_TYPE);
        ResponseEntity<JSONObject> exchange = restTemplate.exchange(url, HttpMethod.GET, null, JSONObject.class);
        JSONObject data = exchange.getBody().getJSONObject("data");
        return data.getString("data");
    }


    //移动端登录首页
    private String handleMobileLogin(String code, String tenantId, String appId) {
        String url = String.format("%s%s?code=%s&tenantId=%s&appId=%s&sourceType=%s&isHideAppBar=false",
                envProperties.getMobileUrl(), MOBILE_SSO_PATH, code, tenantId, appId, SOURCE_TYPE);
        ResponseEntity<JSONObject> exchange = restTemplate.exchange(url, HttpMethod.GET, null, JSONObject.class);
        JSONObject data = exchange.getBody().getJSONObject("data");
        return data.getString("data");
    }


    //PC逻辑
    private ResponseEntity<?> handlePcLogin(HttpServletRequest request, String code, String tenantId, String appId, String taskId, HttpServletResponse response) throws IOException {
        ResponseEntity<JSONObject> respEntity = getJsonObjectResponseEntity(code, tenantId, appId);

        if (respEntity.getStatusCode() == HttpStatus.OK && respEntity.getBody() != null) {
            String token = respEntity.getBody().getString(RESPONSE_TYPE);
            if (StringUtils.isNotEmpty(token)) {
                String redirectUrl = String.format(REDIRECT_URL_TEMPLATE, envProperties.getHomePageUrl(), token, "", taskId, request.getLocale());
                response.sendRedirect(redirectUrl);
                log.info("Authorization successful, redirecting to details.");
                return null;
            }
        }
        redirectToHomePage(response);
        return null;
    }

    //组装数据
    private ResponseEntity<JSONObject> getJsonObjectResponseEntity(String code, String tenantId, String appId) {
        //组装请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        //组装body
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("tenantId", tenantId);
        requestMap.put("code", code);
        requestMap.put("oauthType", WE_COM);
        requestMap.put("appId", appId);

        String url = envProperties.getIamUri() + PC_SSO_PATH;
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        //发送请求
        ResponseEntity<JSONObject> respEntity = null;
        try {
            respEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, JSONObject.class);
        } catch (RestClientException e) {
            log.error("Failed to call IAM API: {}", e.getMessage(), e);
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
        }
        return respEntity;
    }

    private void redirectToHomePage(HttpServletResponse response) throws IOException {
        //授权失败
        String homePageUrl = String.format(HOME_PAGE_REDIRECT_TEMPLATE, envProperties.getHomePageUrl());
        response.sendRedirect(homePageUrl);
        log.info("Authorization failed, redirecting to home page.");
    }


    //PC去往首页逻辑
    private ResponseEntity<?> handlePcLoginHomePage(HttpServletRequest request, String code, String tenantId, HttpServletResponse response, String appId) throws IOException {
        //组装请求头
        ResponseEntity<JSONObject> respEntity = getJsonObjectResponseEntity(code, tenantId, appId);

        if (respEntity.getStatusCode() == HttpStatus.OK && respEntity.getBody() != null) {
            String token = respEntity.getBody().getString(RESPONSE_TYPE);
            if (StringUtils.isNotEmpty(token)) {
                String redirectUrl = String.format(REDIRECT_URL_TEMPLATE_HOME, envProperties.getHomePageUrl(), token, request.getLocale());
                response.sendRedirect(redirectUrl);
                log.info("Authorization to Homepage successful, redirecting to details.");
                return null;
            }
        }
        redirectToHomePage(response);
        return null;
    }


    /**
     * 三方应用企微消息，点击跳转路由
     *
     * @param request          请求
     * @param code             企微成员授权获取到的code
     * @param appId            应用企微agentid
     * @param tenantId         租户id
     * @param messageAppCode   应用code，如HRM
     * @param messageUrl       pc端消息链接
     * @param mobileMessageUrl 移动端消息链接
     * @param response         详情
     * @return 返回
     * @throws IOException 异常
     */
    @Operation(method = "thirdMessageRouting", description = "路由分发")
    @GetMapping("/thirdMessageRouting")
    public ResponseEntity<?> thirdMessageRouting(HttpServletRequest request, @RequestParam("code") String code,
                                                 @RequestParam("appId") String appId,
                                                 @RequestParam("tenantId") String tenantId,
                                                 @RequestParam("messageAppCode") String messageAppCode,
                                                 @RequestParam("messageUrl") String messageUrl,
                                                 @RequestParam("mobileMessageUrl") String mobileMessageUrl,
                                                 HttpServletResponse response) throws Exception {
        log.info("thirdMessageRouting routing param. code:{}, appId:{}, tenantId:{}, messageAppCode:{}", code, appId, tenantId, messageAppCode);
        String callbackUrl = StringUtils.isNotBlank(messageUrl) ? messageUrl : mobileMessageUrl;
        // 获取用户的授权，DAP先去企微里面查询到成员微信账号，然后再通过微信账号查询DAP（关系维护在om）获取到成员微信账号和用户id的绑定关系获取到用户信息
        ResponseEntity<JSONObject> respEntity = getJsonObjectResponseEntity(code, tenantId, appId);
        if (respEntity.getStatusCode() == HttpStatus.OK && respEntity.getBody() != null) {
            String userToken = respEntity.getBody().getString(RESPONSE_TYPE);
            if (StringUtils.isEmpty(userToken)) {
                log.error("thirdMessageRouting query user token return empty. code:{}, tenantId:{}, appId:{}", code, tenantId, appId);
                return null;
            }
            // 查询应用的sso配置
            AuthoredUser authoredUser = new AuthoredUser();
            authoredUser.setToken(userToken);
            authoredUser.setTenantId(tenantId);
            AppAuthContextHolder.getContext().setAuthoredUser(authoredUser);
            List<AppLinkDTO> appLinkDTOList = appLinkService.queryManageListSync(AppAuthContextHolder.getContext().getAuthoredUser());
            AppLinkDTO appLinkDTO = appLinkDTOList.stream().filter(x -> x.getCode().equals(messageAppCode)).findFirst().orElse(null);
            if (ObjectUtil.isNull(appLinkDTO)) {
                log.error("thirdMessageRouting query appInfo return empty. userToken:{}, tenantId:{}", userToken, tenantId);
                return null;
            }
            // 构建回调地址
            String redirectUrl = appLinkService.parseJumpUrl(callbackUrl, appLinkDTO);

            // 获取用户sso授权码
            AuthorizeDTO authorizeDTO = iamService.queryAuthorizeInfo(appLinkDTO.getApplicationAppId(), appLinkDTO.getCallBackUrl());
            if (ObjectUtil.isNotEmpty(authorizeDTO)) {
                redirectUrl = redirectUrl + "&code=" + authorizeDTO.getCode();
            }
            log.info("thirdMessageRouting sso redirect url:{}", redirectUrl);
            response.sendRedirect(redirectUrl);
        }
        // 授权失败
        log.info("thirdMessageRouting redirect url:{}", callbackUrl);
        response.sendRedirect(callbackUrl);
        return null;
    }
}
