package com.digiwin.athena.semc.controller.sso;

import com.google.common.collect.Lists;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.PreinstalledApplicationTypeEnum;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.erpsso.AddSsoInfoReq;
import com.digiwin.athena.semc.dto.erpsso.QuerySsoListReq;
import com.digiwin.athena.semc.dto.erpsso.RegisterDTO;
import com.digiwin.athena.semc.dto.erpsso.SsoBindReq;
import com.digiwin.athena.semc.dto.message.QueryThirdMessageConfigReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.applink.AppLinkGroupDTO;
import com.digiwin.athena.semc.entity.applink.AppLinkListDTO;
import com.digiwin.athena.semc.entity.message.ThirdMessageConfig;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.entity.sso.ErpSsoInfo;
import com.digiwin.athena.semc.entity.sso.ThirdSsoInfo;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.message.ThirdSystemMessageService;
import com.digiwin.athena.semc.service.portal.IPreinstalledApplicationService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.service.sso.IErpSsoInfoService;
import com.digiwin.athena.semc.service.sso.IThirdSsoInfoService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.portal.AppExistVO;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.validation.Valid;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;

/**

 *
 * @author zhangtuo
 * @since 2022-10-28
 */

@Tag(name = "稳态单点登录管理")
@RestController
@RequestMapping("/tenant/erpSsoInfo")
@Slf4j
public class ErpSsoInfoController extends BasicController<IErpSsoInfoService, ErpSsoInfo> {

    private static final Logger logger = LoggerFactory.getLogger(ErpSsoInfoController.class);

    @Autowired
    IErpSsoInfoService erpSsoInfoService;

    @Autowired
    IThirdSsoInfoService thirdSsoInfoService;

    @Resource
    private IPreinstalledApplicationService preinstalledApplicationService;

    @Resource
    private IamService iamService;

    @Autowired
    private AppLinkService appLinkService;

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    LabelSystemDataService labelSystemDataService;

    @Resource
    private ThirdSystemMessageService thirdSystemMessageService;

    @Resource
    private EnvProperties envProperties;

    @Resource
    private RestTemplate restTemplate;


    /**
     * 根据租户id查询稳态配置列表
     *
     * @return
     */
    @Operation(summary = "根据租户id查询稳态配置列表")
    @GetMapping("/listByTenantId")
    public ResponseEntity<BaseResultDTO<List<ErpSsoInfo>>> querySsoInfoList(@RequestParam(required = false, value = "protocolType") Integer protocolType) {
        try {
            String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
            List<ErpSsoInfo> ssoInfoList = erpSsoInfoService.querySsoList(tenantId, protocolType);
            return ResponseEntityWrapperUtil.wrapperOk(ssoInfoList);
        } catch (Exception e) {
            log.error("ErpSsoInfoController query sso info list error", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 保存稳态配置信息
     *
     * @param addSsoInfoReq 请求实体类
     * @return
     * @throws Exception
     */
    @Operation(summary = "保存稳态配置信息")
    @PostMapping("/saveErpInfo")
    public ResponseEntity<BaseResultDTO<ErpSsoInfo>> saveErpSsoInfo(@RequestBody @Valid AddSsoInfoReq addSsoInfoReq) throws Exception {
        logger.info("save erp sso | param:[" + addSsoInfoReq + "]");
        /*if ((Constants.BindFlagEnum.HAND.getFlag().equals(addSsoInfoReq.getUserBindFlag()) || Constants.BindFlagEnum.AUTO.getFlag().equals(addSsoInfoReq.getUserBindFlag()))
                && StringUtils.isBlank(addSsoInfoReq.getAppToken())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "appToken cannot be null");
        }*/
        // 动态参数校验
        if (dynamicParamValid(addSsoInfoReq)) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.DYNAMIC_PARAM_ILLEGAL));
        }
        // 校验应用的重复性
        List<ErpSsoInfo> erpSsoInfoList = erpSsoInfoService.existByNameOrAppToken(addSsoInfoReq.getId(), addSsoInfoReq.getCode(), addSsoInfoReq.getName(), addSsoInfoReq.getAppToken());
        if (CollectionUtils.isNotEmpty(erpSsoInfoList)) {
            if (addSsoInfoReq.getCode().equals(erpSsoInfoList.get(0).getCode())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_CODE_REPEAT));
            }
            if (addSsoInfoReq.getName().equals(erpSsoInfoList.get(0).getName())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_NAME_REPEAT));
            }
            if (StringUtils.isNotBlank(addSsoInfoReq.getAppToken()) && addSsoInfoReq.getAppToken().equals(erpSsoInfoList.get(0).getAppToken())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_TOKEN_REPEAT));
            }
        }
        List<ThirdSsoInfo> thirdSsoInfoList = thirdSsoInfoService.existThirdSso(null, addSsoInfoReq.getCode(), addSsoInfoReq.getName(), addSsoInfoReq.getAppToken());
        if (CollectionUtils.isNotEmpty(thirdSsoInfoList)) {
            if (addSsoInfoReq.getCode().equals(thirdSsoInfoList.get(0).getAppCode())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.BS_APP_CODE_REPEAT));
            }
            if (addSsoInfoReq.getName().equals(thirdSsoInfoList.get(0).getAppName())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.BS_APP_NAME_REPEAT));
            }
            if (StringUtils.isNotBlank(addSsoInfoReq.getAppToken()) && addSsoInfoReq.getAppToken().equals(thirdSsoInfoList.get(0).getAppToken())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.BS_APP_TOKEN_REPEAT));
            }
        }
        List<PreinstalledApplication> applicationList = preinstalledApplicationService.queryPreinstalledAppList(AppAuthContextHolder.getContext().getAuthoredUser());
        for (PreinstalledApplication application : applicationList) {
            if (application.getApplicationName().equals(addSsoInfoReq.getName())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.PRE_APP_NAME_REPEAT));
            }
        }

        ErpSsoInfo erpSsoInfo = new ErpSsoInfo();
        boolean updateFlag = Boolean.FALSE;
        // 修改或新增应用，需要先调IAM更新或注册应用
        // 修改应用
        if (addSsoInfoReq.getId() != null) {
            // 先判断应用是否存在
            erpSsoInfo = erpSsoInfoService.querySsoInfo(addSsoInfoReq.getId());
            if (erpSsoInfo == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_QUERY_NOT_FOUND));
            }
            // 应用id不能修改
            if (!erpSsoInfo.getCode().equals(addSsoInfoReq.getCode())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_CODE_CHANGE));
            }

            // 如果应用是自动模式或手动模式，则需要注册应用或更新应用
            if (Constants.BindFlagEnum.AUTO.getFlag().equals(addSsoInfoReq.getUserBindFlag()) || Constants.BindFlagEnum.HAND.getFlag().equals(addSsoInfoReq.getUserBindFlag())) {
                RegisterDTO registerReq = new RegisterDTO();
                try {
                    // 如果appSid为空，则注册应用
                    if (erpSsoInfo.getAppSid() == null) {
                        registerCsApp(addSsoInfoReq);
                    }
                    // 如果appSid不为空，且修改了应用code、描述则更新应用
                    else if (!StringUtils.equalsIgnoreCase(erpSsoInfo.getCode(), addSsoInfoReq.getCode()) || !StringUtils.equalsIgnoreCase(erpSsoInfo.getAppDesc(), addSsoInfoReq.getDesc())) {
                        registerReq = buildRegisterReq(erpSsoInfo.getAppSid(), addSsoInfoReq.getCode(), addSsoInfoReq.getName(), addSsoInfoReq.getDesc(), addSsoInfoReq.getAppToken());
                        iamService.updateAppInfoForIAM(registerReq);
                        updateFlag = Boolean.TRUE;
                    }
                } catch (Exception e) {
                    logger.error("ErpSsoInfoController saveErpInfo invoke iam error error, registerReq:{}, AddSsoInfoReq:{}", registerReq, addSsoInfoReq, e);
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.ADAPT_SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
                }
            }
        } else {  // 新增应用
            // 如果应用是自动模式或手动模式，则需要注册应用
            if (Constants.BindFlagEnum.AUTO.getFlag().equals(addSsoInfoReq.getUserBindFlag()) || Constants.BindFlagEnum.HAND.getFlag().equals(addSsoInfoReq.getUserBindFlag())) {
                try {
                    registerCsApp(addSsoInfoReq);
                } catch (Exception e) {
                    logger.error("ErpSsoInfoController saveErpInfo invoke iam error error, addSsoInfoReq:{}", addSsoInfoReq, e);
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.ADAPT_SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
                }
            }
        }

        // 保存应用
        try {
            Long ssoInfoId = erpSsoInfoService.saveOrUpdateSsoInfo(addSsoInfoReq);
            ErpSsoInfo erpSsoInfoResp = erpSsoInfoService.querySsoInfo(ssoInfoId);
            if (Constants.BindFlagEnum.NO.getFlag().equals(addSsoInfoReq.getUserBindFlag()) || Constants.BindFlagEnum.OUTSIDE.getFlag().equals(addSsoInfoReq.getUserBindFlag())) {
                erpSsoInfoResp.setAppId("");
                erpSsoInfoResp.setAppSecret("");
            }
            return ResponseEntityWrapperUtil.wrapperOk(erpSsoInfoResp);
        } catch (Exception e) {
            // 保存应用失败，则把应用信息还原
            if (updateFlag) {
                RegisterDTO registerReq = buildRegisterReq(erpSsoInfo.getAppSid(), erpSsoInfo.getCode(), erpSsoInfo.getName(), erpSsoInfo.getAppDesc(), erpSsoInfo.getAppToken());
                iamService.updateAppInfoForIAM(registerReq);
            }
            logger.info("ErpSsoInfoController save erp sso occur error. addSsoInfoReq:{}", addSsoInfoReq, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 动态参数校验
     *
     * @param addSsoInfoReq 请求体
     * @return
     */
    private boolean dynamicParamValid(AddSsoInfoReq addSsoInfoReq) {
        String dynamicParam = addSsoInfoReq.getExt();
        if (StringUtils.isNotBlank(dynamicParam)) {
            try {
                JSONObject jsonObject = JSONObject.parseObject(dynamicParam);
                if (jsonObject.isEmpty()) {
                    return true;
                }
            } catch (Exception e) {
                return true;
            }
        }
        return false;
    }

    /**
     * 应用注册
     *
     * @param addSsoInfoReq 请求入参
     * @throws Exception
     */
    private void registerCsApp(AddSsoInfoReq addSsoInfoReq) throws Exception {
        RegisterDTO registerReq = buildRegisterReq(null, addSsoInfoReq.getCode(), addSsoInfoReq.getName(), addSsoInfoReq.getDesc(), addSsoInfoReq.getAppToken());
        RegisterDTO registerResp = thirdSsoInfoService.registerApp(registerReq);
        addSsoInfoReq.setAppSid(registerResp.getSid());
        addSsoInfoReq.setAppId(registerResp.getId());
        addSsoInfoReq.setAppSecret(registerResp.getSecret());
    }

    /**
     * 构建调用IAM接口请求对象
     *
     * @param appSid   appSId
     * @param name     name
     * @param desc     desc
     * @param appToken appToken
     * @return
     */

    private RegisterDTO buildRegisterReq(Long appSid, String code, String name, String desc, String appToken) {
        RegisterDTO registerReq = new RegisterDTO();
        registerReq.setSid(appSid);  // 新增配置时，appSid为空
        // registerReq.setName(name);
        registerReq.setName(code);
        registerReq.setDescription(desc);
        registerReq.setAppToken(appToken);
        registerReq.setCallbackUrl(String.format(Constants.CS_APP_CALLBACK_URL, code));
        return registerReq;
    }

    /**
     * 修改生效状态
     *
     * @param ssoInfoId 配置id
     * @param status    状态
     * @return
     */
    @Operation(summary = "修改生效状态")
    @GetMapping("/updateStatus")
    public ResponseEntity<BaseResultDTO<Boolean>> editSsoValidateStatus(@RequestParam("ssoInfoId") Long ssoInfoId,
                                                                        @RequestParam("status") Integer status) {
        try {
            if (ssoInfoId == null || status == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
            }
            ErpSsoInfo erpSsoInfo = erpSsoInfoService.querySsoInfo(ssoInfoId);
            if (erpSsoInfo == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_QUERY_NOT_FOUND));
            }
            //如果是停用，校验是否有数据源引用
            if (status == 0) {
                List<String> systemDataList = labelSystemDataService.getDataNameList(Integer.parseInt(PreinstalledApplicationTypeEnum.CS_APPLICATION.getValue()), ssoInfoId + "");
                if (CollectionUtils.isNotEmpty(systemDataList)) {
                    String dataNames = systemDataList.stream().collect(Collectors.joining("、"));
                    String error = String.format(messageUtils.getMessage("error.message.system.sso.erp.fail"), dataNames);
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
                }
            }
            int ssoId = erpSsoInfoService.updateSsoStatus(ssoInfoId, status);
            return ResponseEntityWrapperUtil.wrapperOk(ssoId > 0 ? Boolean.TRUE : Boolean.FALSE);
        } catch (Exception e) {
            logger.info("ErpSsoInfoController update c/s app status error. ssoInfoId:{}, status:{}", ssoInfoId, status, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 删除配置
     *
     * @param ssoInfoId 配置id
     * @return
     */
    @Operation(summary = "删除配置")
    @GetMapping("/deleteById")
    @Transactional
    public ResponseEntity<BaseResultDTO<Boolean>> deleteSsoInfo(@RequestParam("ssoInfoId") Long ssoInfoId) throws Exception {
        try {
            if (ssoInfoId == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
            }
            ErpSsoInfo erpSsoInfo = erpSsoInfoService.querySsoInfo(ssoInfoId);
            if (erpSsoInfo == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_QUERY_NOT_FOUND));
            }
            if (Constants.VALID_STATUS_ENABLE.equals(erpSsoInfo.getValidStatus())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.CONFIG_IN_EFFECT));
            }
            //校验是否有数据源引用
            List<String> systemDataList = labelSystemDataService.getDataNameList(Integer.parseInt(PreinstalledApplicationTypeEnum.CS_APPLICATION.getValue()), ssoInfoId + "");
            if (CollectionUtils.isNotEmpty(systemDataList)) {
                String dataNames = systemDataList.stream().collect(Collectors.joining("、"));
                String error = String.format(messageUtils.getMessage("error.message.system.sso.erp.delfail"), dataNames);
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            // 校验是否在异构消息中配置
            QueryThirdMessageConfigReq queryImportRecordReq = new QueryThirdMessageConfigReq();
            queryImportRecordReq.setAppPrimaryId(String.valueOf(ssoInfoId));
            queryImportRecordReq.setAppSource(ApplicationTypeEnum.CS_APPLICATION.getType());
            PageInfoResp<ThirdMessageConfig> thirdMessageConfigPageInfoResp = thirdSystemMessageService.queryMessageConfigPage(queryImportRecordReq);
            if (CollectionUtils.isNotEmpty(thirdMessageConfigPageInfoResp.getList())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage("error.message.system.sso.del.fail"));
            }
            Boolean flag = erpSsoInfoService.removeById(ssoInfoId);
            erpSsoInfoService.deleteErpSsoParams(ssoInfoId);
            return ResponseEntityWrapperUtil.wrapperOk(flag);
        } catch (Exception e) {
            logger.info("ErpSsoInfoController delete c/s app error. ssoInfoId:{}", ssoInfoId, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 根据应用名称查询应用信息
     *
     * @return
     */
    @Operation(summary = "根据应用名称查询应用信息")
    @PostMapping("/queryByName")
    public ResponseEntity<BaseResultDTO<AppLinkDTO>> querySsoInfoByName(@RequestBody QuerySsoListReq querySsoListReq) {
        if (CollectionUtils.isEmpty(querySsoListReq.getAppNameList())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
        }
        try {
            AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
            AppLinkListDTO appLinkListDTO = appLinkService.queryLinkList(user);
            List<AppLinkGroupDTO> appLinkList = appLinkListDTO.getAll();

            List<AppLinkDTO> allAppLinkList = Lists.newArrayList();
            for (AppLinkGroupDTO appLinkGroupDTO : appLinkList) {
                List<AppLinkDTO> linkList = appLinkGroupDTO.getLinks();
                allAppLinkList.addAll(linkList);
            }
            if (CollectionUtils.isEmpty(allAppLinkList)) {
                return ResponseEntityWrapperUtil.wrapperFail(HttpStatus.INTERNAL_SERVER_ERROR.value(), messageUtils.getMessage(I18NKey.QUERY_NOT_FOUND));
            }

            // 预设应用优先匹配
            Map<String, AppLinkDTO> appNameMap = allAppLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getName, Function.identity(), (a, b) -> a));
            AppLinkDTO appLinkDTO = null;
            for (String appName : querySsoListReq.getAppNameList()) {
                appLinkDTO = appNameMap.get(appName);
                if (appLinkDTO != null) {
                    break;
                }
            }
            if (appLinkDTO == null) {
                return ResponseEntityWrapperUtil.wrapperFail(HttpStatus.INTERNAL_SERVER_ERROR.value(), messageUtils.getMessage(I18NKey.QUERY_NOT_FOUND));
            } else {
                return ResponseEntityWrapperUtil.wrapperOk(appLinkDTO);
            }
        } catch (Exception e) {
            log.error("ErpSsoInfoController query app by name occur error", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.QUERY_OCCUR_ERROR));
        }
    }

    /**
     * 设置归户模式
     *
     * @param ssoBindReq 请求实体
     * @return
     */
    @Operation(summary = "设置归户模式")
    @PostMapping("/updateBindFlag")
    public ResponseEntity<BaseResultDTO<Boolean>> updateBindFlag(@RequestBody SsoBindReq ssoBindReq) throws Exception {
        try {
            if (CollectionUtils.isEmpty(ssoBindReq.getIdList()) || ssoBindReq.getBindFlag() == null || ssoBindReq.getAppType() == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
            }
            if (!Arrays.asList(Constants.BindFlagEnum.AUTO.getFlag(), Constants.BindFlagEnum.HAND.getFlag(), Constants.BindFlagEnum.NO.getFlag()).contains(ssoBindReq.getBindFlag())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
            }
            QuerySsoListReq querySsoListReq = new QuerySsoListReq();
            querySsoListReq.setIdList(ssoBindReq.getIdList());
            // C/S配置
            if (PreinstalledApplicationTypeEnum.CS_APPLICATION.getType().equals(String.valueOf(ssoBindReq.getAppType()))) {
                List<ErpSsoInfo> erpSsoInfoList = erpSsoInfoService.querySsoListByScope(querySsoListReq);
                for (ErpSsoInfo erpSsoInfo : erpSsoInfoList) {
                    if (Objects.equals(erpSsoInfo.getValidStatus(), Constants.VALID_STATUS_ENABLE)) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.BIND_MODE_IN_EFFECT));
                    }
                }
                erpSsoInfoService.updateBindFlag(ssoBindReq.getIdList(), ssoBindReq.getBindFlag());
            } else if (PreinstalledApplicationTypeEnum.BS_APPLICATION.getType().equals(String.valueOf(ssoBindReq.getAppType()))) { // B/S配置
                List<ThirdSsoInfo> thirdSsoInfoList = thirdSsoInfoService.queryThirdListByScope(querySsoListReq);
                for (ThirdSsoInfo thirdSsoInfo : thirdSsoInfoList) {
                    if (Objects.equals(thirdSsoInfo.getValidStatus(), Constants.VALID_STATUS_ENABLE)) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.BIND_MODE_IN_EFFECT));
                    }
                }
                thirdSsoInfoService.updateBindFlag(ssoBindReq.getIdList(), ssoBindReq.getBindFlag());
            } else { // 预设应用
                List<PreinstalledApplication> applicationList = preinstalledApplicationService.queryPreinstalledApplicationList(ssoBindReq.getIdList());
                for (PreinstalledApplication application : applicationList) {
                    if (PreinstalledApplicationTypeEnum.SPECIAL_APPLICATION.getType().equals(application.getApplicationType())) {
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.CONTAIN_SPECIAL_APPLICATION));
                    }
                }
                preinstalledApplicationService.updateAppInfo(ssoBindReq.getIdList(), ssoBindReq.getBindFlag(), null);
            }
            return ResponseEntityWrapperUtil.wrapperOk(Boolean.TRUE);
        } catch (Exception e) {
            log.error("ErpSsoInfoController update bind flag error, ssoBindReq:{}", ssoBindReq, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 查询申请过appToken的应用列表
     *
     * @return
     */
    @GetMapping("/queryAppListExistAppToken")
    public ResponseEntity<BaseResultDTO<List<AppExistVO.AppDTO>>> queryAppListExistAppToken() {
        try {
            List<AppExistVO.AppDTO> list = iamService.queryAppListExistAppToken();
            return ResponseEntityWrapperUtil.wrapperOk(list);
        } catch (Exception e) {
            log.error("query appList exist appToken error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/tenant/erpSsoInfo/queryAppListExistAppToken, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 根据sid查询appToken
     *
     * @return
     */
    @GetMapping("/queryAppTokenBySid")
    public ResponseEntity<BaseResultDTO<Map<String, Object>>> queryAppTokenBySid(@RequestParam Long sid) {
        try {
            Map<String, Object> map = iamService.queryAppTokenBySid(sid);
            return ResponseEntityWrapperUtil.wrapperOk(map);
        } catch (Exception e) {
            log.error("query appToken by sid error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/tenant/erpSsoInfo/queryAppTokenBySid, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}
