package com.digiwin.athena.semc.controller.sso;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.erpsso.RegisterDTO;
import com.digiwin.athena.semc.dto.mobile.CustomApplicationsListPageReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.mobile.MobileDatasourceInfo;
import com.digiwin.athena.semc.entity.sso.MobileSsoInfo;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.mobile.MobileDatasourceInfoService;
import com.digiwin.athena.semc.service.sso.IMobileSsoInfoService;
import com.digiwin.athena.semc.service.sso.IThirdSsoInfoService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.util.ObjectUtils;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 移动端SSO应用配置
 *
 * @author sungq
 * @since 2024-10-24
 */
@Slf4j
@RestController
@RequestMapping("/semc/mobile/sso")
public class MobileSsoInfoController {

    @Autowired
    private IMobileSsoInfoService mobileSsoInfoService;

    @Autowired
    private IThirdSsoInfoService thirdSsoInfoService;

    @Autowired
    private AppLinkService appLinkService;

    @Autowired
    private MobileDatasourceInfoService mobileDatasourceInfoService;

    @Resource
    private MessageUtils messageUtils;


    /**
     * 查询移动端sso应用配置列表
     *
     * @return 返回
     */
    @GetMapping("/queryMobileSsoList")
    public ResponseEntity<BaseResultDTO<List<MobileSsoInfo>>> queryMobileSsoList() {
        try {
            List<MobileSsoInfo> mobileSsoInfoList = mobileSsoInfoService.list();
            return ResponseEntityWrapperUtil.wrapperOk(mobileSsoInfoList);
        } catch (Exception e) {
            log.error("MobileSsoInfoController query mobile sso list error.", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/sso/queryMobileSsoList, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存移动端应用配置
     *
     * @param mobileSsoInfoReq 请求实体类
     * @return 返回
     */
    @PostMapping("/saveMobileInfo")
    public ResponseEntity<BaseResultDTO<MobileSsoInfo>> saveMobileInfo(@RequestBody @Valid MobileSsoInfo mobileSsoInfoReq) {
        // 校验应用是否重复
        List<MobileSsoInfo> mobileSsoInfoList = mobileSsoInfoService.existMobileApp(mobileSsoInfoReq);
        if (CollectionUtils.isNotEmpty(mobileSsoInfoList)) {
            if (mobileSsoInfoReq.getAppCode().equals(mobileSsoInfoList.get(0).getAppCode())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_CODE_REPEAT));
            }
            if (mobileSsoInfoReq.getAppName().equals(mobileSsoInfoList.get(0).getAppName())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_NAME_REPEAT));
            }
            if (StringUtils.isNotBlank(mobileSsoInfoReq.getAppToken()) && mobileSsoInfoReq.getAppToken().equals(mobileSsoInfoList.get(0).getAppToken())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_TOKEN_REPEAT));
            }
        }

        boolean updateFlag = Boolean.FALSE;
        // 修改应用
        if (!ObjectUtils.isEmpty(mobileSsoInfoReq.getId())) {
            MobileSsoInfo mobileSsoInfo = mobileSsoInfoService.getBaseMapper().selectById(mobileSsoInfoReq.getId());
            if (ObjectUtils.isEmpty(mobileSsoInfo)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_QUERY_NOT_FOUND));
            }
            // 判断应用是否注册过
            if (ObjectUtils.isEmpty(mobileSsoInfo.getAppSid())) {
                updateFlag = Boolean.TRUE;
            }
        }

        // 如果应用没有注册，则需要注册应用获取appId和appSecret，用于三方应用sso登录
        if (updateFlag || ObjectUtils.isEmpty(mobileSsoInfoReq.getId())) {
            // 自动归户或手动归户需要
            if (Arrays.asList(Constants.BindFlagEnum.AUTO.getFlag(), Constants.BindFlagEnum.HAND.getFlag()).contains(mobileSsoInfoReq.getUserBindFlag())) {
                try {
                    registerMobileApp(mobileSsoInfoReq);
                } catch (Exception e) {
                    log.error("MobileSsoInfoController.saveMobileInfo invoke iam error error. param:{}", mobileSsoInfoReq, e);
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.ADAPT_SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR) + ":" + e.getMessage());
                }
            }
        }

        // 保存应用
        try {
            Long id = mobileSsoInfoService.saveMobileSso(mobileSsoInfoReq);
            MobileSsoInfo mobileSsoInfo = mobileSsoInfoService.queryMobileSsoInfo(id);
            if (Arrays.asList(Constants.BindFlagEnum.NO.getFlag(), Constants.BindFlagEnum.OUTSIDE.getFlag()).contains(mobileSsoInfoReq.getUserBindFlag())) {
                mobileSsoInfo.setAppId("");
                mobileSsoInfo.setAppSecret("");
            }
            return ResponseEntityWrapperUtil.wrapperOk(mobileSsoInfo);
        } catch (Exception e) {
            log.error("MobileSsoInfoController save mobile info error. param:{}", mobileSsoInfoReq, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/sso/saveMobileInfo, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 注册应用获取appId和appSecret，用于三方应用sso登录
     *
     * @param mobileSsoInfoReq 请求入参
     * @throws Exception 异常
     */
    private void registerMobileApp(MobileSsoInfo mobileSsoInfoReq) throws Exception {
        RegisterDTO registerReq = new RegisterDTO();
        registerReq.setName(mobileSsoInfoReq.getAppCode());
        registerReq.setDescription(mobileSsoInfoReq.getAppDesc());
        registerReq.setCallbackUrl(mobileSsoInfoReq.getCallBackUrl());
        RegisterDTO registerResp = thirdSsoInfoService.registerApp(registerReq);
        mobileSsoInfoReq.setAppSid(registerResp.getSid());
        mobileSsoInfoReq.setAppId(registerResp.getId());
        mobileSsoInfoReq.setAppSecret(registerResp.getSecret());
    }

    /**
     * 删除应用
     *
     * @param id 主键id
     * @return 返回
     */
    @GetMapping("/delMobileInfo")
    public ResponseEntity<BaseResultDTO<Boolean>> deleteInfo(@RequestParam Long id) {
        try {
            if (ObjectUtils.isEmpty(id)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
            }
            MobileSsoInfo mobileSsoInfo = mobileSsoInfoService.queryMobileSsoInfo(id);
            if (ObjectUtils.isEmpty(mobileSsoInfo)) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_QUERY_NOT_FOUND));
            }

            // 校验是否在【移动应用/作业管理列表】中配置
            CustomApplicationsListPageReq customApplicationReq = new CustomApplicationsListPageReq();
            customApplicationReq.setPrimaryId(String.valueOf(id));
            List<MobileDatasourceInfo> mobileDatasourceList = mobileDatasourceInfoService.queryDatasourceInfo(customApplicationReq);
            if (CollectionUtils.isNotEmpty(mobileDatasourceList)) {
                List<AppLinkDTO> mobileAppLinkList = appLinkService.queryMobileManageListSync(AppAuthContextHolder.getContext().getAuthoredUser());
                Map<String, String> appLinkMap = mobileAppLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getPrimaryId, AppLinkDTO::getName, (a, b) -> a));

                List<String> appNameList = Lists.newArrayList();
                mobileDatasourceList.forEach(x -> {
                    if (StringUtils.isNotBlank(appLinkMap.get(x.getApplicationPrimaryId()))) {
                        appNameList.add(appLinkMap.get(x.getApplicationPrimaryId()));
                    }
                });
                String dataNames = String.join("、", appNameList);
                String error = String.format(messageUtils.getMessage("error.message.system.sso.mobile.del"), dataNames);
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            mobileSsoInfoService.removeById(id);
            return ResponseEntityWrapperUtil.wrapperOk(true);
        } catch (Exception e) {
            log.error("MobileSsoInfoController del mobile info error. id:{}", id, e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/mobile/sso/delMobileInfo, message:" + e.getMessage());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}
