package com.digiwin.athena.semc.controller.sso;

import com.google.common.collect.Lists;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.ResultBean;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.PreinstalledApplicationTypeEnum;
import com.digiwin.athena.semc.controller.BasicController;
import com.digiwin.athena.semc.dto.erpsso.RegisterDTO;
import com.digiwin.athena.semc.dto.message.QueryThirdMessageConfigReq;
import com.digiwin.athena.semc.entity.message.ThirdMessageConfig;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.entity.sso.ErpSsoInfo;
import com.digiwin.athena.semc.entity.sso.ThirdSsoInfo;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.message.ThirdSystemMessageService;
import com.digiwin.athena.semc.service.portal.IPreinstalledApplicationService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.service.sso.IErpSsoInfoService;
import com.digiwin.athena.semc.service.sso.IThirdSsoInfoService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.util.ObjectUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.validation.Valid;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;


/**

 *
 * @author sungq
 * @since 2022-11-28
 */

@Tag(name = "第三方web接入sso配置")
@RestController
@RequestMapping("/tenant/thirdSsoInfo")
@Slf4j
public class ThirdSsoInfoController extends BasicController<IThirdSsoInfoService, ThirdSsoInfo> {

    private static final Logger logger = LoggerFactory.getLogger(ThirdSsoInfoController.class);

    /**
     * 配置生效标识
     */
    public static final int SSO_INFO_EFFECTIVE = 1;

    @Autowired
    IThirdSsoInfoService thirdSsoInfoService;

    @Autowired
    IErpSsoInfoService erpSsoInfoService;

    @Resource
    private IamService iamService;

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    LabelSystemDataService labelSystemDataService;

    @Resource
    private ThirdSystemMessageService thirdSystemMessageService;

    @Resource
    private IPreinstalledApplicationService preinstalledApplicationService;

    /**
     * 查询敏态配置列表
     *
     * @return
     */
    @Operation(summary = "根据租户id查询敏态配置列表")
    @GetMapping("/listByTenantId")
    public ResponseEntity<BaseResultDTO<List<ThirdSsoInfo>>> queryThirdSsoList(@RequestParam(required = false, value = "protocolType") Integer protocolType) {
        try {
            List<ThirdSsoInfo> thirdSsoInfoList = Lists.newArrayList();
            QueryWrapper<ThirdSsoInfo> condition = new QueryWrapper<>();
            if ( null !=protocolType) {
                condition.eq("protocol_type", protocolType);
            }
            condition.orderByDesc("modify_time").orderByDesc("id");
            thirdSsoInfoList = thirdSsoInfoService.list(condition);
            return ResponseEntityWrapperUtil.wrapperOk(thirdSsoInfoList);
        } catch (Exception e) {
            log.error("ThirdSsoInfoController query third sso list error", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 保存敏态配置信息
     *
     * @param thirdSsoInfo 请求实体类
     * @return
     * @throws Exception
     */
    @Operation(summary = "保存敏态配置信息")
    @PostMapping("/saveThirdInfo")
    public ResponseEntity<BaseResultDTO<ThirdSsoInfo>> saveThirdInfo(@RequestBody @Valid ThirdSsoInfo thirdSsoInfo) throws Exception {
        logger.info("save or update third ssoInfo | param:" + JSON.toJSONString(thirdSsoInfo) + "");
        // 是否为SAML协议标志位（用于app校验和注册判断）
        boolean isSamlType = thirdSsoInfo.getProtocolType() == 3;
        boolean isCasType = thirdSsoInfo.getProtocolType() == 2;
        // 如果应用是手动归户或者自动归户，则appToken必传
        /*if ((Constants.BindFlagEnum.HAND.getFlag().equals(thirdSsoInfo.getUserBindFlag()) || Constants.BindFlagEnum.AUTO.getFlag().equals(thirdSsoInfo.getUserBindFlag()))
                && StringUtils.isBlank(thirdSsoInfo.getAppToken()) && !isSamlType &&!isCasType) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "appToken cannot be null");
        }*/

        // 校验应用的重复性
        List<ThirdSsoInfo> thirdSsoInfoList = thirdSsoInfoService.existThirdSso(thirdSsoInfo.getId(), thirdSsoInfo.getAppCode(), thirdSsoInfo.getAppName(), thirdSsoInfo.getAppToken());
        if (CollectionUtils.isNotEmpty(thirdSsoInfoList)) {
            if (thirdSsoInfo.getAppCode().equals(thirdSsoInfoList.get(0).getAppCode())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_CODE_REPEAT));
            }
            if (thirdSsoInfo.getAppName().equals(thirdSsoInfoList.get(0).getAppName())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_NAME_REPEAT));
            }
            if (StringUtils.isNotBlank(thirdSsoInfo.getAppToken()) && thirdSsoInfo.getAppToken().equals(thirdSsoInfoList.get(0).getAppToken())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_TOKEN_REPEAT));
            }
        }
        List<ErpSsoInfo> erpSsoInfoList = erpSsoInfoService.existByNameOrAppToken(null, thirdSsoInfo.getAppCode(), thirdSsoInfo.getAppName(), thirdSsoInfo.getAppToken());
        if (CollectionUtils.isNotEmpty(erpSsoInfoList)) {
            if (thirdSsoInfo.getAppCode().equals(erpSsoInfoList.get(0).getCode())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.CS_APP_CODE_REPEAT));
            }
            if (thirdSsoInfo.getAppName().equals(erpSsoInfoList.get(0).getName())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.CS_APP_NAME_REPEAT));
            }
            if (StringUtils.isNotBlank(thirdSsoInfo.getAppToken()) && thirdSsoInfo.getAppToken().equals(erpSsoInfoList.get(0).getAppToken())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.CS_APP_TOKEN_REPEAT));
            }
        }
        List<PreinstalledApplication> applicationList = preinstalledApplicationService.queryPreinstalledAppList(AppAuthContextHolder.getContext().getAuthoredUser());
        for (PreinstalledApplication application : applicationList) {
            if (application.getApplicationName().equals(thirdSsoInfo.getAppName())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.PRE_APP_NAME_REPEAT));
            }
        }

        ThirdSsoInfo ssoInfo = new ThirdSsoInfo();
        boolean updateFlag = Boolean.FALSE;
        // 修改或新增应用，需要先调IAM更新或注册应用
        // 修改应用
        if (thirdSsoInfo.getId() != null) {
            ssoInfo = thirdSsoInfoService.queryThirdSsoInfo(thirdSsoInfo.getId());
            if (ssoInfo == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_QUERY_NOT_FOUND));
            }
            // 应用id不能修改
            if (!ssoInfo.getAppCode().equals(thirdSsoInfo.getAppCode())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_CODE_CHANGE));
            }

            // 如果应用是自动模式或手动模式，则需要注册应用或更新应用(saml无appToken，不需要注册)
            if (Constants.BindFlagEnum.AUTO.getFlag().equals(thirdSsoInfo.getUserBindFlag()) || Constants.BindFlagEnum.HAND.getFlag().equals(thirdSsoInfo.getUserBindFlag())) {
                RegisterDTO registerReq = new RegisterDTO();
                try {
                    if ((isSamlType || isCasType) ? false : true) {
                        // 如果appSid为空，则注册应用
                        if (ssoInfo.getAppSid() == null) {
                            registerBsApp(thirdSsoInfo);
                        } else {  // 如果appSid不为空，则更新应用
                            registerReq = buildRegisterReq(thirdSsoInfo, ssoInfo.getAppSid());
                            iamService.updateAppInfoForIAM(registerReq);
                            updateFlag = Boolean.TRUE;
                        }
                    }
                } catch (Exception e) {
                    logger.error("ThirdSsoInfoController saveThirdInfo invoke iam error error, registerReq:{}, thirdSsoInfo:{}", registerReq, thirdSsoInfo, e);
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.ADAPT_SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
                }
            }
        } else {    // 新增应用
            // 如果是应用是自动模式或手动模式，则需要注册应用
            if (Constants.BindFlagEnum.AUTO.getFlag().equals(thirdSsoInfo.getUserBindFlag()) || Constants.BindFlagEnum.HAND.getFlag().equals(thirdSsoInfo.getUserBindFlag())) {
                if ((isSamlType || isCasType) ? false : true) {
                    try {
                        registerBsApp(thirdSsoInfo);
                    } catch (Exception e) {
                        logger.error("ThirdSsoInfoController saveErpInfo invoke iam error error, addSsoInfoReq:{}", thirdSsoInfo, e);
                        return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.ADAPT_SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
                    }
                }
            }
        }

        // 保存应用
        try {
            if (StringUtils.isBlank(thirdSsoInfo.getAppId()) || ObjectUtils.isEmpty(thirdSsoInfo.getAppId())) {
                String uuid = UUID.randomUUID().toString();
                uuid = uuid.replace("-", "");
                thirdSsoInfo.setAppId(uuid);
            }
            Long ssoInfoId = thirdSsoInfoService.saveOrUpdateThirdSso(thirdSsoInfo);
            ThirdSsoInfo thirdSsoInfoResp = thirdSsoInfoService.queryThirdSsoInfo(ssoInfoId);
            if (Constants.BindFlagEnum.NO.getFlag().equals(thirdSsoInfo.getUserBindFlag()) || Constants.BindFlagEnum.OUTSIDE.getFlag().equals(thirdSsoInfo.getUserBindFlag())) {
                thirdSsoInfoResp.setAppId("");
                thirdSsoInfoResp.setAppSecret("");
            }
            return ResponseEntityWrapperUtil.wrapperOk(thirdSsoInfoResp);
        } catch (Exception e) {
            // 保存应用失败，则把应用信息还原
            if (updateFlag) {
                RegisterDTO registerReq = buildRegisterReq(ssoInfo, ssoInfo.getAppSid());
                iamService.updateAppInfoForIAM(registerReq);
            }
            logger.info("ThirdSsoInfoController save third app error, thirdSsoInfo:{}", thirdSsoInfo, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.ADAPT_SYSTEM_ERROR, messageUtils.getMessage(I18NKey.SAVE_APP_ERROR));
        }
    }


    /**
     * 应用注册
     *
     * @param thirdSsoInfo 请求入参
     * @throws Exception
     */
    private void registerBsApp(ThirdSsoInfo thirdSsoInfo) throws Exception {
        RegisterDTO registerReq = buildRegisterReq(thirdSsoInfo, null);
        RegisterDTO registerResp = thirdSsoInfoService.registerApp(registerReq);
        thirdSsoInfo.setAppSid(registerResp.getSid());
        thirdSsoInfo.setAppId(registerResp.getId());
        thirdSsoInfo.setAppSecret(registerResp.getSecret());
    }

    /**
     * 构建调用IAM接口请求对象
     *
     * @param thirdSsoInfo 请求对象
     * @return
     */

    private RegisterDTO buildRegisterReq(ThirdSsoInfo thirdSsoInfo, Long appSid) {
        RegisterDTO registerReq = new RegisterDTO();
        registerReq.setSid(appSid);
        // registerReq.setName(thirdSsoInfo.getAppName());
        registerReq.setName(thirdSsoInfo.getAppCode());
        registerReq.setDescription(thirdSsoInfo.getAppDesc());
        registerReq.setCallbackUrl(thirdSsoInfo.getCallBackUrl());
        registerReq.setAppToken(thirdSsoInfo.getAppToken());
        return registerReq;
    }


    /**
     * 修改生效状态
     *
     * @param ssoInfoId 配置id
     * @param status    状态
     * @return
     */
    @Operation(summary = "修改生效状态")
    @GetMapping("/updateStatus")
    public ResponseEntity<BaseResultDTO<Boolean>> editThirdSsoStatus(@RequestParam("ssoInfoId") Long ssoInfoId,
                                                                     @RequestParam("status") Integer status) throws Exception {
        try {
            if (ssoInfoId == null || status == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
            }
            ThirdSsoInfo ssoInfo = thirdSsoInfoService.queryThirdSsoInfo(ssoInfoId);
            if (ssoInfo == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_QUERY_NOT_FOUND));
            }

            //如果是停用，校验是否有数据源引用
            if (status == 0) {
                List<String> systemDataList = labelSystemDataService.getDataNameList(Integer.parseInt(PreinstalledApplicationTypeEnum.BS_APPLICATION.getValue()), ssoInfoId + "");
                if (CollectionUtils.isNotEmpty(systemDataList)) {
                    String dataNames = systemDataList.stream().collect(Collectors.joining("、"));
                    String error = String.format(messageUtils.getMessage("error.message.system.sso.erp.fail"), dataNames);
                    return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
                }
            }
            int ssoId = thirdSsoInfoService.updateThirdSsoStatus(ssoInfoId, status);
            return ResponseEntityWrapperUtil.wrapperOk(ssoId > 0 ? Boolean.TRUE : Boolean.FALSE);
        } catch (Exception e) {
            logger.info("ThirdSsoInfoController update b/s app status error. ssoInfoId:{}, status:{}", ssoInfoId, status, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 删除配置
     *
     * @param ssoInfoId 配置id
     * @return
     */
    @Operation(summary = "删除配置")
    @GetMapping("/deleteById")
    public ResponseEntity<BaseResultDTO<Boolean>> deleteInfo(@RequestParam("ssoInfoId") Long ssoInfoId) throws Exception {
        try {
            if (ssoInfoId == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
            }
            ThirdSsoInfo ssoInfo = thirdSsoInfoService.queryThirdSsoInfo(ssoInfoId);
            if (ssoInfo == null) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.APP_QUERY_NOT_FOUND));
            }
            if (SSO_INFO_EFFECTIVE == ssoInfo.getValidStatus()) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.CONFIG_IN_EFFECT));
            }
            //校验是否有数据源引用
            List<String> systemDataList = labelSystemDataService.getDataNameList(Integer.parseInt(PreinstalledApplicationTypeEnum.BS_APPLICATION.getValue()), ssoInfoId + "");
            if (CollectionUtils.isNotEmpty(systemDataList)) {
                String dataNames = systemDataList.stream().collect(Collectors.joining("、"));
                String error = String.format(messageUtils.getMessage("error.message.system.sso.erp.delfail"), dataNames);
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            // 校验是否在异构消息中配置
            QueryThirdMessageConfigReq queryImportRecordReq = new QueryThirdMessageConfigReq();
            queryImportRecordReq.setAppPrimaryId(String.valueOf(ssoInfoId));
            queryImportRecordReq.setAppSource(ApplicationTypeEnum.BS_APPLICATION.getType());
            PageInfoResp<ThirdMessageConfig> thirdMessageConfigPageInfoResp = thirdSystemMessageService.queryMessageConfigPage(queryImportRecordReq);
            if (CollectionUtils.isNotEmpty(thirdMessageConfigPageInfoResp.getList())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage("error.message.system.sso.del.fail"));
            }
            Boolean flag = thirdSsoInfoService.removeById(ssoInfoId);
            return ResponseEntityWrapperUtil.wrapperOk(flag);
        } catch (Exception e) {
            logger.info("ThirdSsoInfoController delete b/s app error. ssoInfoId:{}", ssoInfoId, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 注册应用（目前不用了）
     *
     * @param registerDTO 请求参数
     * @return
     */
    @Operation(summary = "注册应用")
    @PostMapping("/register")
    public ResponseEntity<BaseResultDTO<RegisterDTO>> registerApp(@RequestBody RegisterDTO registerDTO) {
        try {
            if (StringUtils.isBlank(registerDTO.getName()) || StringUtils.isBlank(registerDTO.getDescription())) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING));
            }
            RegisterDTO registerDTOResp = thirdSsoInfoService.registerApp(registerDTO);
            ResultBean resultBean = new ResultBean();
            resultBean.setResponse(registerDTOResp);
            return ResponseEntityWrapperUtil.wrapperOk(registerDTOResp);
        } catch (Exception e) {
            logger.info("ThirdSsoInfoController register app error. registerDTO:{}", registerDTO, e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }
}
