package com.digiwin.athena.semc.controller.temp;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.util.StringUtils;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.dto.portal.QueryImportRecordReq;
import com.digiwin.athena.semc.entity.portal.LabelSystemDataImport;
import com.digiwin.athena.semc.entity.temp.TemplateBench;
import com.digiwin.athena.semc.entity.temp.TemplateBenchImport;
import com.digiwin.athena.semc.entity.temp.TemplateInfo;
import com.digiwin.athena.semc.service.temp.TemplateBenchService;
import com.digiwin.athena.semc.service.temp.TemplateInfoService;
import com.digiwin.athena.semc.service.temp.impl.TemplateBenchWriteHandler;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.bench.JobBenchExportVo;
import com.digiwin.athena.semc.vo.portal.ImportExcelResp;
import com.digiwin.athena.semc.vo.temp.TemplateBenchExportVo;
import com.digiwin.athena.semc.vo.temp.TemplateBenchImportVo;
import com.digiwin.athena.semc.vo.temp.TemplateBenchVO;
import com.digiwin.athena.semc.vo.temp.TemplateSaveVO;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;

import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;

/**
 * @description ISV工作台模版
 * @author: dengkaiwen
 */
@RestController
@Slf4j
@RequestMapping("/semc/job/bench/temp/")
public class TemplateBenchController {

    @Autowired
    TemplateInfoService templateInfoService;

    @Autowired
    TemplateBenchService templateBenchService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 作业工作台模版保存接口
     *
     * @param req
     * @return
     */
    @Operation(method = "saveBenchTemp", description = "作业工作台模版保存接口")
    @PostMapping("saveBenchTemp")
    public ResponseEntity<?> saveBenchTemp(@RequestBody  TemplateBenchVO req) {
        TemplateSaveVO respVo=new TemplateSaveVO();
        //校验名称是否必填
        if (StringUtils.isEmpty(req.getTemplateName())) {
            String error = messageUtils.getMessage("error.message.job.bench.name.empty");
            respVo.setMsg(error);
            return ResponseEntityWrapper.wrapperOk(respVo);
        }
        if (req.getTemplateName().trim().length()>12) {
            String error = String.format(messageUtils.getMessage("error.message.job.bench.name.empty.length"), 12);
            respVo.setMsg(error);
            return ResponseEntityWrapper.wrapperOk(respVo);
        }
        //描述超过100字符
        if (!StringUtils.isEmpty(req.getTemplateDesc()) && req.getTemplateDesc().trim().length() >100) {
            String error = String.format(messageUtils.getMessage("error.message.template.desc.length.error"), 100);
            respVo.setMsg(error);
            return ResponseEntityWrapper.wrapperOk(respVo);
        }
        //校验模版名称是否唯一
        TemplateInfo nameCheckReq = new TemplateInfo();
        nameCheckReq.setId(req.getTemplateId());
        nameCheckReq.setTemplateName(req.getTemplateName());
        nameCheckReq.setTemplateType(Constants.TemplateTypeEnum.JOB_BENCH.getVal());
        List<TemplateInfo> nameInfoList = templateInfoService.queryByParam(nameCheckReq);
        if (CollectionUtils.isNotEmpty(nameInfoList)) {
            String error = messageUtils.getMessage("error.message.template.name.exist.error");
            respVo.setMsg(error);
            return ResponseEntityWrapper.wrapperOk(respVo);
        }
        try {
            return ResponseEntityWrapper.wrapperOk(templateBenchService.saveBenchTemp(req));
        } catch (Exception e) {
            log.error("saveBenchTemp exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/saveTemplate");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 查询模版下作业工作台接口
     *
     * @param req
     * @return
     */
    @Operation(method = "queryBenchByTemp", description = "查询模版下作业工作台接口")
    @PostMapping("queryBenchByTemp")
    public ResponseEntity<BaseResultDTO<TemplateBenchVO>> queryBenchByTemp(@RequestBody TemplateBenchVO req) {
        //校验是否必填
        if (null == req.getTemplateId()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "templateId is null");
        }
        try {
            return ResponseEntityWrapperUtil.wrapperOk(templateBenchService.queryBenchByTemp(req));
        } catch (Exception e) {
            log.error("saveBenchTemp exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/saveTemplate");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 查询租户可见的作业工作台模版
     * @return
     */
    @PostMapping("/queryTempBenchList")
    public ResponseEntity<BaseResultDTO<List<TemplateBench>>> queryTempBenchList() {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(templateBenchService.queryTempBenchList());
        } catch (Exception e) {
            log.error("queryTempBenchList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/job/bench/queryTempBenchList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 导入作业工作台
     *
     * @return
     */
    @PostMapping("/importReport")
    public ResponseEntity<BaseResultDTO<ImportExcelResp>> importReport(@RequestBody LabelSystemDataImport req) {
        //校验文件id
        if (org.apache.commons.lang.StringUtils.isEmpty(req.getFileId())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "file code cannot be empty");
        }
        ImportExcelResp excelResp = new ImportExcelResp();
        try {
            // 读取报表内容
            List<TemplateBenchImportVo> excelVoList = templateBenchService.readExcel(req.getFileId());
            if (CollectionUtils.isEmpty(excelVoList)) {
                String error = messageUtils.getMessage("error.message.import.cannot.be.empty");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            //去掉前后空格
            excelVoList=trimDataList(excelVoList);
            excelResp = templateBenchService.importReport(excelVoList);
            return ResponseEntityWrapperUtil.wrapperOk(excelResp);
        } catch (Exception e) {
            log.error("TemplateBenchController import excel exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 去掉空格
     * @param excelVoList
     * @return
     */
    public List<TemplateBenchImportVo> trimDataList(List<TemplateBenchImportVo> excelVoList){
        for(TemplateBenchImportVo excelVo:excelVoList){
            if(org.apache.commons.lang.StringUtils.isNotEmpty(excelVo.getBenchName())){
                excelVo.setBenchName(excelVo.getBenchName().trim());
            }
            if(org.apache.commons.lang.StringUtils.isNotEmpty(excelVo.getJobName())){
                excelVo.setJobName(excelVo.getJobName().trim());
            }
            if(org.apache.commons.lang.StringUtils.isNotEmpty(excelVo.getLaneName())){
                excelVo.setLaneName(excelVo.getLaneName().trim());
            }
        }
        return excelVoList;
    }
    /**
     * 查询导入记录
     *
     * @param queryImportRecordReq 入参
     * @return
     */
    @PostMapping("/queryImportRecord")
    public ResponseEntity<BaseResultDTO<PageInfoResp<TemplateBenchImport>>> queryImportRecord(@RequestBody @Valid QueryImportRecordReq queryImportRecordReq) {
        try {
            // 校验时间格式
            if ((org.apache.commons.lang.StringUtils.isNotBlank(queryImportRecordReq.getStartTime()) && !DateUtils.validateDate(queryImportRecordReq.getStartTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) || (org.apache.commons.lang.StringUtils.isNotBlank(queryImportRecordReq.getEndTime()) && !DateUtils.validateDate(queryImportRecordReq.getEndTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER))) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
            }
            PageInfoResp<TemplateBenchImport> pageInfoResp = templateBenchService.queryImportRecord(queryImportRecordReq);
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("TemplateBenchController queryImportRecord exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 导出作业工作台
     *
     * @return
     */
    @GetMapping("/exportReport")
    public void exportReport(HttpServletResponse response) {
        try {
            List<TemplateBenchExportVo> excelVoList = templateBenchService.exportReport();
            //设置文件名
            String filename = "作业工作台_" + LocalDate.now().toString().replace("-", "") + ".xlsx";
            //设置下载信息
            response.setContentType("application/vnd.ms-excel");
            response.setCharacterEncoding("utf-8");
            response.setHeader("Content-disposition", "attachment;filename=" + new String(filename.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1));
            EasyExcel.write(response.getOutputStream(), JobBenchExportVo.class)
                    .registerWriteHandler(Utils.initHorizontalCell())
                    .sheet("作业工作台").relativeHeadRowIndex(1).registerWriteHandler(new TemplateBenchWriteHandler()).doWrite(excelVoList);
        } catch (Exception e) {
            log.error("TemplateBenchController exportReport exception", e);
            throw new RuntimeException(e);
        }
    }
}
