package com.digiwin.athena.semc.controller.temp;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.dto.portal.PortalInfoQueryReq;
import com.digiwin.athena.semc.dto.portal.TemplateQueryReq;
import com.digiwin.athena.semc.dto.portal.UsePortalInfoDto;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.portal.PortalInfo;
import com.digiwin.athena.semc.entity.temp.TemplateInfo;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.service.temp.TemplateInfoService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.portal.QueryTempAppListVo;
import com.digiwin.athena.semc.vo.temp.TemplateCheckResp;
import com.digiwin.athena.semc.vo.temp.TemplateSaveVO;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @description 模版配置管理
 * @author: dengkaiwen
 */
@RestController
@Slf4j
@RequestMapping("/semc/template/info/")
public class TemplateInfoController {

    @Autowired
    TemplateInfoService templateInfoService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private EnvProperties envProperties;

    /**
     * 查询模版配置列表
     *
     * @param
     * @return
     */
    @Operation(method = "/pageQuery", description = "查询模版配置列表")
    @PostMapping("/pageQuery")
    public ResultPageBean pageQuery(@RequestBody @Valid TemplateQueryReq req) {
        return templateInfoService.pageQuery(req);
    }



    /**
     * 批量删除
     *
     * @param req
     * @return
     */
    @Operation(method = "batDel", description = "批量删除")
    @PostMapping("batDel")
    public ResponseEntity<?> batDel(@RequestBody PortalInfoQueryReq req) {
        if (CollectionUtils.isEmpty(req.getIdList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "idList is null");
        }
        try {
            return templateInfoService.batDel(req.getIdList());
        } catch (Exception e) {
            log.error("batDel exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/batDel");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 发布和停用
     *
     * @param req
     * @return
     */
    @Operation(method = "updateStatus", description = "发布和停用")
    @PostMapping("updateStatus")
    public ResponseEntity<?> updateStatus(@RequestBody PortalInfoQueryReq req) {
        if (CollectionUtils.isEmpty(req.getIdList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "idList is null");
        }
        if (null == req.getStatus()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "status is null");
        }
        try {
          return   templateInfoService.updateStatus(req.getIdList(), req.getStatus());
        } catch (Exception e) {
            log.error("updateStatus exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/updateStatus");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 批量发布模版到租户接口
     *
     * @param req
     * @return
     */
    @Operation(method = "batTempToTenant", description = "批量发布模版到租户接口")
    @PostMapping("batTempToTenant")
    public ResponseEntity<BaseResultDTO<TemplateSaveVO>> batTempToTenant(@RequestBody TemplateInfo req) {
        TemplateSaveVO respVo=new TemplateSaveVO();
        //校验必填参数模版ID集合
        if (CollectionUtils.isEmpty(req.getIdList())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "idList is null");
        }
        //校验必填参数订阅的类型
        if (null == req.getSubType()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "subType is null");
        }
        //校验入参中是否存在非登录用户的模版
        List<Long> loginUserIdList =checkToTenant(req.getIdList());
        respVo.setIsExistOther(false);
        if(loginUserIdList.size() != req.getIdList().size()){
            respVo.setIsExistOther(true);
        }
        req.setIdList(loginUserIdList);
        //校验过滤掉非本人创建的模版后，剩下的模版是否为空
        if (CollectionUtils.isEmpty(req.getIdList())) {
            return ResponseEntityWrapperUtil.wrapperOk(respVo);
        }
        try {
            templateInfoService.batTempToTenant(req);
        } catch (Exception e) {
            log.error("batTempToTenant exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/batTempToTenant");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        return ResponseEntityWrapperUtil.wrapperOk(respVo);
    }

    /**
     *
     * @param idList 登录用户本人创建的模版ID
     * @return
     */
    public List<Long> checkToTenant(List<Long> idList) {
        List<TemplateInfo> templateInfoList = templateInfoService.listByIds(idList);
        List<TemplateInfo> loginUserList = templateInfoList.stream().filter(item -> Utils.getUserId().equals(item.getCreateUserId())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(loginUserList)) {
            return new ArrayList<>();
        }
        List<Long> loginUserIdList = loginUserList.stream().map(TemplateInfo::getId).collect(Collectors.toList());
        return loginUserIdList;
    }

    /**
     * 获取用户是否有ISV权限
     *
     * @return
     */
    @Operation(method = "getUserTemplate", description = "获取用户是否有ISV权限")
    @PostMapping("getUserTemplate")
    public ResponseEntity<BaseResultDTO<TemplateCheckResp>> getUserTemplate() {
        TemplateCheckResp templateCheckResp = new TemplateCheckResp();
        templateCheckResp.setIsHaveTemp(false);
        templateCheckResp.setIsHaveTenant(false);
        try {
            //获取ISV账号信息
            List<String> userIdList = Utils.strToList(envProperties.getTemplateUserId());
            //是否有88888888租户权限
            if (Utils.getTenantId().equals(envProperties.getTemplateTenantId())) {
                templateCheckResp.setIsHaveTenant(true);
            }
            //是否有88888888租户权限且是泽成账号
            if (CollectionUtils.isNotEmpty(userIdList) && userIdList.contains(Utils.getUserId())
                    && templateCheckResp.getIsHaveTenant()) {
                templateCheckResp.setIsHaveTemp(true);
            }
        } catch (Exception e) {
            log.error("getUserTemplate exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/getUserTemplate");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        return ResponseEntityWrapperUtil.wrapperOk(templateCheckResp);
    }


    /**
     * 查询模版下SSO应用
     *
     * @return
     */
    @PostMapping("/queryTempAppList")
    public ResponseEntity<BaseResultDTO<List<QueryTempAppListVo>>> queryTempAppList(@RequestBody TemplateInfo req) {
        //经典门户中，门户ID必填
        if (req.getServiceSource() == 0 && null == req.getId()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id is not empty");
        }
        //作业工作台，用户选择的工作台模版ID必填
        if (req.getServiceSource() == 1 && CollectionUtils.isEmpty(req.getBenchIdList())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "BenchIdList is not empty");
        }
        try {
            List<QueryTempAppListVo> appLinkList = templateInfoService.queryTempAppList(req);
            return ResponseEntityWrapperUtil.wrapperOk(appLinkList);
        } catch (Exception e) {
            log.error("queryTempAppList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/queryTempAppList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 校验租户该模版中源应用涉及的雅典娜应用，是否具备雅典娜应用应用权限
     * 点击【立即使用】请求
     *
     * @return
     */
    @Operation(method = "checkUserTemplate", description = "校验租户该模版中源应用权限")
    @PostMapping("checkUserTemplate")
    public ResponseEntity<?> checkUserTemplate(@RequestBody TemplateInfo req) {
        if (null == req.getId()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id is not empty");
        }
        try {
            return templateInfoService.checkUserTemplate(req.getId());
        } catch (Exception e) {
            log.error("checkUserTemplate exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/checkUserTemplate");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 租户雅典娜涉及组件权限同步
     * 点击租户端无权限弹框中【确定】请求该接口
     *
     * @return
     */
    @Operation(method = "confimTenTemplate", description = "租户雅典娜涉及组件权限同步")
    @PostMapping("confimTenTemplate")
    public ResponseEntity<?> confimTenTemplate(@RequestBody TemplateInfo req) {
        if (null == req.getId()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id is not empty");
        }
        try {
            return templateInfoService.confimTenTemplate(req.getId());
        } catch (Exception e) {
            log.error("confimTenTemplate exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/confimTenTemplate");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 立即使用ISV门户模版
     * 点击关联源应用【确定】请求
     *
     * @param req
     * @return
     */
    @Operation(method = "usePortalInfo", description = "立即使用ISV门户模版")
    @PostMapping("/usePortalInfo")
    public ResponseEntity<BaseResultDTO<PortalInfo>> usePortalInfo(@RequestBody @Valid UsePortalInfoDto req) {
        try {
            return ResponseEntityWrapperUtil.wrapperOk(templateInfoService.usePortalInfo(req));
        } catch (Exception e) {
            log.error("usePortalInfo exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/usePortalInfo");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 查询作业工作台中源应用权限
     * 校验租户该模版中源应用涉及的雅典娜应用，是否具备雅典娜应用应用权限
     * 作业工作台 点击【立即使用】请求
     *
     * @return
     */
    @PostMapping("checkBenchTemplate")
    public ResponseEntity<?> checkBenchTemplate(@RequestBody UsePortalInfoDto req) {
        if(CollectionUtils.isEmpty(req.getBenchIdList()) ){
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "BenchIdList is not empty");
        }
        try {
            return templateInfoService.checkBenchTemplate(req.getBenchIdList());
        } catch (Exception e) {
            log.error("checkBenchTemplate exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/checkBenchTemplate");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 雅典娜涉及组件权限同步【作业工作台】
     * 点击租户端无权限弹框中【确定】请求该接口
     *
     * @return
     */
    @PostMapping("confimBenchTemplate")
    public ResponseEntity<?> confimBenchTemplate(@RequestBody UsePortalInfoDto req) {
        if(CollectionUtils.isEmpty(req.getBenchIdList()) ){
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "BenchIdList is not empty");
        }
        try {
            return templateInfoService.confimBenchTemplate(req.getBenchIdList());
        } catch (Exception e) {
            log.error("confimBenchTemplate exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/confimBenchTemplate");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 立即使用ISV门户模版【作业工作台】
     * 点击关联源应用【确定】请求
     *
     * @param req
     * @return
     */
    @PostMapping("/useBenchTemp")
    public ResponseEntity<?> useBenchTemp(@RequestBody @Valid UsePortalInfoDto req) {
        if(CollectionUtils.isEmpty(req.getBenchIdList()) ){
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "BenchIdList is not empty");
        }
        try {
            return templateInfoService.useBenchTemp(req);
        } catch (Exception e) {
            log.error("useBenchTemp exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/template/info/useBenchTemp");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

}
