package com.digiwin.athena.semc.controller.temp;

import com.alibaba.excel.util.StringUtils;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.*;
import com.digiwin.athena.semc.dto.portal.PortalInfoQueryReq;
import com.digiwin.athena.semc.entity.temp.TemplateInfo;
import com.digiwin.athena.semc.service.temp.TemplateBenchService;
import com.digiwin.athena.semc.service.temp.TemplateInfoService;
import com.digiwin.athena.semc.service.temp.TemplatePortalInfoService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.temp.TemplateBenchVO;
import com.digiwin.athena.semc.vo.temp.TemplatePortalVO;
import com.digiwin.athena.semc.vo.temp.TemplateSaveVO;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.List;

/**
 * @description ISV门户模版
 * @author: dengkaiwen
 */
@RestController
@Slf4j
@RequestMapping("/semc/portal/temp/")
public class TemplatePortalController {

    @Autowired
    TemplateInfoService templateInfoService;

    @Autowired
    TemplatePortalInfoService templatePortalInfoService;

    @Resource
    private MessageUtils messageUtils;



    /**
     * 保存ISV门户模版布局信息
     *
     * @param req
     * @return
     */
    @Operation(method = "savePortalTemp", description = "保存ISV门户模版布局信息")
    @PostMapping("savePortalTemp")
    public ResponseEntity<?> savePortalTemp(@RequestBody  @Valid TemplatePortalVO req) {
        TemplateSaveVO respVo=new TemplateSaveVO();
        //校验名称是否必填
        if (StringUtils.isEmpty(req.getName())) {
            String error = messageUtils.getMessage("error.message.job.bench.name.empty");
            respVo.setMsg(error);
            return ResponseEntityWrapper.wrapperOk(respVo);
        }
        if (req.getName().trim().length()>30) {
            String error = String.format(messageUtils.getMessage("error.message.job.bench.name.empty.length"), 30);
            respVo.setMsg(error);
            return ResponseEntityWrapper.wrapperOk(respVo);
        }
        //描述超过100字符
        if (!StringUtils.isEmpty(req.getPortalDesc()) && req.getPortalDesc().trim().length() >100) {
            String error = String.format(messageUtils.getMessage("error.message.template.desc.length.error"), 100);
            respVo.setMsg(error);
            return ResponseEntityWrapper.wrapperOk(respVo);
        }
        //校验模版名称是否唯一
        TemplateInfo nameCheckReq = new TemplateInfo();
        nameCheckReq.setId(req.getTemplateId());
        nameCheckReq.setTemplateName(req.getName());
        nameCheckReq.setTemplateType(Constants.TemplateTypeEnum.CLASSIC_PORTAL.getVal());
        List<TemplateInfo> nameInfoList = templateInfoService.queryByParam(nameCheckReq);
        if (CollectionUtils.isNotEmpty(nameInfoList)) {
            String error = messageUtils.getMessage("error.message.template.name.exist.error");
            respVo.setMsg(error);
            return ResponseEntityWrapper.wrapperOk(respVo);
        }
        try {
            return ResponseEntityWrapper.wrapperOk(templatePortalInfoService.savePortalTemp(req));
        } catch (Exception e) {
            log.error("savePortalTemp exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/temp/savePortalTemp");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 获取ISV门户模版详情数据
     *
     * @param req
     * @return
     */
    @Operation(method = "getPortalTemp", description = "获取ISV门户模版详情数据")
    @PostMapping("getPortalTemp")
    public ResponseEntity<BaseResultDTO<TemplatePortalVO>> getPortalTemp(@RequestBody TemplatePortalVO req) {
        //校验是否必填
        if (null == req.getTemplateId()) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "templateId is null");
        }
        try {
            return ResponseEntityWrapperUtil.wrapperOk(templatePortalInfoService.getPortalTemp(req));
        } catch (Exception e) {
            log.error("getPortalTemp exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/temp/getPortalTemp");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 查询用户可见门户模版列表
     * @return
     */
    @PostMapping("/pageTemplateQuery")
    public ResultPageBean pageTemplateQuery(@RequestBody @Valid PortalInfoQueryReq req) {
        try {
            return templatePortalInfoService.pageQueryTemp(req);
        } catch (Exception e) {
            log.error("pageTemplateQuery exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/portal/temp/pageTemplateQuery");
            return  ResultPageBean.sysException(new ServiceException(ErrorCodeConstant.LABEL_SYSTEM_CUSTOM,
                    messageUtils.getMessage(I18NKey.SYSTEM_CUSTOM)));
        }
    }

}
