package com.digiwin.athena.semc.controller.temp;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.dto.portal.QueryLabelSystemReq;
import com.digiwin.athena.semc.entity.temp.TemplateSystemData;
import com.digiwin.athena.semc.service.temp.TemplateSystemDataService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;

/**
 * @description ISV模版数据源
 * @author: dengkaiwen
 */
@RestController
@Slf4j
@RequestMapping("/semc/system/data/temp")
public class TemplateSystemDataController {


    @Autowired
    TemplateSystemDataService templateSystemDataService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 查询数据源
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "查询数据源")
    @PostMapping("/queryDataList")
    public ResponseEntity<BaseResultDTO<PageInfoResp<TemplateSystemData>>> queryDataList(@RequestBody QueryLabelSystemReq req) {
        try {
            if (null == req.getPageNum() || null == req.getPageSize()) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "missing paging parameter");
            }
            return ResponseEntityWrapperUtil.wrapperOk(templateSystemDataService.queryDataList(req));
        } catch (Exception e) {
            log.error("queryDataList exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/system/data/temp/queryDataList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 启用停用数据源
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "启用停用")
    @PostMapping("/updateValidStatus")
    public ResponseEntity<?> updateValidStatus(@RequestBody QueryLabelSystemReq req) {
        try {
            if (null == req.getId()) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "id cannot be empty");
            }
            if (null == req.getValidStatus()) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "the status cannot be empty");
            }
            return templateSystemDataService.updateValidStatus(req);
        } catch (Exception e) {
            log.error("updateValidStatus exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/system/data/temp/updateValidStatus");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 批量删除数据源
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "批量删除数据源")
    @PostMapping("/delData")
    public ResponseEntity<?> delData(@RequestBody QueryLabelSystemReq req) {
        try {
            if (CollectionUtils.isEmpty(req.getIds())) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "id cannot be empty");
            }
            return templateSystemDataService.delData(req);
        } catch (Exception e) {
            log.error("delData exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/system/data/temp/delData");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 保存和修改数据源
     *
     * @param req 请求实体
     * @return
     * @throws Exception
     */
    @Operation(summary = "保存和修改数据源")
    @PostMapping("/saveData")
    public ResponseEntity<?> saveData(@RequestBody @Valid TemplateSystemData req) {
        try {
            return templateSystemDataService.saveData(req);
        } catch (Exception e) {
            log.error("saveData exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/system/data/temp/saveData");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

}
