package com.digiwin.athena.semc.controller.temp;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.dto.portal.QueryImportRecordReq;
import com.digiwin.athena.semc.entity.portal.LabelSystemDataImport;
import com.digiwin.athena.semc.entity.temp.TemplateSystemDataImport;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.service.common.ImportTemplateService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.service.portal.impl.SystemDataWriteHandler;
import com.digiwin.athena.semc.service.temp.TemplateSystemDataService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.portal.ImportExcelResp;
import com.digiwin.athena.semc.vo.portal.SystemDataExportVo;
import com.digiwin.athena.semc.vo.portal.SystemDataImportVo;
import com.digiwin.athena.semc.vo.portal.SystemDataJobExportVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.util.List;

/**
 * @description: ISV数据源导入Controller
 */
@Slf4j
@RestController
@RequestMapping("/semc/system/data/temp")
public class TemplateSystemDataImportController {

    @Autowired
    TemplateSystemDataService templateSystemDataService;

    @Resource
    private MessageUtils messageUtils;


    /**
     * 导入数据源
     *
     * @return
     */
    @PostMapping("/importReport")
    public ResponseEntity<BaseResultDTO<ImportExcelResp>> importReport(@RequestBody LabelSystemDataImport req) {
        //校验文件id
        if (StringUtils.isEmpty(req.getFileId())) {
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, "file code cannot be empty");
        }
        ImportExcelResp excelResp = new ImportExcelResp();
        try {
            // 读取报表内容
            List<SystemDataImportVo> excelVoList = templateSystemDataService.readExcel(req.getFileId());
            if (CollectionUtils.isEmpty(excelVoList)) {
                String error = messageUtils.getMessage("error.message.import.cannot.be.empty");
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, error);
            }
            //去掉前后空格
            excelVoList=trimDataList(excelVoList);
            excelResp = templateSystemDataService.importReport(excelVoList);
            return ResponseEntityWrapperUtil.wrapperOk(excelResp);
        } catch (Exception e) {
            log.error("TemplateSystemDataImportController import excel exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 去掉空格
     * @param excelVoList
     * @return
     */
    public List<SystemDataImportVo> trimDataList(List<SystemDataImportVo> excelVoList){
        for(SystemDataImportVo excelVo:excelVoList){
            if(StringUtils.isNotEmpty(excelVo.getAppName())){
                excelVo.setAppName(excelVo.getAppName().trim());
            }
            if(StringUtils.isNotEmpty(excelVo.getCategory())){
                excelVo.setCategory(excelVo.getCategory().trim());
            }
            if(StringUtils.isNotEmpty(excelVo.getJobType())){
                excelVo.setJobType(excelVo.getJobType().trim());
            }
            if(StringUtils.isNotEmpty(excelVo.getJobVal())){
                excelVo.setJobVal(excelVo.getJobVal().trim());
            }
            if(StringUtils.isNotEmpty(excelVo.getName())){
                excelVo.setName(excelVo.getName().trim());
            }
            if(StringUtils.isNotEmpty(excelVo.getRemark())){
                excelVo.setRemark(excelVo.getRemark().trim());
            }
            if(StringUtils.isNotEmpty(excelVo.getValidStatus())){
                excelVo.setValidStatus(excelVo.getValidStatus().trim());
            }
        }
        return excelVoList;
    }
    /**
     * 查询导入记录
     *
     * @param queryImportRecordReq 入参
     * @return
     */
    @PostMapping("/queryImportRecord")
    public ResponseEntity<BaseResultDTO<PageInfoResp<TemplateSystemDataImport>>> queryImportRecord(@RequestBody @Valid QueryImportRecordReq queryImportRecordReq) {
        try {
            // 校验时间格式
            if ((StringUtils.isNotBlank(queryImportRecordReq.getStartTime()) && !DateUtils.validateDate(queryImportRecordReq.getStartTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) || (StringUtils.isNotBlank(queryImportRecordReq.getEndTime()) && !DateUtils.validateDate(queryImportRecordReq.getEndTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER))) {
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
            }
            PageInfoResp<TemplateSystemDataImport> pageInfoResp = templateSystemDataService.queryImportRecord(queryImportRecordReq);
            return ResponseEntityWrapperUtil.wrapperOk(pageInfoResp);
        } catch (Exception e) {
            log.error("queryImportRecord exception", e);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.COMMON_SYSTEM_ERROR));
        }
    }

    /**
     * 导出数据源
     *
     * @return
     */
    @GetMapping("/exportReport")
    public void exportReport(HttpServletResponse response) {
        //设置下载信息
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");
        String filename = "数据源配置_" + LocalDate.now().toString().replace("-", "") + ".xlsx";
        response.setHeader("Content-disposition", "attachment;filename=" + new String(filename.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1));
        try {
            List<SystemDataExportVo> excelVoList = templateSystemDataService.exportReport();
            //设置文件名
            EasyExcel.write(response.getOutputStream(), SystemDataExportVo.class)
                    .sheet("数据源配置").relativeHeadRowIndex(1)
                    .registerWriteHandler(Utils.initHorizontalCell())
                    .registerWriteHandler(new SystemDataWriteHandler()).doWrite(excelVoList);
        } catch (Exception e) {
            log.error("exportReport exception", e);
            throw new RuntimeException(e);
        }
    }

}