package com.digiwin.athena.semc.controller.temp;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.BizException;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.dto.portal.LabelSystemDataSourceUpdateReq;
import com.digiwin.athena.semc.dto.portal.LabelSystemPreQueryReq;
import com.digiwin.athena.semc.dto.portal.LabelSystemUpdateValidReq;
import com.digiwin.athena.semc.service.temp.TemplateSystemPreService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.portal.LabelSystemAllVo;
import com.digiwin.athena.semc.vo.portal.LabelSystemPreCustomVo;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.validation.Valid;


/**
 * @author CR-7
 * create: 2024-01-23 15:29
 * Description: 预设组件
 */
@RestController
@Slf4j
@RequestMapping("/semc/preset/component/temp")
public class TemplateSystemPreController {


    @Resource
    TemplateSystemPreService templateSystemPreService;


    @Resource
    private MessageUtils messageUtils;


    /**
     * 查询预设组件列表
     *
     * @param labelSystemPreQueryReq
     * @return
     */
    @Operation(method = "pageQuery", description = "查询预设组件列表")
    @PostMapping("/pageQuery")
    public ResultPageBean pageQuery(@RequestBody @Valid LabelSystemPreQueryReq labelSystemPreQueryReq) {
        try {
            return templateSystemPreService.pageQuery(labelSystemPreQueryReq);
        } catch (Exception e) {
            return ResultPageBean.bizException(BizException.getDefaultBizException(ErrorCodeConstant.LABEL_SYSTEM_PRE,
                    messageUtils.getMessage(I18NKey.SYSTEM_PRE)));
        }
    }


    /**
     * 更新组件数据源
     *
     * @param labelSystemUpdateReq
     * @return
     */
    @Operation(method = "updateDataSource", description = "更新组件数据源")
    @PostMapping("/updateDataSource")
    public ResponseEntity<?> updateDataSource(@RequestBody @Valid LabelSystemDataSourceUpdateReq labelSystemUpdateReq) {

        if (CollectionUtils.isEmpty(labelSystemUpdateReq.getDataSourceInfo())) {
            String error = messageUtils.getMessage("error.message.system.data.source.empty");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        try {
            return templateSystemPreService.updateDataSource(labelSystemUpdateReq);
        } catch (Exception e) {
            log.error("temp update dataSource error", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.UPDATE_SYSTEM_DATA_SOURCE, messageUtils.getMessage(I18NKey.DATASOURCE));
        }
    }


    /**
     * 更新启用停用状态
     *
     * @param labelSystemUpdateValidReq
     * @return
     */
    @Operation(method = "updateValidStatus", description = "更新启用停用状态")
    @PostMapping("/updateValidStatus")
    public ResponseEntity<?> updateValidStatus(@RequestBody @Valid LabelSystemUpdateValidReq labelSystemUpdateValidReq) {
        try {
            return templateSystemPreService.updateValidStatus(labelSystemUpdateValidReq.getId(),
                    labelSystemUpdateValidReq.getValidStatus(), labelSystemUpdateValidReq.getComponentSource());
        } catch (Exception e) {
            log.error("temp update validStatus error", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.UPDATE_SYSTEM_VALID_STATUS, messageUtils.getMessage(I18NKey.VALID_STATUS));
        }
    }


    /**
     * 查询预设组件和自定义组件列表
     *
     * @param req
     * @return
     */
    @Operation(method = "queryAllSystem", description = "查询预设组件和自定义组件列表")
    @PostMapping("/queryAllSystem")
    public ResponseEntity<BaseResultDTO<LabelSystemPreCustomVo>> queryAllSystem(@RequestBody LabelSystemAllVo req) {
        return ResponseEntityWrapperUtil.wrapperOk(templateSystemPreService.queryAllSystem(req));
    }
}
