package com.digiwin.athena.semc.controller.temp;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.*;
import com.digiwin.athena.semc.dto.portal.TemplateQueryReq;
import com.digiwin.athena.semc.entity.temp.TemplateUserMapping;
import com.digiwin.athena.semc.entity.temp.TemplateUserTenant;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.AppTenantDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.TenantInfoDTO;
import com.digiwin.athena.semc.service.temp.TemplateUserMappingService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * @description 账号租户关联管理
 * @author: dengkaiwen
 */
@RestController
@Slf4j
@RequestMapping("/semc/temp/user/mapping/")
public class TemplateUserMappingController {

    @Autowired
    TemplateUserMappingService templateUserMappingService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private IamService iamService;


    /**
     * 查询应用下全量租户
     *
     * @return
     */
    @PostMapping("/queryAppAllTenant")
    public ResponseEntity<?> queryAppAllTenant(@RequestBody TemplateUserMapping req) {
        if (StringUtils.isEmpty(req.getAppCode())) {
            String error = messageUtils.getMessage(I18NKey.USER_MAPPING_APPCODE_MISSING);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, error);
        }
        try {
            List<String> appIds=new ArrayList<>();
            appIds.add(req.getAppCode());
            List<AppTenantDTO> tenantInfoDTOS = iamService.queryTenantByApp(appIds);
            if (CollectionUtils.isNotEmpty(tenantInfoDTOS)){
                return ResponseEntityWrapper.wrapperOk(tenantInfoDTOS.get(0).getTenants());
            }
            return ResponseEntityWrapper.wrapperOk(tenantInfoDTOS);
        } catch (Exception e) {
            log.error("queryAppAllTenant exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/mapping/selMappingUserList");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
    /**
     * 查询已关联账号列表
     *
     * @return
     */
    @PostMapping("/selMappingUserList")
    public ResponseEntity<BaseResultDTO<List<TemplateUserMapping>>> selMappingUserList(@RequestBody TemplateQueryReq req ) {

        try {
            List<TemplateUserMapping> userMappingList = templateUserMappingService.selAllMappingUser(req.getSubUserNameId());
            return ResponseEntityWrapperUtil.wrapperOk(userMappingList);
        } catch (Exception e) {
            log.error("selMappingUserList exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/mapping/selMappingUserList");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 分页查询账号关联租户列表
     *
     * @param
     * @return
     */
    @Operation(method = "/pageQueryUserTenant", description = "分页查询账号关联租户列表")
    @PostMapping("/pageQueryUserTenant")
    public ResultPageBean pageQueryUserTenant(@RequestBody TemplateQueryReq req) {
        if(null == req.getPageNum()){
            req.setPageNum(Constants.pageNum);
        }
        if(null == req.getPageSize()){
            req.setPageSize(Constants.DEF_PAGE_SIZE);
        }
        //ID为空校验
        if(null == req.getId()){
            return ResultPageBean.bizException(BizException.getDefaultBizException(ErrorCodeConstant.PARAM_ILLEGAL_ERROR,
                    messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING)));
        }
        return templateUserMappingService.pageQueryUserTenant(req);
    }

    /**
     * 分页查询当前登录用户关联租户列表
     *
     * @param
     * @return
     */
    @Operation(method = "/pageLongUserTenant", description = "分页查询当前登录用户关联租户列表")
    @PostMapping("/pageLongUserTenant")
    public ResultPageBean pageLongUserTenant(@RequestBody TemplateQueryReq req) {
        if(null == req.getPageNum()){
            req.setPageNum(Constants.pageNum);
        }
        if(null == req.getPageSize()){
            req.setPageSize(Constants.DEF_PAGE_SIZE);
        }
        return templateUserMappingService.pageLongUserTenant(req);
    }


    /**
     * 查询登录账号关联的全量租户
     *
     * @param
     * @return
     */
    @Operation(method = "/getLongUserTenant", description = "查询登录账号关联的全量租户")
    @PostMapping("/getLongUserTenant")
    public ResponseEntity<BaseResultDTO<List<TemplateUserTenant>>>   getLongUserTenant() {
        return ResponseEntityWrapperUtil.wrapperOk(templateUserMappingService.getLongUserTenant());
    }


    /**
     * 查询账号关联的全量租户
     *
     * @param
     * @return
     */
    @Operation(method = "/queryUserAllTenant", description = "查询账号关联的全量租户")
    @PostMapping("/queryUserAllTenant")
    public ResponseEntity<BaseResultDTO<List<TemplateUserTenant>>>  queryUserAllTenant(@RequestBody TemplateQueryReq req) {
        //ID为空校验
        if(null == req.getId()){
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id is null");
        }
        return ResponseEntityWrapperUtil.wrapperOk(templateUserMappingService.queryUserAllTenant(req.getId()));
    }

    /**
     * 批量保存账号关联租户信息
     *
     * @param req
     * @return
     */
    @Operation(method = "saveUserTenant", description = "批量保存账号关联租户信息")
    @PostMapping("saveUserTenant")
    public ResponseEntity<?> saveUserTenant(@RequestBody List<TemplateUserMapping> req) {
        //检查入参合法性
        ResponseEntity<?> responseEntity = checkSaveUser(req);
        if (null != responseEntity) {
            return responseEntity;
        }

        try {
            templateUserMappingService.saveUserTenant(req);
        } catch (Exception e) {
            log.error("saveUserTenant exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/mapping/saveUserTenant");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 校验入参的合法性
     * @param req
     * @return
     */
   public ResponseEntity<?> checkSaveUser(List<TemplateUserMapping> req) {
       if(CollectionUtils.isEmpty(req)){
           return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "req is null");
       }
       for (TemplateUserMapping mapping : req) {
           if (StringUtils.isEmpty(mapping.getSubUserId())) {
               return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "subUserId is null");
           }
           mapping.setModifyUserId(Utils.getUserId());
           mapping.setModifyUserName(Utils.getUserName());
           mapping.setModifyTime(DateUtils.getNowTime(""));
           mapping.setAppCode(mapping.getAppCode());
           for(TemplateUserTenant tenant:mapping.getSubTenantList()){
               if (StringUtils.isEmpty(tenant.getSubTenantId())) {
                   return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "subTenantId is null");
               }
               tenant.setCreateUserName(Utils.getUserName());
               tenant.setTenantId(Utils.getTenantId());
               tenant.setCreateUserId(Utils.getUserId());
               tenant.setCreateTime(DateUtils.getNowTime(""));
               tenant.setModifyTime(DateUtils.getNowTime(""));
               tenant.setModifyUserId(Utils.getUserId());
               tenant.setModifyUserName(Utils.getUserName());
           }
       }
       return null;
   }

    /**
     * 删除绑定关系账号
     *
     * @param req
     * @return
     */
    @Operation(method = "delUserMapping", description = "删除绑定关系账号")
    @PostMapping("delUserMapping")
    public ResponseEntity<?> delUserMapping(@RequestBody TemplateUserMapping req) {
        if (null == req.getId()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id is null");
        }
        try {
            templateUserMappingService.delUserMapping(req.getId());
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("delUserMapping exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/mapping/delUserMapping");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
    /**
     * 批量删除用户关联下的租户
     *
     * @param req
     * @return
     */
    @Operation(method = "delUserTenant", description = "批量删除用户关联下的租户")
    @PostMapping("delUserTenant")
    public ResponseEntity<?> delUserTenant(@RequestBody TemplateQueryReq req) {
        if (CollectionUtils.isEmpty(req.getIdList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "idList is null");
        }
        try {
            templateUserMappingService.delUserTenant(req.getIdList());
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("delUserTenant exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/mapping/delUserTenant");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}
