package com.digiwin.athena.semc.controller.temp;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.*;
import com.digiwin.athena.semc.dto.portal.TemplateQueryReq;
import com.digiwin.athena.semc.entity.temp.TemplateTenantGroup;
import com.digiwin.athena.semc.entity.temp.TemplateTenantGroupItem;
import com.digiwin.athena.semc.service.temp.TemplateTenantGroupService;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.vo.temp.TemplateSaveVO;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;

/**
 * @description 租户分组管理
 * @author: dengkaiwen
 */
@RestController
@Slf4j
@RequestMapping("/semc/temp/user/tenant/group/")
public class TemplatetTenantGroupController {

    @Autowired
    TemplateTenantGroupService templateTenantGroupService;

    @Resource
    private MessageUtils messageUtils;




    /**
     * 保存分组
     *
     * @param req
     * @return
     */
    @Operation(method = "saveGroup", description = "保存分组")
    @PostMapping("saveGroup")
    public ResponseEntity<BaseResultDTO<TemplateSaveVO>> saveGroup(@RequestBody TemplateTenantGroup req) {
        TemplateSaveVO respVo=new TemplateSaveVO();
        respVo.setId(req.getId());
        //校验名称
        if (StringUtils.isEmpty(req.getName()) || StringUtils.isEmpty(req.getName().trim())) {
            String error = String.format(messageUtils.getMessage("error.message.job.bench.name.empty"), 20);
            respVo.setMsg(error);
            return ResponseEntityWrapperUtil.wrapperOk(respVo);
        }
        //分组名称长度校验
        if (req.getName().length() > 20) {
            String error = String.format(messageUtils.getMessage("error.message.name.length.max"), 20);
            respVo.setMsg(error);
            return ResponseEntityWrapperUtil.wrapperOk(respVo);
        }
        List<TemplateTenantGroup> tenantGroupList = templateTenantGroupService.selGroupBy(req);
        //校验分组名称是否存在
        if (CollectionUtils.isNotEmpty(tenantGroupList)) {
            String error = messageUtils.getMessage("error.message.template.group.exist.msg");
            respVo.setMsg(error);
            return ResponseEntityWrapperUtil.wrapperOk(respVo);
        }
        Long id=req.getId();
        try {
            id=templateTenantGroupService.saveTenantGroup(req);
            respVo.setId(id);
        } catch (Exception e) {
            log.error("saveGroup exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/tenant/group/saveGroup");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        return ResponseEntityWrapperUtil.wrapperOk(respVo);
    }


    /**
     * 保存分组下租户
     *
     * @param req
     * @return
     */
    @Operation(method = "saveGroupTenant", description = "保存分组下租户")
    @PostMapping("saveGroupTenant")
    public ResponseEntity<?> saveGroupTenant(@RequestBody TemplateTenantGroup req) {
        //检查入参合法性
        ResponseEntity<?> responseEntity = checkSaveGroup(req);
        if (null != responseEntity) {
            return responseEntity;
        }
        try {
            templateTenantGroupService.saveGroupTenant(req);
        } catch (Exception e) {
            log.error("saveGroupTenant exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/tenant/group/saveGroupTenant");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 校验入参的合法性
     * @param req
     * @return
     */
    public ResponseEntity<?> checkSaveGroup(TemplateTenantGroup req) {
        //分组ID校验
        if (null == req.getId()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id is null");
        }
        TemplateTenantGroup tenantGroup=templateTenantGroupService.getById(req.getId());
        //分组不存在
        if (null == tenantGroup) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "group is not exist");
        }
        //分组下的租户
        if (CollectionUtils.isEmpty(req.getSubTenantList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "subTenantList is null");
        }
        for(TemplateTenantGroupItem tenant:req.getSubTenantList()){
            //校验分组下的租户ID
            if(StringUtils.isEmpty(tenant.getSubTenantId())){
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "subTenantId is null");
            }
            if(StringUtils.isEmpty(tenant.getSubTenantName())){
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "subTenantName is null");
            }
        }
        return null;
    }

    /**
     * 查询分组下全量租户
     *
     * @return
     */
    @PostMapping("/queryGroupTenant")
    public ResponseEntity<BaseResultDTO<List<TemplateTenantGroupItem>>> queryGroupTenant(@RequestBody TemplateTenantGroupItem req) {
        if (null == req.getGroupId()) {
            String error = messageUtils.getMessage(I18NKey.USER_MAPPING_APPCODE_MISSING);
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_MISSING_ERROR, error);
        }
        try {
            List<TemplateTenantGroupItem> groupItemList = templateTenantGroupService.queryGroupTenant(req.getGroupId());
            return ResponseEntityWrapperUtil.wrapperOk(groupItemList);
        } catch (Exception e) {
            log.error("queryGroupTenant exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/tenant/group/queryGroupTenant");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 查询所有的分组
     *
     * @return
     */
    @PostMapping("/selTenantGroup")
    public ResponseEntity<BaseResultDTO<List<TemplateTenantGroup>>> selTenantGroup(@RequestBody TemplateTenantGroup req) {
        try {
            List<TemplateTenantGroup> groupList = templateTenantGroupService.selTenantGroup(req);
            return ResponseEntityWrapperUtil.wrapperOk(groupList);
        } catch (Exception e) {
            log.error("selTenantGroup exceptiona", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/tenant/group/selTenantGroup");
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }

    /**
     * 分页查询分组下租户列表
     *
     * @param
     * @return
     */
    @Operation(method = "/pageQueryGroupTenant", description = "分页查询分组下租户列表")
    @PostMapping("/pageQueryGroupTenant")
    public ResultPageBean pageQueryGroupTenant(@RequestBody  TemplateQueryReq req) {
        if(null == req.getPageNum()){
            req.setPageNum(Constants.pageNum);
        }
        if(null == req.getPageSize()){
            req.setPageSize(Constants.DEF_PAGE_SIZE);
        }
        //ID为空校验
        if(null == req.getGroupId()){
            return ResultPageBean.bizException(BizException.getDefaultBizException(ErrorCodeConstant.PARAM_ILLEGAL_ERROR,
                    messageUtils.getMessage(I18NKey.COMMON_PARAM_MISSING)));
        }
        return templateTenantGroupService.pageQueryGroupTenant(req);
    }


    /**
     * 启用停用分组
     *
     * @param
     * @return
     */
    @Operation(method = "/updateStatus", description = "启用停用分组")
    @PostMapping("/updateStatus")
    public ResponseEntity<?>  updateStatus(@RequestBody TemplateQueryReq req) {
        if (null == req.getValidStatus()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "validStatus is null");
        }
        if (CollectionUtils.isEmpty(req.getIdList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "IdList is null");
        }
        try {
            templateTenantGroupService.updateStatus(req);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("updateStatus exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/tenant/group/updateStatus");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }


    /**
     * 删除分组
     *
     * @param req
     * @return
     */
    @Operation(method = "delGroup", description = "删除分组")
    @PostMapping("delGroup")
    public ResponseEntity<?> delGroup(@RequestBody TemplateQueryReq req) {
        if (null == req.getId()) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "id is null");
        }
        try {
            templateTenantGroupService.delGroup(req);
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("delGroup exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/tenant/group/delGroup");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }



    /**
     * 批量删除分组下租户
     *
     * @param req
     * @return
     */
    @Operation(method = "delGroupTenant", description = "批量删除分组下租户")
    @PostMapping("delGroupTenant")
    public ResponseEntity<?> delGroupTenant(@RequestBody TemplateQueryReq req) {
        if (CollectionUtils.isEmpty(req.getIdList())) {
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "idList is null");
        }
        try {
            templateTenantGroupService.delGroupTenant(req.getIdList());
            return ResponseEntityWrapper.wrapperOk();
        } catch (Exception e) {
            log.error("delGroupTenant exception", e);
            String error = String.format(messageUtils.getMessage(I18NKey.SYSTEM_ERROR), LocalDateTime.now(), "/semc/temp/user/tenant/group/delGroupTenant");
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, error);
        }
    }
}
