package com.digiwin.athena.semc.mq.listener;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.semc.configuration.RabbitConfig;
import com.digiwin.athena.semc.mq.dto.MQMessageDTO;
import com.digiwin.athena.semc.mq.strategy.WorkCommonMessageStrategyRunner;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.util.FormatUtil;
import com.rabbitmq.client.Channel;

import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Map;

import cn.hutool.core.thread.ThreadUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Component
@RequiredArgsConstructor
public class RabbitMessageListener {

    private final WorkCommonMessageStrategyRunner workCommonMessageStrategyRunner;

    private final ICacheService cacheService;


    @RabbitListener(queues = {RabbitConfig.SEMC_WORK_COMMON_QUEUE_NAME})
    public void onWorkCommonMessage(Message message, Channel channel) throws IOException {
        MQMessageDTO messageDTO = null;
        long deliveryTag = message.getMessageProperties().getDeliveryTag();
        boolean hasMdcContext=false;
        try {
            Map<String, String> mdcContext = (Map<String, String>) message.getMessageProperties().getHeaders().get("mdc_context");
            if (MapUtils.isNotEmpty(mdcContext)) {
                MDC.setContextMap(mdcContext);
                hasMdcContext=true;
            }
            String msg = new String(message.getBody(), StandardCharsets.UTF_8);
            log.info("onWorkCommonMessage：{}", msg);
            messageDTO= JSON.parseObject(msg, MQMessageDTO.class);
            workCommonMessageStrategyRunner.doHandler(messageDTO);
            //处理成功，手动确认消息
            channel.basicAck(deliveryTag, false);

        } catch (Exception e) {
            log.error("onWorkCommonMessageError：", e);
            if (messageDTO==null) {
                channel.basicNack(deliveryTag, false, false);
            }
            else {
                //不重试
                if (messageDTO.getRetryMaxCount() == null || messageDTO.getRetryMaxCount() <= 0) {
                    channel.basicNack(deliveryTag, false, false);
                }
                //重试
                else {
                    String retryCountKey = FormatUtil.format("retryCount:{}:{}", messageDTO.getMessageType(), messageDTO.getMessageId());
                    // 获取当前重试次数
                    String retryCountStr = cacheService.getValue(retryCountKey);
                    int retryCount = 0;
                    if (StringUtils.isNotBlank(retryCountStr)) {
                        retryCount = Integer.parseInt(retryCountStr);
                    }
                    if (retryCount <= messageDTO.getRetryMaxCount()) {
                        cacheService.cache(retryCountKey, String.valueOf(retryCount + 1), Duration.ofDays(1));
                        ThreadUtil.sleep(2000);
                        //不确认消息，并且重新入队，这样又可以重新消费，最后一个参数如果为false就表示不重回队列，进入死信队列了
                        channel.basicNack(deliveryTag, false, true);
                    } else {
                        cacheService.delete(retryCountKey);
                        channel.basicNack(deliveryTag, false, false);
                    }
                }
            }
        }
        finally {
            if (hasMdcContext) {
                MDC.clear();
            }
        }
    }

}
