package com.digiwin.athena.semc.proxy.athena.service.audc.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.semc.dto.BaseResultDTO;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.athena.service.audc.AudcService;
import com.digiwin.athena.semc.proxy.athena.service.audc.model.AuthorityMenuDTO;
import com.digiwin.athena.semc.proxy.athena.service.audc.model.SystemMenuDTO;
import com.digiwin.athena.semc.proxy.athena.service.audc.model.UserTrackExtendDTO;
import com.digiwin.athena.semc.util.Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 请求小AI相关接口
 */
@Service
@Slf4j
public class AudcServiceImpl implements AudcService {

    @Resource
    private EnvProperties envProperties;
    @Resource
    private RestTemplate restTemplate;

    /**
     * 查询所有系统菜单
     */
    private static final String QUERY_SYSTEM_MENU_URL = "/api/amtc/v1/commonUsed/getSystemMenu";

    /**
     * 查询用户级的菜单（作业和系统菜单）权限
     */
    private static final String QUERY_AUTHORITY_MENU_URL = "/api/amtc/v1/commonUsed/getAuthorityMenu";
    /**
     * 埋点接口
     */
    private static final String USER_TRACK_REPORT = "/api/v1/user/track/report";

    /**
     * 查询所有系统菜单
     *
     * @return 返回结果
     */
    @Override
    public List<SystemMenuDTO> querySystemMenu() {
        List<SystemMenuDTO> systemMenuDTOList = new ArrayList<>();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-user", Utils.getUserToken());
        // 请求体
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getBaseUrl()).append(QUERY_SYSTEM_MENU_URL);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.GET, requestEntity,
                    new ParameterizedTypeReference<JSONObject>() {
                    });
            log.info("query system menu end. url:{}, resp:{}", url, JSON.toJSONString(respEntity.getBody()));
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                BaseResultDTO baseResultDTO = JSON.parseObject(JSON.toJSONString(respEntity.getBody()), BaseResultDTO.class);
                systemMenuDTOList = JSON.parseArray(JSON.toJSONString(baseResultDTO.getResponse()), SystemMenuDTO.class);
            } else {
                log.error("query system menu fail, url:{}, statusCode:{}, resp:{}", url, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return systemMenuDTOList;
            }
        } catch (Exception e) {
            log.error("query system menu error, url:{}", url, e);
            return systemMenuDTOList;
        }
        return systemMenuDTOList;
    }

    /**
     * 查询用户级的菜单（作业和系统菜单）权限
     *
     * @return 返回结果
     * <p>
     * [
     * //基础资料
     * {
     * "code": "baseDataEntry_company_maintenance_common",
     * "type": "baseDataEntry",
     * }
     * <p>
     * //报表
     * {
     * <p>
     * "code": "91629aee-8aaf-4318-b57d-01e3f9eaca8f",
     * "type": "report",
     * }
     * <p>
     * //发起项目
     * {
     * "code": "ProjectBudgetCoordination_mainLineProject",
     * "type": "startProject"
     * }
     * <p>
     * //系统菜单
     * {
     * "code":"calendar",
     * "type": "sysMenu"
     * }
     * ]
     */
    @Override
    public List<AuthorityMenuDTO> queryAuthorityMenu(JSONArray reqBody) {
        List<AuthorityMenuDTO> authorityList = new ArrayList<>();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-user", Utils.getUserToken());

        // 请求体
        HttpEntity<?> requestEntity = new HttpEntity<>(reqBody, headers);
        StringBuilder url = new StringBuilder(envProperties.getBaseUrl()).append(QUERY_AUTHORITY_MENU_URL);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<JSONObject>() {
                    });
            log.info("query system authority end. url:{}, param:{}, resp:{}", url, requestEntity, JSON.toJSONString(respEntity.getBody()));
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                BaseResultDTO baseResultDTO = JSON.parseObject(JSON.toJSONString(respEntity.getBody()), BaseResultDTO.class);
                authorityList = JSON.parseArray(JSON.toJSONString(baseResultDTO.getResponse()), AuthorityMenuDTO.class);
            } else {
                log.error("query authority menu fail, url:{}, param:{}, statusCode:{}, resp:{}", url, requestEntity, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return authorityList;
            }
        } catch (Exception e) {
            log.error("query authority menu error, url:{}, param:{}", url, requestEntity, e);
            return authorityList;
        }
        return authorityList;
    }

    /**
     * 埋点
     *
     * @param authoredUser
     * @param list
     */
    @Override
    public void reportUserTrack(AuthoredUser authoredUser, List<UserTrackExtendDTO> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        String uri = envProperties.getBaseUrl() + USER_TRACK_REPORT;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("token", authoredUser.getToken());
        try {
            HttpEntity<?> httpEntity = new HttpEntity<>(JsonUtils.objectToString(list), headers);
            ResponseEntity<com.digiwin.athena.appcore.domain.BaseResultDTO<Object>> respEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<com.digiwin.athena.appcore.domain.BaseResultDTO<Object>>() {
            });
            // 请求失败，直接抛出异常
            if (respEntity.getStatusCode() == HttpStatus.OK) {
                Objects.requireNonNull(respEntity.getBody()).getResponse();
            } else {
                log.error("exception.audc.report.userTrack.fail", httpEntity);
            }
        } catch (Exception e) {
            log.error("exception.audc.report.userTrack.fail", e.getMessage());
        }
    }
}