package com.digiwin.athena.semc.proxy.cac.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.cac.service.CacApiConstant;
import com.digiwin.athena.semc.proxy.cac.service.CacService;
import com.digiwin.athena.semc.proxy.cac.service.dto.CacBaseResultDTO;
import com.digiwin.athena.semc.proxy.cac.service.dto.TenantAuthAppDTO;
import com.digiwin.athena.semc.proxy.cac.service.dto.UserAppDTO;
import com.digiwin.athena.semc.util.DateUtils;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * EocServiceImpl Description
 *
 * @author sungqz
 * @date 2023/8/7
 * @since
 */
@Service
@Slf4j
public class CacServiceImpl implements CacService {

    @Resource
    private EnvProperties envProperties;

    @Resource
    private RestTemplate restTemplate;

    /**
     * 调用鼎捷云接口校验租户应用是否过期
     *
     * @param tenantId 租户id
     * @param code     应用code
     * @return
     */
    public boolean verifyAppAuthorized(String tenantId, String code) {
        return verifyAppAuthorizedResult(tenantId, code)._2;
    }

    @Override
    public Tuple2<String, Boolean> verifyAppAuthorizedResult(String tenantId, String code) {
        String url = envProperties.getCacUri() + "/api/cac/v4/authorizations/tenants/" + tenantId + "/goods/" + code;
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            headers.add("digi-middleware-auth-app", envProperties.getAppToken());
            HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
            ResponseEntity<Map> response = restTemplate.exchange(url, HttpMethod.GET, requestEntity, Map.class);
            log.debug("verify if the app is authorized, tenantId:{}, code:{}, result:{}", tenantId, code, JsonUtils.objectToString(response));
            if (response.getBody() != null && response.getBody().containsKey("purchaseId")) {
                String expiredTime = String.valueOf(response.getBody().get("expiredTime"));
                boolean result = LocalDateTime.now().compareTo(LocalDateTime.parse(expiredTime, DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_NORMAL_FORMATTER))) < 0;
                return new Tuple2<>(code, result);
            }
        } catch (Exception ex) {
            log.error("verify if the app is authorized occur error, tenantId:{}, code:{}", tenantId, code, ex);
        }
        return new Tuple2<>(code, false);
    }

    /**
     * 查询当前登录用户可用的应用清单信息
     *
     * @return
     */
    public JSONArray queryCurrentUserAppList() {
        String url = envProperties.getCacUri() + "/api/cac/v4/authorizations/current/user/apps";
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            headers.add("digi-middleware-auth-app", envProperties.getAppToken());
            headers.add("digi-middleware-auth-user", userToken);
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.GET, new HttpEntity<>(null, headers), JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return respEntity.getBody().getJSONArray("data");
            } else {
                log.info("query current user app list return empty. url:{}, token:{}, statusCode:{}, response:{}", url, userToken, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("query current user app list occur error, userToken:{}, code:{}", userToken, ex);
        }
        return null;
    }

    @Override
    public List<UserAppDTO> queryUserApp(String userId) {
        String uri = envProperties.getCacUri() + CacApiConstant.AUTHORIZATIONS_CURRENT_USER_APPS_BY_USER_ID;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        // 入参
        Map<String, Object> params = new HashMap<>();
        params.put("userId", userId);
        try {
            ResponseEntity<CacBaseResultDTO<List<UserAppDTO>>> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, new HttpEntity<>(null, headers),
                    new ParameterizedTypeReference<CacBaseResultDTO<List<UserAppDTO>>>() {
                    }, params);
            if (null != responseEntity.getBody() && String.valueOf(HttpStatus.OK.value()).equals(responseEntity.getBody().getCode())) {
                return responseEntity.getBody().getData();
            }
        } catch (Exception ex) {
            log.error("execute cac [/authorizations/current/user/apps] fail, message = {}", ex.getMessage());
        }
        return null;
    }

    @Override
    public List<UserAppDTO> queryUserApp() {
        String uri = envProperties.getCacUri() + CacApiConstant.AUTHORIZATIONS_CURRENT_USER_APPS;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        try {
            ResponseEntity<CacBaseResultDTO<List<UserAppDTO>>> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, new HttpEntity<>(null, headers),
                    new ParameterizedTypeReference<CacBaseResultDTO<List<UserAppDTO>>>() {
                    });
            if (null != responseEntity.getBody() && String.valueOf(HttpStatus.OK.value()).equals(responseEntity.getBody().getCode())) {
                return responseEntity.getBody().getData();
            }
        } catch (Exception ex) {
            log.error("execute cac [/authorizations/current/user/apps] fail, message = {}", ex.getMessage());
        }
        return null;
    }

    @Override
    public TenantAuthAppDTO queryTenantApp(int expireType) {
        String uri = envProperties.getCacUri() + CacApiConstant.QUERY_CURRENT_TENANT_APP_AUTHORIZATIONS;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        Map<String, Object> body = new HashMap<>();
        body.put("queryExpire", expireType);
        try {
            ResponseEntity<CacBaseResultDTO<TenantAuthAppDTO>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, new HttpEntity<>(body, headers),
                    new ParameterizedTypeReference<CacBaseResultDTO<TenantAuthAppDTO>>() {
                    });
            if (null != responseEntity.getBody() && String.valueOf(HttpStatus.OK.value()).equals(responseEntity.getBody().getCode())) {
                return responseEntity.getBody().getData();
            }
        } catch (Exception ex) {
            log.error("execute cac [/authorizations/current/user/apps] fail, message = {}", ex.getMessage());
        }
        return null;
    }
}