package com.digiwin.athena.semc.proxy.dmc.service.impl;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.dmc.service.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.*;
import org.springframework.integration.metadata.ListenableMetadataStore;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.Resource;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * EocServiceImpl Description
 *
 * @author sungqz
 * @date 2023/8/2
 * @since
 */
@Service
@Slf4j
public class DmcServiceImpl implements DmcService {
    @Resource
    private EnvProperties envProperties;

    @Resource
    private RestTemplate restTemplate;

    @Autowired
    private DMCTokenBean dmcTokenBean;

    @Autowired
    private DmcConfig dmcConfig;

    /**
     * 登录
     */
    private static final String DMC_LOGIN_PATH = "/api/dmc/v1/auth/login";

    /**
     * 查询文件信息
     */
    private static final String DMC_FILE_INFO_PATH = "/api/dmc/v2/fileinfo/intelligententry/ids";

    /**
     * 登录DMC
     *
     * @return
     */
    public String loginDmc() {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put(Constants.USER_NAME, Constants.DMC_USER_NAME);
        requestMap.put(Constants.DMC_HASH, Constants.DMC_ENCRYPTION);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getDmcUri()).append(DMC_LOGIN_PATH);
        try {
            log.info("loginDmc param:" + JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return respEntity.getBody().getString("userToken");
            } else {
                log.error("loginDmc return fail, url:{}, status:{}, response:{}", url, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                throw BusinessException.create("loginDmc return fail");
            }
        } catch (Exception ex) {
            log.error("loginDmc error, url:{}, errorMessage:{}", url, ex.getMessage());
            throw BusinessException.create(ex.getMessage());
        }
    }

    /**
     * 批量查询文件信息
     *
     * @return 文件信息数组
     */
    @Override
    public JSONArray queryFileInfoList(List<String> fileIdList, String dmcUserToken) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-user", dmcUserToken);

        // 请求体
        HttpEntity<List<String>> requestEntity = new HttpEntity<>(fileIdList, headers);
        StringBuilder url = new StringBuilder(envProperties.getDmcUri()).append(DMC_FILE_INFO_PATH);
        try {
            log.info("query file info list, param:" + JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return respEntity.getBody().getJSONArray("data");
            } else {
                log.info("query file info list return empty, url:{}, status:{}, response:{}", url, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return new JSONArray();
            }
        } catch (Exception ex) {
            log.error("query file info list error, url:{}, errorMessage:{}", url, ex.getMessage());
            return new JSONArray();
        }
    }

    @Override
    public String upload(File file, String contentType, FileInfo fileInfo) {
        return this.upload(file.getName(), FileUtil.readBytes(file), contentType, fileInfo);
    }

    public String upload(String fileName, byte[] bytes, String contentType, FileInfo fileInfo) {
        dmcTokenBean.setToken(this.loginDmc());
        String dmcUrl = dmcConfig.getUrl();

        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        HttpHeaders header = new HttpHeaders();
        header.add("digi-middleware-auth-user", dmcTokenBean.getToken());

        List<String> fileIdList = new ArrayList<>();
        fileIdList.add(fileInfo.getId());
        JSONArray jsonArray = queryFileInfoList(fileIdList,dmcTokenBean.getToken());
        if (jsonArray != null && jsonArray.size() == 1){
            log.info("文件存在，文件id:{}",fileInfo.getId());
            return null;
        }

        HttpHeaders fileHeader = new HttpHeaders();
        fileHeader.setContentType(MediaType.parseMediaType(contentType));
        fileHeader.setContentDispositionFormData("file", fileName);

        HttpHeaders fileInfoHeader = new HttpHeaders();
        fileInfoHeader.setContentType(MediaType.APPLICATION_JSON);

        try {
            HttpEntity<ByteArrayResource> fileEntity = new HttpEntity<>(new ByteArrayResource(bytes),
                    fileHeader);
            JSONObject jsonObject = new JSONObject();
            jsonObject.fluentPut("id",fileInfo.getId());
            HttpEntity<JSONObject> fileInfoEntity = new HttpEntity<>(jsonObject, fileInfoHeader);
            multiValueMap.add("file", fileEntity);
            multiValueMap.add("fileInfo", fileInfoEntity);

            String url = dmcUrl + "/api/dmc/v2/file" + Constants.DMC_FILE_BUCKET + "/upload";

            HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<>(multiValueMap, header);

            log.info("调用DMC的url是: {}", url);
            log.info("调用DMC的header是: {}", JSONObject.toJSONString(header));
            ResponseEntity<MidWareResponse> postForEntity = restTemplate.postForEntity(url, httpEntity, MidWareResponse.class);
            MidWareResponse<Map> midWareResponse = postForEntity.getBody();
            if (midWareResponse.getSuccess()) {
                return (String) midWareResponse.getData().get("id");
            } else {
                log.error("调用上传api失败,返回数据:" + midWareResponse);
                throw BusinessException.create("Failed to call the upload API");
            }
        } catch (Exception ex) {
            if (ex instanceof HttpClientErrorException) {
                // 处理4xx错误
                HttpClientErrorException httpClientErrorException = (HttpClientErrorException) ex;
                int statusCode = httpClientErrorException.getStatusCode().value();
                if (statusCode == 401) {
                    // 401判定为用户token失效，需要重新获取用户token
                    this.dmcTokenBean.setToken(this.loginDmc());
                }
                log.error("dmc用户token失效，错误信息：" + ex.getMessage(), ex);
                throw BusinessException.create("dmc用户token失效！用户token现已更新，请重新编译");
            } else {
                log.error("调用DMC上传文件api失败，错误信息：" + ex.getMessage(), ex);
                throw BusinessException.create("调用DMC上传文件api失败，错误信息：" + ex.getMessage(), ex);
            }
        }
    }
}