package com.digiwin.athena.semc.proxy.emc.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.emc.service.EmcService;
import com.digiwin.athena.semc.proxy.emc.service.model.OmEventResp;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * EmcServiceImpl Description
 *
 * @author sungqz
 * @since 2023-1-8
 */
@Service
@Slf4j
public class EmcServiceImpl implements EmcService {

    @Resource
    private EnvProperties envProperties;

    @Resource
    private RestTemplate restTemplate;

    private static final String EMC_OM_TEMPLATE = "/api/emc/v2/notices/settings/entrance";

    /**
     * 查询企微配置接口
     */
    private static final String EMC_WECOM_CONFIG = "/api/emc/v1/appconfig/outsource/current";

    /**
     * 请求emc发送企微消息接口
     */
    public static final String EMC_MESSAGE_WECHAT = "/api/emc/v1/message/sendWechat";

    /**
     * 查询om平台事件及模板列表
     *
     * @return 查询om平台事件及模板列表
     */
    @Override
    public List<OmEventResp> queryOmTemplateList(String channelFlag, String nameConditon) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("source", "Athena");
        requestMap.put("type", channelFlag.equals(Constants.MessageChannelEnum.WECOM.getFlag()) ? "weCom" : channelFlag);
        requestMap.put("noticeContent", nameConditon);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getEmcUrl()).append(EMC_OM_TEMPLATE);
        try {
            log.info("query om template list. param:{}", JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONArray data = respEntity.getBody().getJSONArray("data");
                return JSONArray.parseArray(data.toJSONString(), OmEventResp.class);
            } else {
                log.info("query om template list return empty, userToken:{}, request:{}, url:{}, code:{}, response:{}", userToken,
                        requestMap, url, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return Lists.newArrayList();
            }
        } catch (Exception ex) {
            log.error("query om template list error, userToken:{}, request:{}, url:{}, errorMessage:{}", userToken, requestMap, url, ex.getMessage());
            throw BusinessException.create(ex.getMessage());
        }
    }

    /**
     * 查询企微配置
     *
     * @param userToken 用户token
     * @return 返回
     */
    @Override
    public JSONObject queryWeComConfig(String userToken) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 组装请求体
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        String url = envProperties.getEmcUrl() + EMC_WECOM_CONFIG;

        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONArray data = respEntity.getBody().getJSONArray("data");
                JSONObject jsonObject = data.getJSONObject(0);
                return jsonObject;
            } else {
                log.info("query wecom config return empty. url:{}, request:{}, response:{}", url, requestEntity, JSONObject.toJSONString(respEntity.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("query wecom config occur error. url:{}, request:{}, errorMessage:{}", url, requestEntity, ex.getMessage(), ex);
            return null;
        }
    }

    /**
     * 发送企微消息
     *
     * @param userIdList 消息发送用户列表
     * @param msg        消息内容
     * @param userToken  userToken
     */
    @Override
    public void sendWeChat(List<String> userIdList, String msg, String userToken) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", userToken);

        // 构建请求体
        JSONObject reqObject = new JSONObject();
        reqObject.put("msgType", "text");
        reqObject.put("content", msg);
        reqObject.put("userIds", userIdList);
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(reqObject, headers);
        String url = envProperties.getEmcUrl() + EMC_MESSAGE_WECHAT;
        try {
            ResponseEntity<Object> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Object>() {
            });
            log.info("invoke emc | send WeChat success. url:{}, request:{}, response：{}", url, JsonUtils.objectToString(httpEntity), JsonUtils.objectToString(respEntity.getBody()));
        } catch (Exception ex) {
            log.error("invoke emc | send WeChat error. url:{}, request:{}", url, JsonUtils.objectToString(httpEntity), ex);
        }
    }
}