package com.digiwin.athena.semc.proxy.eoc.service.impl;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.semc.proxy.eoc.service.model.UserDeptDTO;
import com.google.common.collect.Lists;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.eoc.service.EocService;
import com.digiwin.athena.semc.proxy.eoc.service.model.UserDeptInfoDTO;
import com.digiwin.athena.semc.proxy.eoc.service.model.UserInfoDetailDTO;
import com.digiwin.athena.semc.util.FormatUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;
import org.springframework.web.client.RestTemplate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * EocServiceImpl Description
 *
 * @author sungq
 * @date 2023/7/18
 * @since
 */
@Service
@Slf4j
public class EocServiceImpl implements EocService {

    @Resource
    private EnvProperties envProperties;

    @Resource
    private RestTemplate restTemplate;

    private static final String EOC_EMP_DEPTS = "/api/eoc/v2/emp/depts";

    //员工工号
    private static final String EOC_EMP_JOB = "/api/eoc/v2/emp/id";

    /**
     * 获取员工详情的接口
     * 接口文档：http://172.16.2.100:22694/html/web/controller/share/share.html#5fd1d3d43eb9e9355be5a3a3
     */
    private static final String EOC_EMP_INFO = "/api/eoc/v2/emp/info";

    /**
     * 批量查询多个部门下的员工
     * 接口文档：https://doc.apipost.net/docs/detail/3aa4f95cbc02000?target_id=3aa47c762407000&locale=zh-cn
     */
    private static final String QUERY_DEPT_USER_INFO_EOC = "/api/eoc/v2/dept/emp/list/emp/simple?deptSids={}&bindUser=true";

    /**
     * 查询当前租户下部门列表
     */
    private static final String CURRENT_TENANT_DEPTS = "/api/eoc/v2/dept/list?pageNum=1&pageSize=99999";

    /**
     * 获取员工所在部门以及所有的上级部门
     *
     * @return 员工所在部门以及所有的上级部门
     */
    @Override
    public List<UserDeptInfoDTO> queryUserDeptList(String empId) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("empId", empId);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getEocDomain()).append(EOC_EMP_DEPTS);
        try {
            log.info("query user dept list param:" + JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONArray data = respEntity.getBody().getJSONArray("data");
                return JSONArray.parseArray(data.toJSONString(), UserDeptInfoDTO.class);
            } else {
                log.info("query user dept list return empty, userToken:{}, request:{}, url:{}, code:{}, response:{}", userToken,
                        requestMap, url, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return Lists.newArrayList();
            }
        } catch (Exception ex) {
            log.error("query user dept list error, userToken:{}, request:{}, url:{}, errorMessage:{}", userToken, requestMap, url, ex.getMessage());
            throw BusinessException.create(ex.getMessage());
        }
    }

    @Override
    public String getEmpJobNum() {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        String userId = AppAuthContextHolder.getContext().getAuthoredUser().getUserId();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("userId", userId);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getEocDomain()).append(EOC_EMP_JOB);
        try {
            log.info("query user getEmpJobNum  param:" + JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                String data = respEntity.getBody().getString("data");
                return data;
            } else {
                log.info("query user getEmpJobNum return empty, userToken:{}, request:{}, url:{}, code:{}, response:{}", userToken,
                        requestMap, url, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return "";
            }
        } catch (Exception ex) {
            log.error("query user getEmpJobNum error, userToken:{}, request:{}, url:{}, errorMessage:{}", userToken, requestMap, url, ex.getMessage());
            return "";
        }
    }

    /**
     * 获取员工详情的接口
     *
     * @param userId 用户id
     * @return 返回
     */
    @Override
    public UserInfoDetailDTO queryUserInfoDetail(String userId) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("userId", userId);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getEocDomain()).append(EOC_EMP_INFO);
        try {
            log.info("query user info detail param:" + JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONObject jsonObject = respEntity.getBody().getJSONObject("data");
                return JSONObject.parseObject(JSONObject.toJSONString(jsonObject), UserInfoDetailDTO.class);
            } else {
                log.info("query user info detail return empty. url:{}, request:{}, response:{}", url, requestEntity, JSONObject.toJSONString(respEntity));
                return null;
            }
        } catch (Exception ex) {
            log.warn("query user info detail errorMessage:{}", ex.getMessage(), ex);
            return null;
        }
    }

    @Override
    public List<Long> queryUserDeptSidList(String userId) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("userId", userId);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getEocDomain()).append(EOC_EMP_INFO);
        try {
            log.info("query user info detail param:" + JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONObject jsonObject = respEntity.getBody().getJSONObject("data");
                List<Long> deptSidList = Lists.newArrayList();
                if (MapUtils.isEmpty(jsonObject)) {
                    return deptSidList;
                }
                JSONArray depts = jsonObject.getJSONArray("depts");
                if (CollectionUtils.isEmpty(depts)) {
                    return deptSidList;
                }
                for (int i = 0; i < depts.size(); i++) {
                    deptSidList.add(depts.getJSONObject(i).getLong("sid"));
                }
                return deptSidList;

            } else {
                log.info("query user info detail return empty. url:{}, request:{}, response:{}", url, requestEntity, JSONObject.toJSONString(respEntity));
                return Lists.newArrayList();
            }
        } catch (Exception ex) {
            log.warn("query user info detail errorMessage:{}", ex.getMessage(), ex);
            return Lists.newArrayList();
        }
    }

    /**
     * 批量查询部门下用户列表
     *
     * @param deptSidList 部门sid列表
     * @return 返回对象
     */
    @Override
    public List<UserDeptInfoDTO> queryUserByDeptSidList(List<Long> deptSidList) {
        if (CollectionUtils.isEmpty(deptSidList)) {
            return Lists.newArrayList();
        }
        // url
        String deptSidStr = deptSidList.stream().map(Objects::toString).collect(Collectors.joining(",", "", ""));
        String url = envProperties.getEocDomain() + FormatUtil.format(QUERY_DEPT_USER_INFO_EOC, deptSidStr);

        // header
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // body
        HttpEntity<?> httpEntity = new HttpEntity<>(null, headers);
        ResponseEntity<JSONObject> respEntity;
        try {
            respEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, JSONObject.class);
        } catch (Exception ex) {
            log.error("query user by dept sid list error, url:{}, request:{}, errorMsg:{}", url, httpEntity, ex.getMessage(), ex);
            return Lists.newArrayList();
        }
        // 处理返回对象
        if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
            JSONArray data = respEntity.getBody().getJSONArray("data");
            return JSONObject.parseArray(data.toJSONString(), UserDeptInfoDTO.class);
        } else {
            log.warn("query user by role dept list return empty, url:{}, request:{}, statusCode:{}, response:{}",
                    url, httpEntity, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
            return Lists.newArrayList();
        }
    }

    /**
     * 查询当前租户下部门列表
     *
     * @return 返回
     */
    public List<UserDeptDTO> queryTenantDeptList() {
        // url
        String url = envProperties.getEocDomain() + CURRENT_TENANT_DEPTS;

        // header
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // body
        HttpEntity<?> httpEntity = new HttpEntity<>(null, headers);
        ResponseEntity<JSONObject> respEntity;
        try {
            respEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, JSONObject.class);
        } catch (Exception ex) {
            log.error("query tenant dept list error, url:{}, request:{}, errorMsg:{}", url, httpEntity, ex.getMessage(), ex);
            return Lists.newArrayList();
        }
        // 处理返回对象
        if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
            JSONObject jsonObject = respEntity.getBody().getJSONObject("data");
            if (!ObjectUtils.isEmpty(jsonObject)) {
                return JSONObject.parseArray(jsonObject.getJSONArray("list").toJSONString(), UserDeptDTO.class);
            }
        } else {
            log.warn("query tenant dept list return empty, url:{}, request:{}, statusCode:{}, response:{}",
                    url, httpEntity, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
        }
        return Lists.newArrayList();
    }
}