package com.digiwin.athena.semc.proxy.es.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.semc.dto.homepage.AdvertisementDTO;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.es.service.EsService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * EsServiceImpl Description
 *
 * @author sungqz
 * @date 2023/8/23
 * @since
 */
@Service
@Slf4j
public class EsServiceImpl implements EsService {

    @Resource
    private EnvProperties envProperties;

    @Resource
    private RestTemplate restTemplate;

    /**
     * 获取广告列表
     */
    private static final String ES_QUERY_ADS_PATH = "/announcementservice/api/ads/escloud?serviceRegion=%s&adLaunchArea=%s&accessToken=%s";

    /**
     * 阅读广告记录
     */
    private static final String ES_READ_ADS_PATH = "/announcementservice/api/ads/read?accessToken=";

    /**
     * 调用es服务获取广告列表
     *
     * @param serviceRegion 服务中心
     * @param adLaunchArea  广告投放区域
     * @param esAccessToken esAccessToken
     * @return
     */
    @Override
    public List<AdvertisementDTO> queryAdList(String serviceRegion, String adLaunchArea, String esAccessToken) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        // 请求体
        String invokePath = String.format(ES_QUERY_ADS_PATH, serviceRegion, adLaunchArea, esAccessToken);
        HttpEntity<List<String>> requestEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getEsUri()).append(invokePath);
        // StringBuilder url = new StringBuilder("http://es-api-pre.digiwincloud.com.cn").append(invokePath);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.GET, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONArray array = respEntity.getBody().getJSONArray("ads");
                return JSONArray.parseArray(array.toJSONString(), AdvertisementDTO.class);
            } else {
                log.info("query advertisement list from es return empty, url:{}, status:{}, response:{}", url, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return Lists.newArrayList();
            }
        } catch (Exception ex) {
            log.error("query ad list from es error, url:{}, errorMessage:{}", url, ex.getMessage());
            throw BusinessException.create(ex.getMessage());
        }
    }

    /**
     * 阅读广告记录
     *
     * @param adId                广告id
     * @param customerServiceCode 客服代号
     * @param esUserId            esUserId
     * @param esAccessToken       esAccessToken
     * @return
     */
    @Override
    public boolean readServiceCloud(Long adId, String customerServiceCode, String esUserId, String esAccessToken) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("Adid", adId);
        requestMap.put("CustomerServiceCode", customerServiceCode);
        requestMap.put("UserId", esUserId);
        HttpEntity<Object> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getEsUri()).append(ES_READ_ADS_PATH).append(esAccessToken);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return respEntity.getBody().getBoolean("data");
            } else {
                log.info("query advertisement list from es return empty, url:{}, status:{}, response:{}", url, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return false;
            }
        } catch (Exception ex) {
            log.error("query ad list from es error, url:{}, errorMessage:{}", url, ex.getMessage());
            throw BusinessException.create(ex.getMessage());
        }
    }
}