package com.digiwin.athena.semc.proxy.esp.service.impl;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.esp.sdk.Invoker;
import com.digiwin.athena.esp.sdk.model.RequestModel;
import com.digiwin.athena.esp.sdk.model.ResponseModel;
import com.digiwin.athena.esp.sdk.util.JsonUtil;
import com.digiwin.athena.semc.common.BizException;
import com.digiwin.athena.semc.common.ServiceException;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.util.DataTypeAdapter;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * @Author: sunyfa
 * @CreateTime: 2022/12/9 13:13
 * @Description: 对接esp服务
 * @Version: 1.0
 */
@Service
public class ESPServiceImpl implements ESPService {
    private static final String BODY_STD_DATA = "std_data";
    private static final String EXECUTION_FAIL = "exception.execution.api.fail";
    private static Logger logger = LoggerFactory.getLogger(ESPServiceImpl.class);
    @Resource
    private MessageUtils messageUtils;

    /**
     * 发起 ESP 查询
     *
     * @param eaiSysName 地中台产品名称
     * @param serviceName 服务端产品提供的服务名称
     * @param eaiSysUid 地中台UID，确定是哪个实例
     * @param extHeader 额外的header信息
     * @param parameter 请求参数
     * @param queryCondition 查询条件
     * @param pageInfo 分页
     * @return 查询结果
     */
    public Map<String, Object> queryByEsp(String eaiSysName, String eaiSysUid, String serviceName,
        Map<String, String> extHeader, Map<String, Object> parameter, String queryCondition, PageInfo pageInfo) {
        // 创建 Header
        Map<String, String> headers = createHeader(extHeader);

        // 创建查询条件
        Map<String, Object> bodyMap = new HashMap<>();
        Map<String, Object> stdData = new HashMap<>();
        if (parameter == null) {
            parameter = new HashMap<>();
        }

        // 包装分页信息
        if (pageInfo != null) {
            if (pageInfo.getPageSize() != null) {
                parameter.put("use_has_next", true);
                parameter.put("page_size", pageInfo.getPageSize());
            }
            if (pageInfo.getPageNo() != null) {
                parameter.put("page_no", pageInfo.getPageNo());
            }
        }

        // 包装查询条件
        if (!org.apache.commons.lang3.StringUtils.isEmpty(queryCondition)) {
            parameter.put("query_condition", queryCondition);
        }

        stdData.put("parameter", parameter);
        bodyMap.put(BODY_STD_DATA, stdData);

        // 创建调用ESP的请求对象
        RequestModel requestModel = createRequestModel(eaiSysName, eaiSysUid, serviceName, headers,
            JsonUtil.toString(bodyMap));

        return invokeEsp(requestModel, true);
    }

    /**
     * 创建请求头
     *
     * @return Header 信息
     */
    private Map<String, String> createHeader(Map<String, String> extHeader) {
        Map<String, String> headers = new HashMap<>();
        if (!CollectionUtils.isEmpty(extHeader)) {
            headers.putAll(extHeader);
        }

        return headers;
    }

    /**
     * 创建对接 ESP 的 RequestModel
     *
     * @param eaiSysName 地中台产品名称
     * @param eaiSysUid 地中台Uid
     * @param serviceName 服务端提供的服务名称
     * @param headers header信息
     * @param body 请求报文
     * @return RequestModel 对象
     */
    private RequestModel createRequestModel(String eaiSysName, String eaiSysUid, String serviceName,
        Map<String, String> headers, String body) {
        RequestModel requestModel = new RequestModel();
        // 请求端产品名称（注册在中台的信息）
        requestModel.setHostProd("Athena");
        // 请求端产品版本（注册在中台的信息）
        requestModel.setHostVer("1.0.0");
        requestModel.setHostAcct("athena");
        // 请求端识别码 ID (注册在中台的信息)
        requestModel.setHostId("SEMC");
        // 执行服务的账户信息（对应服务产品的账号）
        // 语系
        requestModel.setLanguage(LocaleContextHolder.getLocale().toString());
        // 租户ID
        requestModel.setTenantId(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        // 服务端产品（若能明确知道应用/产品,请传入；若无法得知可不设，SDK会向KM询问）
        // 暂时取组件配置的应用名称，在地中台配置服务端服务时，产品名称需要和组件配置的应用名称一致
        requestModel.setServiceProd(eaiSysName);
        // 调用服务名称
        requestModel.setServiceName(serviceName);
        // 服务端产品识别码 UID
        requestModel.setServiceProdUid(eaiSysUid);
        // 应用内容JSON字符串
        requestModel.setBodyJsonString(body);
        // 封装头部信息
        requestModel.setHeaderMap(headers);

        return requestModel;
    }

    /**
     * 发起ESP调用
     *
     * @param requestModel 请求参数
     * @param sync 是否同步调用
     * @return 调用结果
     */
    private Map<String, Object> invokeEsp(RequestModel requestModel, boolean sync) {
        String serviceId = requestModel.getServiceProd().concat(".").concat(requestModel.getServiceName());

        ResponseModel responseModel;
        try {
            logger.info("Call ESP param:{}", new Gson().toJson(requestModel));

            if (sync) {
                // 同步
                responseModel = Invoker.invokeRestSync(requestModel);
            }
            else {
                // 全异步
                responseModel = Invoker.invokeRestFasync(requestModel);
                // TODO: 需要调回调接口
            }

            logger.info("Call ESP result:{}", new Gson().toJson(responseModel));
        }
        catch (Exception e) {
            logger.error("Call ESP Error", e);
            String error = String.format(messageUtils.getMessage(EXECUTION_FAIL), LocalDateTime.now(), serviceId, e.getMessage());
            throw new ServiceException(500, error);
        }

        Map<String, Object> resultMap = throwEspError(serviceId, responseModel);
        Map<String, Object> stdDataMap = (Map<String, Object>) resultMap.getOrDefault(BODY_STD_DATA, null);
        if (stdDataMap != null) {
            return (Map<String, Object>) stdDataMap.getOrDefault("parameter", null);
        }
        return Maps.newHashMap();
    }

    private Map<String, Object> throwEspError(String serviceId, ResponseModel responseModel) {
        if (Objects.isNull(responseModel) || Objects.isNull(responseModel.getEspCode())) {
            String error = String.format(messageUtils.getMessage("exception.execution.api.fail6"), serviceId);
            throw BizException.getDefaultBizException(599, error);
        }

        // 稳敏API发生调用异常 => digi-code 非0开头
        else if (!responseModel.getEspCode().startsWith("0")) {
            String error = String.format(messageUtils.getMessage(EXECUTION_FAIL), LocalDateTime.now(), serviceId,
                responseModel.getEspMessage());
            throw BizException.getDefaultBizException(599, error);
        }

        if (!StringUtils.hasText(responseModel.getBodyJsonString())) {
            String error = String.format(messageUtils.getMessage("exception.execution.api.fail2"), serviceId);
            throw BizException.getDefaultBizException(599, error);
        }
        Gson gson = new GsonBuilder().registerTypeAdapter(new TypeToken<Map<String, Object>>() {
        }.getType(), new DataTypeAdapter()).create();
        Map<String, Object> espResponse = gson.fromJson(responseModel.getBodyJsonString(),
            (new TypeToken<Map<String, Object>>() {
            }).getType());

        if (espResponse.containsKey("error")) {
            String error = String.format(messageUtils.getMessage(EXECUTION_FAIL), LocalDateTime.now(), serviceId, espResponse.get("error"));
            throw BizException.getDefaultBizException(599, error);
        }

        Map<String, Object> stdDataMap = (Map<String, Object>) espResponse.getOrDefault(BODY_STD_DATA, null);
        if (stdDataMap == null) {
            String error = String.format(messageUtils.getMessage("exception.execution.api.fail3"), serviceId,
                responseModel.getBodyJsonString());
            throw BizException.getDefaultBizException(599, error);
        }

        Map<String, Object> execution = (Map<String, Object>) stdDataMap.getOrDefault("execution", null);
        if (execution == null) {
            String error = String.format(messageUtils.getMessage("exception.execution.api.fail4"), serviceId,
                responseModel.getBodyJsonString());
            throw BizException.getDefaultBizException(599, error);
        }

        // 业务异常
        if (!"000".equals(responseModel.getSrvCode())) {
            String description = (String) execution.getOrDefault("description", "");
            String code = (String) execution.getOrDefault("code", "");
            throw BizException.getDefaultBizException(Integer.parseInt(code), description);
        }

        return espResponse;
    }
}