package com.digiwin.athena.semc.proxy.fineReport.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.fineReport.model.resp.FineReportTreeVO;
import com.digiwin.athena.semc.proxy.fineReport.model.resp.FineReportUserVO;
import com.digiwin.athena.semc.proxy.fineReport.service.FineReportProxyService;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * 帆软报表系统服务实现
 *
 * @author sungqz
 * @since 2025-08-11
 */
@Service
@Slf4j
public class FineReportProxyServiceImpl implements FineReportProxyService {

    @Resource
    private EnvProperties envProperties;

    @Resource
    private RestTemplate restTemplate;

    /**
     * 查询帆软平台目录树接口path
     */
    private static final String QUERY_FINE_REPORT_ENTRY_TREE = "/webroot/decision/sp/client/api/v3/entry/tree?device=";

    /**
     * 批量添加用户接口path
     */
    private static final String ADD_FINE_REPORT_USER = "/webroot/decision/sp/client/api/v3/user";

    /**
     * 错误码映射
     */
    private static final Map<String, String> FINE_REPORT_ERROR_CODE_MAP = ImmutableMap.of("21300015", "邮箱格式错误");

    /**
     * 查询帆软平台目录树接口
     *
     * @param deviceType 设备类型
     * @return 返回
     */
    @Override
    public List<FineReportTreeVO> queryFineReportEntryTree(String clientId, String secret, String deviceType) {
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.APPLICATION_JSON);
        header.add("clientId", clientId);
        header.add("secret", secret);

        HttpEntity<Map<String, Object>> requestMap = new HttpEntity<>(null, header);
        StringBuilder url = new StringBuilder(envProperties.getFineReportUrl()).append(QUERY_FINE_REPORT_ENTRY_TREE).append(deviceType);
        try {
            ResponseEntity<JSONObject> response = restTemplate.exchange(url.toString(), HttpMethod.GET, requestMap, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == response.getStatusCodeValue() && response.getBody() != null) {
                JSONArray data = response.getBody().getJSONArray("data");
                return JSONArray.parseArray(data.toJSONString(), FineReportTreeVO.class);
            } else {
                log.info("query fineReport entry tree return empty, url:{}, request:{}, response:{}", url, requestMap, JSONObject.toJSONString(response));
                return Lists.newArrayList();
            }
        } catch (Exception ex) {
            log.error("query fineReport entry tree occur error, url:{}, request:{}, errorMsg:{}", url, requestMap, ex.getMessage(), ex);
            return Lists.newArrayList();
        }
    }

    /**
     * 添加帆软用户信息
     *
     * @param fineReportUserVO 帆软用户
     */
    @Override
    public void addFineReportUser(String clientId, String secret, FineReportUserVO fineReportUserVO) {
        JSONObject result = operateFineReportUser(clientId, secret, ADD_FINE_REPORT_USER, Collections.singletonList(fineReportUserVO), HttpMethod.POST);
        checkFineReportUserResponse(result);
    }

    /**
     * 修改帆软用户信息
     *
     * @param fineReportUserVO 帆软用户
     */
    @Override
    public void updateFineReportUser(String clientId, String secret, FineReportUserVO fineReportUserVO) {
        JSONObject result = operateFineReportUser(clientId, secret, ADD_FINE_REPORT_USER, Collections.singletonList(fineReportUserVO), HttpMethod.PUT);
        checkFineReportUserResponse(result);
    }

    /**
     * 校验帆软用户返回结果
     *
     * @param response 帆软用户返回结果
     */
    private void checkFineReportUserResponse(JSONObject response) {
        Object data = JSONPath.eval(response, "$.data[0]");
        if (data != null) {
            JSONObject parseResult = JSON.parseObject(JSON.toJSONString(data));
            if (!parseResult.getBooleanValue("status")) {
                String errorCode = parseResult.getString("errorCode");
                String errorMessage = FINE_REPORT_ERROR_CODE_MAP.get(errorCode);
                throw new RuntimeException("调用帆软服务异常：" + (errorMessage != null ? errorMessage : parseResult.getString("errorMsg")));
            }
        }
    }

    @Override
    public void deleteFineReportUser(String clientId, String secret, String frUserId) {
        JSONObject result = operateFineReportUser(clientId, secret, ADD_FINE_REPORT_USER + "?user=[\"" + frUserId + "\"]", null, HttpMethod.DELETE);
        if (result != null && result.getInteger("status") != 200) {
            throw new RuntimeException("调用帆软服务异常：" + result.getString("errorMsg"));
        }
    }

    @Override
    public List<FineReportUserVO> queryFineReportUser(String clientId, String secret, String frUserId) {
        JSONObject response = operateFineReportUser(clientId, secret, ADD_FINE_REPORT_USER + "?page=1&count=9999&username=" + frUserId, null, HttpMethod.GET);
        if (response != null && response.get("data") != null) {
            JSONObject data = response.getJSONObject("data");
            if (data.get("items") != null) {
                return JSON.parseArray(data.getJSONArray("items").toJSONString(), FineReportUserVO.class);
            }
        }
        return Collections.emptyList();
    }

    /**
     * 保存帆软用户信息
     *
     * @param data 请求入参
     */
    private JSONObject operateFineReportUser(String clientId, String secret, String url, Object data, HttpMethod method) {
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.APPLICATION_JSON);
        header.add("clientId", clientId);
        header.add("secret", secret);

        HttpEntity<Object> requestMap = new HttpEntity<>(data, header);
        ResponseEntity<JSONObject> response = restTemplate.exchange(envProperties.getFineReportUrl() + url, method, requestMap, JSONObject.class);
        log.info("save fineReport user, url:{}, request:{}, response:{}", envProperties.getFineReportUrl() + url, JSONObject.toJSONString(data), JSONObject.toJSONString(response));
        if (org.apache.http.HttpStatus.SC_OK != response.getStatusCodeValue()) {
            throw new RuntimeException("调用帆软服务异常");
        }
        return response.getBody();
    }

}