package com.digiwin.athena.semc.proxy.iam.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.ListUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.exception.OperateException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.erpsso.AuthorizeDTO;
import com.digiwin.athena.semc.dto.erpsso.RegisterDTO;
import com.digiwin.athena.semc.entity.applink.TenantAppDTO;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.eoc.service.model.EocResultDTO;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.*;
import com.digiwin.athena.semc.proxy.iam.service.model.req.UserAllPermissionReqDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.resp.AppTenantRespVO;
import com.digiwin.athena.semc.proxy.iam.service.model.resp.TenantUserBaseRespVO;
import com.digiwin.athena.semc.proxy.iam.service.model.resp.TenantUserRespVO;
import com.digiwin.athena.semc.proxy.iam.service.model.resp.UserAllPermissionBaseRespVO;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.common.PermissionUserFunctionalDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationModulesSimpleDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationSimpleDTO;
import com.digiwin.athena.semc.vo.portal.AppExistVO;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.*;

import static com.digiwin.athena.semc.util.Utils.encodeSignArg;

/**
 * IamServiceImpl Description
 *
 * @author sungq
 * @date 2022/11/29
 * @since
 */
@Service
@Slf4j
public class IamServiceImpl implements IamService {

    private static final String APPLICATION_REGISTER_V2 = "/api/iam/v2/oauth2/app/register";
    private static final String APPLICATION_QUERY_INFO_V2 = "/api/iam/v2/oauth2/app";
    private static final String TOKEN_ANALYZE_URL = "/api/iam/v2/identity/token/analyze";
    private static final String TENANT_APPLICATION_LIST_URL = "/api/iam/v2/tenant/application/list";
    private static final String MAPPING_QUERY_EMP_ID = "/api/iam/v2/mapping/query";
    private static final String USER_PERMISSION_V2 = "/api/iam/v2/permission/user";
    private static final String DEV_APP_LIST_V2 = "/api/iam/v2/dev/app";
    //用户授权的商品信息
    private static final String USER_APPLICATION = "/api/iam/v2/user/application";
    //获取用户功能权限
    private static final String USER_FUNCTIONAL = "/api/iam/v2/permission/user/functional";
    //公司别厂别
    private static final String COMPANY_FACORY = "/api/iam/v2/org/cascade";

    // 当前登录用户的详细信息
    private static final String USER_LOGIN_INFO = "/api/iam/v2/user/current?withEmp=" + true;

    private static final String IAM_AUTHORIZE_V2 = "/api/iam/v2/oauth2/authorize";
    private static final String SAML_CALLBACK_URL_EXIT = "/api/iam/v2/saml/sys/exist";
    private static final String SAML_CALLBACK_URL_ADD = "/api/iam/v2/saml/sys/add";
    private static final String AUTHCODE_SAML_TOKEN = "/api/iam/v2/identity/token/saml/access";

    /**
     * 获取租户语言别设置
     */
    public static final String USERMETADATA_USER_GET = "/api/iam/v2/usermetadata/user/get";

    private static final String CAS_CALLBACK_URL_ADD = "/api/iam/v2/sys/sso/url/cfg/add";

    /**
     * 根据用户id获取用户相关的信息
     */
    public static final String USER = "/api/iam/v2/user/full";

    /**
     * 批量根据应用id查询授权的租户基本信息
     */
    public static final String QUERY_TENANT_BASIC_PATH = "/api/iam/v2/tenant/basic/by/apps";

    /**
     * 查询当前租户下所有角色列表
     */
    public static final String QUERY_TENANT_ROLE_LIST = "/api/iam/v2/role/simple";

    /**
     * 根据邮箱获取个人租户
     * status的值
     * -1 调用接口失败
     * 0 没有个人租户
     * 1 没有授权
     * 2 有个人租户并且有授权
     */
    public static final String USER_EMAIL_TENANTS_AUTH = "/api/iam/v2/user/email/tenants/auth";

    /**
     * 获取当前租户版本
     */
    public static final String TENANT_CURRENT_SIMPLE = "/api/iam/v2/tenant/current/simple";

    /**
     * 获取租户的全部组织信息
     */
    public static final String ORG_ASPECT = "/api/iam/v2/org/aspect";

    /**
     * 获取集成租户token
     */
    public static final String INTEGRATION_TOKEN = "/api/iam/v2/identity/login/internal";

    /**
     * 查询当前租户下第三方账号映射的用户信息
     */
    public static final String QUERY_USER_BY_VERIFY_USER_ID = "/api/iam/v2/mapping/query/appuser";

    /**
     * 查询当前租户下指定应用系统下的所有归户用户的信息
     */
    public static final String QUERY_All_VERIFY_MAPPING_USER = "/api/iam/v2/mapping/query/app";

    @Resource
    private EnvProperties envProperties;
    @Resource
    private RestTemplate restTemplate;
    @Resource
    private MessageUtils messageUtils;

    /**
     * 注册应用
     *
     * @param registerDTO 请求体
     * @return
     */
    @Override
    public Long registerApp(RegisterDTO registerDTO) throws Exception {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("name", registerDTO.getName());
        requestMap.put("description", registerDTO.getDescription());
        requestMap.put("callbackUrl", StringUtils.isBlank(registerDTO.getCallbackUrl()) ? String.format(Constants.CS_APP_CALLBACK_URL, registerDTO.getName()) : registerDTO.getCallbackUrl());
        requestMap.put("logoImage", "");
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(APPLICATION_REGISTER_V2);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST,
                    requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return respEntity.getBody().getLong("data");
            } else {
                log.error("从IAM注册应用失败, 用户token:{}, appToken:{}, request:{}, url:{}, statusCode:{}, response：{}", userToken,
                        envProperties.getAppToken(), requestMap, url, respEntity.getStatusCodeValue(),
                        JSONObject.toJSONString(respEntity.getBody()));
                throw new Exception(Optional.ofNullable(respEntity.getBody()).orElse(new JSONObject()).getString("errorMessage"));
            }
        } catch (Exception ex) {
            log.error("从IAM注册应用发生异常, 用户token:{}, appToken:{}, request:{}, url:{}, errorMsg:{}", userToken,
                    envProperties.getAppToken(), requestMap, url, ex.getMessage());
            throw ex;
        }
    }

    /**
     * 更新应用该信息，目的是将appId和appSecret重新绑定应用
     *
     * @param registerDTO 请求体
     * @return
     */
    @Override
    public Long updateAppInfoForIAM(RegisterDTO registerDTO) throws Exception {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("name", registerDTO.getName());
        requestMap.put("description", registerDTO.getDescription());
        requestMap.put("callbackUrl", StringUtils.isBlank(registerDTO.getCallbackUrl()) ? String.format(Constants.CS_APP_CALLBACK_URL, registerDTO.getName()) : registerDTO.getCallbackUrl());
        requestMap.put("logoImage", "");
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(APPLICATION_QUERY_INFO_V2).append("/").append(registerDTO.getSid());
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST,
                    requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return respEntity.getBody().getLong("data");
            } else {
                log.error("调用IAM更新应用失败, 用户token:{}, appToken:{}, request:{}, url:{}, statusCode：{}, response:{}", userToken,
                        envProperties.getAppToken(), requestMap, url, respEntity.getStatusCodeValue(),
                        JSONObject.toJSONString(respEntity.getBody()));
                throw new Exception(Optional.ofNullable(respEntity.getBody()).orElse(new JSONObject()).getString("errorMessage"));
            }
        } catch (Exception ex) {
            log.error("调用IAM更新应用异常, 用户token:{}, appToken:{}, request:{}, url:{}, errorMsg:{}", userToken,
                    envProperties.getAppToken(), requestMap, url, ex.getMessage());
            throw ex;
        }
    }

    /**
     * 查询应用appId和appSecret
     *
     * @param appSid   应用sid
     * @param appToken appToken
     * @return
     */
    @Override
    public RegisterDTO queryAppInfo(Long appSid, String appToken) throws Exception {
        if (appSid == null) {
            return null;
        }
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(APPLICATION_QUERY_INFO_V2).append("/").append(appSid);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.GET,
                    requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONObject jsonObject = respEntity.getBody().getJSONObject("data");
                return JSONObject.parseObject(JSONObject.toJSONString(jsonObject), RegisterDTO.class);
            } else {
                log.error("从IAM查询应用信息失败, appSid：{}, token:{}, appToken:{}, url：{}, statusCode:{}, response：{}", appSid,
                        userToken, envProperties.getAppToken(), url, respEntity.getStatusCodeValue(),
                        JSONObject.toJSONString(respEntity.getBody()));
                throw new Exception(Optional.ofNullable(respEntity.getBody()).orElse(new JSONObject()).getString("errorMessage"));
            }
        } catch (Exception ex) {
            log.error("从IAM查询应用信息发生异常, appSid:{}, token: {}, appToken:{}, statusCode：{}, errorMsg:{}", appSid, userToken,
                    envProperties.getAppToken(), url, ex.getMessage());
            throw ex;
        }
    }

    /**
     * 校验 user token 合法性
     *
     * @param userToken
     * @param appToken
     * @return
     */
    @Override
    public boolean checkUserToken(String userToken, String appToken) {
        boolean isUserTokenValid = false;

        // 请求头
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", appToken);

        // 请求参数
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);

        // 请求地址
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(TOKEN_ANALYZE_URL);

        try {
            log.info("Call IAM checks user token url:{}, param:{}", url, JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST,
                    requestEntity, JSONObject.class);
            log.info("Call IAM checks user token result:{}", JSONObject.toJSONString(respEntity));

            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONObject jsonObject = respEntity.getBody();
                isUserTokenValid = Objects.nonNull(jsonObject.get("id"));
            }
        } catch (Exception ex) {
            log.error("Call IAM checks user token error", ex);

            isUserTokenValid = false;
        }

        return isUserTokenValid;
    }

    /**
     * 获取当前租户下的应用清单
     *
     * @return
     */
    @Override
    public List<TenantAppDTO> queryTenantApps() {
        List<TenantAppDTO> appDTOList = new ArrayList<>();
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        setAcceptLanguage(headers);

        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("showEntry", true);

        HttpEntity<Map> reqEntity = new HttpEntity<>(requestMap, headers);
        String url = envProperties.getIamUri() + TENANT_APPLICATION_LIST_URL;
        //本地mockiam
        //String url = "http://127.0.0.1:4523/m1/1492309-0-default"+ TENANT_APPLICATION_LIST_URL;
        try {
            ResponseEntity<List<TenantAppDTO>> respEntity = restTemplate.exchange(url, HttpMethod.POST, reqEntity,
                    new ParameterizedTypeReference<List<TenantAppDTO>>() {
                    });
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue()) {
                return respEntity.getBody();
            } else {
                log.error("获取租户下的应用清单失败，用户token: {}, 状态码：{}，响应内容：{}", AppAuthContextHolder.getContext().getAuthoredUser().getToken(), respEntity.getStatusCodeValue(),
                        JSONObject.toJSONString(respEntity.getBody()));
                return appDTOList;
            }
        } catch (Exception ex) {
            log.error("获取租户下的应用清单失败，用户token: {}, 错误内容：{}", AppAuthContextHolder.getContext().getAuthoredUser().getToken(), ex.getMessage());
            return appDTOList;
        }
    }

    /**
     * iam以Accept-Language为先
     *
     * @param headers
     */
    private void setAcceptLanguage(HttpHeaders headers) {
        String locale = LocaleContextHolder.getLocale().toString();
        if (StringUtils.isEmpty(locale)) {
            return;
        }
        locale = locale.replaceAll("_", "-");
        headers.add("Accept-Language", locale);
    }

    /**
     * 获取当前租户下登录的IAM云端账号绑定的三方员工账号
     *
     * @param appId appId
     * @return IAM云端账号绑定的三方员工账号
     */
    @Override
    public String queryMappingEmpId(String appId) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("tenantSid", 0);
        requestMap.put("userSid", 0);
        requestMap.put("identityId", appId);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(MAPPING_QUERY_EMP_ID);
        try {
            log.info("param:" + JSONObject.toJSONString(requestEntity));
            ResponseEntity<List<JSONObject>> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<List<JSONObject>>() {
                    });
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && CollectionUtils.isNotEmpty(respEntity.getBody())) {
                return respEntity.getBody().get(0).getString("verifyUserId");
            } else {
                log.info("query mappingEmpId return empty, token:{}, request:{}, url:{}, statusCode:{}, response:{}", userToken, requestMap, url, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("query mappingEmpId occur error, token:{}, request:{}, url:{}, errorMsg:{}", userToken, requestMap, url, ex.getMessage());
            return null;
        }
    }

    /**
     * 鼎捷云获取用户行为权限配置
     *
     * @param authoredUser 用户信息
     * @return
     */
    @Override
    public JSONArray queryUserPermissionV2(AuthoredUser authoredUser) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", authoredUser.getToken());

        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("userId", authoredUser.getUserId());
        requestMap.put("sysId", envProperties.getSemcAppId());
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);

        String url = envProperties.getIamUri() + USER_PERMISSION_V2;
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return respEntity.getBody().getJSONArray("data");
            } else {
                log.info("query user permission return empty, token:{}, url:{}, request:{}, statusCode:{}, response:{}", authoredUser.getToken(),
                        url, requestMap, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("query user permission occur error, token:{}, url:{}, request：{}, errorMsg：{}", authoredUser.getToken(),
                    url, requestMap, ex.getMessage());
            throw BusinessException.create(ex.getMessage());
        }
    }

    /**
     * @param
     * @return
     * @description: 查询当前用户所在的组织
     * @author: sunyfa
     */
    @Override
    public UserOrgInfoDTO qryUserOrgInfo(String userId, String token) {
        String url = envProperties.getIamUri() + "/api/iam/v2/user/org";

        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.APPLICATION_JSON);
        httpHeaders.add(GlobalConstant.AUTH_USER, token);

        Map<String, Object> reqBody = new HashMap<>();
        reqBody.put("id", userId);

        HttpEntity httpEntity = new HttpEntity(reqBody, new HttpHeaders());

        ResponseEntity<UserOrgInfoDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, UserOrgInfoDTO.class);
        if (org.apache.http.HttpStatus.SC_OK == responseEntity.getStatusCodeValue()) {
            return responseEntity.getBody();
        } else {
            throw new OperateException("Call iam get user organization info failed, status code: "
                    + responseEntity.getStatusCodeValue() + ", response: " + responseEntity.getBody());
        }
    }

    @Override
    public List<Long> qryUserOrgSidList(String userId, String token) {
        String url = envProperties.getIamUri() + "/api/iam/v2/user/org";

        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.APPLICATION_JSON);
        httpHeaders.add(GlobalConstant.AUTH_USER, token);

        Map<String, Object> reqBody = new HashMap<>();
        reqBody.put("id", userId);

        HttpEntity httpEntity = new HttpEntity(reqBody, new HttpHeaders());
        ResponseEntity<JSONObject> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, JSONObject.class);
        if (org.apache.http.HttpStatus.SC_OK == responseEntity.getStatusCodeValue()) {
            List<Long> userOrgSidList = Lists.newArrayList();
            JSONObject body = responseEntity.getBody();
            if (MapUtils.isEmpty(body)) {
                return userOrgSidList;
            }
            JSONArray userInOrgs = body.getJSONArray("userInOrgs");
            if (CollectionUtils.isEmpty(userInOrgs)) {
                return userOrgSidList;
            }
            for (int i = 0; i < userInOrgs.size(); i++) {
                userOrgSidList.add(userInOrgs.getJSONObject(i).getLong("orgSid"));
            }
            return userOrgSidList;


        } else {
            log.warn("Call iam get user organization info failed：{},{}", responseEntity.getStatusCodeValue(), responseEntity.getBody().toString());
            return Lists.newArrayList();
        }
    }

    /**
     * @param
     * @return
     * @description: 查询当前用户所在的角色
     * @author: sunyfa
     */
    @Override
    public List<RoleDTO> queryUserRoles(String userId, String token) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        Map<String, String> param = new HashMap<>();
        param.put("id", userId);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        String url = envProperties.getIamUri() + "/api/iam/v2/user/role";
        ResponseEntity<UserRoleDTO> roleResponseEntity = restTemplate.postForEntity(url, httpEntity, UserRoleDTO.class);
        if (roleResponseEntity.getBody() == null) {
            return new ArrayList<>();
        } else {
            return roleResponseEntity.getBody().getRoles();
        }
    }

    @Override
    public List<Long> queryUserRolesSidList(String userId, String token) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        Map<String, String> param = new HashMap<>();
        param.put("id", userId);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        String url = envProperties.getIamUri() + "/api/iam/v2/user/role";
        ResponseEntity<JSONObject> roleResponseEntity = restTemplate.postForEntity(url, httpEntity, JSONObject.class);
        JSONObject body = roleResponseEntity.getBody();
        List<Long> roleSidList = Lists.newArrayList();
        if (MapUtils.isEmpty(body)) {
            return roleSidList;
        }
        JSONArray roles = roleResponseEntity.getBody().getJSONArray("roles");
        if (CollectionUtils.isEmpty(roles)) {
            return roleSidList;
        }
        for (int i = 0; i < roles.size(); i++) {
            roleSidList.add(roles.getJSONObject(i).getLong("sid"));
        }
        return roleSidList;

    }

    /**
     * 查询当前租户下所有用户列表
     *
     * @return 返回对象
     */
    @Override
    public List<RoleDTO> queryAllUserList() {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        Map<String, String> param = new HashMap<>();
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + "/api/iam/v2/user/simple";
        log.info("start query all user, url:{}, request:{}", url, httpEntity);
        ResponseEntity<List<RoleDTO>> respEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity,
                new ParameterizedTypeReference<List<RoleDTO>>() {
                });
        return respEntity.getBody();
    }

    /**
     * 查询当前租户下所有用户列表
     *
     * @return 返回对象
     */
    @Override
    public List<RoleDTO> queryEnabledUserList() {
        HttpHeaders headers = new HttpHeaders();

        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add(GlobalConstant.IAM_APP_TOKEN, envProperties.getAppToken());

        Map<String, String> param = new HashMap<>();
        param.put("tenantId", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        String url = envProperties.getIamUri() + "/api/iam/v2/tenant/user/list?pageNum=1&pageSize=99999";
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONArray data = respEntity.getBody().getJSONArray("list");
                return JSONObject.parseArray(data.toJSONString(), RoleDTO.class);
            } else {
                log.info("query enabled user list return empty. url:{}, request:{}, response:{}", url, httpEntity, respEntity);
                return Lists.newArrayList();
            }
        } catch (Exception ex) {
            log.error("query enabled user list occur error. url:{}, request:{}, errorMsg:{}", url, httpEntity, ex);
            return Lists.newArrayList();
        }
    }

    /**
     * 批量查询组织下用户列表
     *
     * @param orgSidList 组织sid列表
     * @return 返回对象
     */
    @Override
    public List<OrgRoleUserDTO> queryUserByOrgSidList(List<Long> orgSidList) {
        if (CollectionUtils.isEmpty(orgSidList)) {
            return Lists.newArrayList();
        }
        // url
        String url = envProperties.getIamUri() + "/api/iam/v2/org/userinorg/list/simple";

        // header
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // body
        Map<String, List<Long>> param = new HashMap<>();
        param.put("orgSids", orgSidList);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<JSONObject> respEntity;
        try {
            respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, JSONObject.class);
        } catch (Exception ex) {
            log.error("query user by org sid list error, url:{}, token:{}, request:{}, errorMsg:{}", url, userToken, httpEntity, ex);
            throw BusinessException.create(ex.getMessage());
        }
        // 处理返回对象
        if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
            JSONArray data = respEntity.getBody().getJSONArray("data");
            return JSONObject.parseArray(data.toJSONString(), OrgRoleUserDTO.class);
        } else {
            log.warn("query user by org sid list return empty, url:{}, token:{}, request:{}, statusCode:{}, response:{}", url, userToken,
                    httpEntity, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
            return Lists.newArrayList();
        }
    }

    @Override
    public List<OrgRoleUserDTO> queryUserByAllOrgSidList(List<Long> orgSidList) {
        List<List<Long>> list = ListUtil.split(orgSidList, 100);
        List<OrgRoleUserDTO> resultList = Lists.newArrayList();
        for (List<Long> tempList : list) {
            resultList.addAll(queryUserByOrgSidList(tempList));
        }
        return resultList;
    }

    /**
     * 批量查询角色下用户列表
     *
     * @param roleSidList 角色sid列表
     * @return 返回对象
     */
    @Override
    public List<OrgRoleUserDTO> queryUserByRoleSidList(List<Long> roleSidList) {
        if (CollectionUtils.isEmpty(roleSidList)) {
            return Lists.newArrayList();
        }
        // url
        String url = envProperties.getIamUri() + "/api/iam/v2/association/userinrole/list/simple";

        // header
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // body
        Map<String, List<Long>> param = new HashMap<>();
        param.put("roleSids", roleSidList);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<JSONObject> respEntity;
        try {
            respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, JSONObject.class);
        } catch (Exception ex) {
            log.error("query user by role sid list error, url:{}, token:{}, request:{}, errorMsg:{}", url, userToken, httpEntity, ex);
            throw BusinessException.create(ex.getMessage());
        }
        // 处理返回对象
        if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
            JSONArray data = respEntity.getBody().getJSONArray("data");
            return JSONObject.parseArray(data.toJSONString(), OrgRoleUserDTO.class);
        } else {
            log.warn("query user by role sid list return empty, url:{}, token:{}, request:{}, statusCode:{}, response:{}", url, userToken,
                    httpEntity, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
            return Lists.newArrayList();
        }
    }

    @Override
    public List<OrgRoleUserDTO> queryUserByAllRoleSidList(List<Long> roleSidList) {
        List<List<Long>> list = ListUtil.split(roleSidList, 100);
        List<OrgRoleUserDTO> resultList = Lists.newArrayList();
        for (List<Long> tempList : list) {
            resultList.addAll(queryUserByRoleSidList(tempList));
        }
        return resultList;
    }


    /**
     * 查询所有申请过appToken的应用列表
     *
     * @return 应用列表
     */
    @Override
    public List<AppExistVO.AppDTO> queryAppListExistAppToken() throws Exception {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        try {
            URIBuilder builder = new URIBuilder(Constants.isExist);
            builder.addParameter("params", "{\"nonDev\":\"true\"}");
            // 创建get请求
            HttpGet get = new HttpGet(builder.build());
            get.setHeader("digi-middleware-auth-app", envProperties.getAppToken());
            String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
            get.setHeader("digi-middleware-auth-user", this.getFormalZoneUserToken());

            log.info("srart query appList exist appToken, url:{}, headers:{}", builder, get);
            CloseableHttpResponse execute = httpClient.execute(get);
            log.info("srart query appList resp:{}", JSONObject.toJSONString(execute.getEntity()));
            //获取响应的内容
            if (org.apache.http.HttpStatus.SC_OK == execute.getStatusLine().getStatusCode()) {
                org.apache.http.HttpEntity entity = execute.getEntity();
                String content = EntityUtils.toString(entity, "utf-8");
                Gson gson = new Gson();
                AppExistVO appExistDeveloperVO = gson.fromJson(content, AppExistVO.class);
                if (!ObjectUtils.isEmpty(appExistDeveloperVO)) {
                    return appExistDeveloperVO.getData();
                }
            } else {
                log.error("query appList exist appToken fail, url:{}, headers:{}, status:{}", builder, get, execute.getStatusLine().getStatusCode());
                throw BusinessException.create("query appList exist appToken fail");
            }
        } catch (Exception e) {
            log.error("query appList exist appToken occur error.", e);
            throw BusinessException.create(e.getMessage());
        } finally {
            httpClient.close();
        }
        return null;

         /*HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);

        Map<String,Object> bodyMap = new HashMap<>();
        bodyMap.put("nonDev",true);

        String encode = "{\"nonDev\":\"true\"}";
       *//* try {
            encode = URLEncoder.encode("{\"nonDev\":\"true\"}", "utf-8");
        } catch (UnsupportedEncodingException e) {
        }*//*
        String url = envProperties.getIamUri() + DEV_APP_LIST_V2 + encode;
        try {

            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return respEntity.getBody().getJSONArray("data");
            } else {
                log.info("query appToken app list return empty, url:{}, headers:{}, statusCode:{}, response:{}", url, headers,
                         respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("query appToken app list occur error, url:{}, headers:{}, errorMsg：{}", url, headers, ex.getMessage());
            throw BusinessException.create(ex.getMessage());
        }*/

    }

    /**
     * 获取iam正式区用户token
     *
     * @return
     */
    private String getFormalZoneUserToken() {
        String userToken = null;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        Map<String, Object> bodyMap = new HashMap<>();
        bodyMap.put("tenantId", Constants.ZONE_TENANT_ID);
        bodyMap.put("userId", Constants.ZONE_USER_ID);
        bodyMap.put(Constants.PASS_WORD_HASH, Constants.ZONE_USER_HASH);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(bodyMap, headers);
        ResponseEntity<Map> responseEntity = restTemplate.exchange(Constants.urlToken, HttpMethod.POST, httpEntity, Map.class);
        if (null != responseEntity.getBody()) {
            userToken = (String) responseEntity.getBody().get("token");
        }
        log.info("获取 userToken, userToken:{}", userToken);
        return userToken;
    }

    /**
     * 获取集成租户的token
     *
     * @param tenantId 租户id
     * @return 返回
     */
    public String queryIntegrationUserToken(String tenantId) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        Map<String, Object> bodyMap = new HashMap<>();
        bodyMap.put("tenantId", tenantId);
        bodyMap.put("userId", Constants.ZONE_USER_ID);
        bodyMap.put(Constants.IDENTITY_TYPE_KEY, Constants.IDENTITY_TYPE_VALUE);
        bodyMap.put(Constants.PASS_WORD_HASH, Constants.ZONE_USER_HASH);

        String url = envProperties.getIamUri() + INTEGRATION_TOKEN;
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(bodyMap, headers);
        try {
            ResponseEntity<Map> response = restTemplate.exchange(url, HttpMethod.POST, httpEntity, Map.class);
            if (null != response.getBody()) {
                return (String) response.getBody().get("token");
            } else {
                log.info("query integration user token return empty, url:{}, request:{}, response:{}", url, httpEntity, JSONObject.toJSONString(response.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("query integration user token error, url:{}, request:{}, errMsg:{}", url, httpEntity, ex.getMessage(), ex);
            return null;
        }
    }

    /**
     * 根据sid查询appToken
     *
     * @param sid sid
     * @return
     */
    @Override
    public Map<String, Object> queryAppTokenBySid(Long sid) {
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.APPLICATION_JSON);
        String userToken = this.getFormalZoneUserToken();
        header.add("digi-middleware-auth-app", envProperties.getAppToken());
        header.add("digi-middleware-auth-user", userToken);
        HttpEntity<Map<String, Object>> requestMap = new HttpEntity<>(null, header);
        String url = Constants.url + sid;
        try {
            ResponseEntity<Map> response = restTemplate.exchange(url, HttpMethod.GET, requestMap, Map.class);
            if (org.apache.http.HttpStatus.SC_OK == response.getStatusCodeValue() && response.getBody() != null) {
                Map<String, Object> map = (Map<String, Object>) response.getBody().get("data");
                return map;
            } else {
                log.info("query appToken by sid return empty, url:{}, userToken:{}, appToken:{}, request:{}, statusCode:{}, response:{}", url, userToken,
                        envProperties.getAppToken(), requestMap, response.getStatusCodeValue(), JSONObject.toJSONString(response.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("query appToken by sid occur error, url:{}, userToken:{}, appToken:{}, request:{}, errorMsg:{}", url, userToken,
                    envProperties.getAppToken(), requestMap, ex.getMessage());
            return null;
        }
    }

    @Override
    public AuthorizeDTO queryAuthorizeInfo(String appId, String callbackUrl) throws Exception {
        if (appId == null || StringUtils.isBlank(callbackUrl)) {
            return null;
        }
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("appId", appId);
        requestMap.put("callbackUrl", callbackUrl);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(IAM_AUTHORIZE_V2);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST,
                    requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONObject jsonObject = respEntity.getBody().getJSONObject("data");
                return JSONObject.parseObject(JSONObject.toJSONString(jsonObject), AuthorizeDTO.class);
            } else {
                log.error("从IAM获取授权信息失败, appId：{}, token:{}, callbackUrl:{}, url：{}, statusCode:{}, response：{}", appId,
                        userToken, callbackUrl, url, respEntity.getStatusCodeValue(),
                        JSONObject.toJSONString(respEntity.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("从IAM获取授权信息发生异常, appId:{}, token: {}, callbackUrl:{}, url：{}, errorMsg:{}", appId, userToken,
                    callbackUrl, url, ex.getMessage());
            String error = String.format(messageUtils.getMessage(I18NKey.IAM_AUTHORIZE_EXCEPTION), LocalDateTime.now(), IAM_AUTHORIZE_V2);
            throw BusinessException.create(error);
        }
    }


    /**
     * 查询用户授权的商品信息
     */
    @Override
    public List<UserApplicationDTO> queryUserApplication() {
        List<UserApplicationDTO> userApplicationList = new ArrayList<>();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(USER_APPLICATION);
        try {
            log.info("query user application param:{}", JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONArray> respEntity = restTemplate.exchange(url.toString(), HttpMethod.GET,
                    requestEntity, JSONArray.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                userApplicationList = JSONArray.parseArray(respEntity.getBody().toString(), UserApplicationDTO.class);
                return userApplicationList;
            }
        } catch (Exception ex) {
            log.error("query user application error. url:{}, request:{}, errorMsg:{}", url, JSONObject.toJSONString(requestEntity), ex.getMessage());
            return userApplicationList;
        }
        return userApplicationList;
    }

    @Override
    public List<UserApplicationSimpleDTO> queryUserApplicationSimple(boolean filterAppExpired, boolean filterModuleExpired, boolean validPermission) {
        List<UserApplicationSimpleDTO> userApplicationList = new ArrayList<>();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(USER_APPLICATION);
        try {
            log.info("query user application param:{}", JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONArray> respEntity = restTemplate.exchange(url.toString(), HttpMethod.GET,
                    requestEntity, JSONArray.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && CollectionUtils.isNotEmpty(respEntity.getBody())) {
                JSONArray body = respEntity.getBody();
                for (int i = 0; i < body.size(); i++) {
                    JSONObject appJSONObject = body.getJSONObject(i);
                    if (filterAppExpired && Boolean.TRUE.equals(appJSONObject.getBoolean("expired"))) {
                        continue;
                    }
                    UserApplicationSimpleDTO userApplicationSimpleDTO = new UserApplicationSimpleDTO();
                    userApplicationSimpleDTO.setId(appJSONObject.getString("id"));
                    if (!validPermission) {
                        userApplicationSimpleDTO.setName(appJSONObject.getString("name"));
                    }
                    JSONArray enabledModules = appJSONObject.getJSONArray("enabledModules");
                    List<UserApplicationModulesSimpleDTO> modulesSimpleDTOList = Lists.newArrayList();
                    if (CollectionUtils.isNotEmpty(enabledModules)) {
                        for (int j = 0; j < enabledModules.size(); j++) {
                            JSONObject modulesJSONObject = enabledModules.getJSONObject(j);
                            if (filterModuleExpired && Boolean.TRUE.equals(modulesJSONObject.getBoolean("expired"))) {
                                continue;
                            }
                            UserApplicationModulesSimpleDTO userApplicationModulesSimpleDTO = new UserApplicationModulesSimpleDTO();
                            userApplicationModulesSimpleDTO.setId(modulesJSONObject.getString("id"));
                            if (!validPermission) {
                                userApplicationModulesSimpleDTO.setName(modulesJSONObject.getString("name"));
                            }
                            modulesSimpleDTOList.add(userApplicationModulesSimpleDTO);
                        }
                    }
                    userApplicationSimpleDTO.setEnabledModules(modulesSimpleDTOList);
                    userApplicationList.add(userApplicationSimpleDTO);
                }
            }
        } catch (Exception ex) {
            log.error("query user application error. url:{}, request:{}, errorMsg:{}", url, JSONObject.toJSONString(requestEntity), ex.getMessage());
            return userApplicationList;
        }
        return userApplicationList;
    }

    @Override
    public List<PermissionUserFunctionalDTO> queryUserFunctional(String target, AuthoredUser authoredUser) {
        List<PermissionUserFunctionalDTO> functionalDTOList = new ArrayList<>();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = authoredUser.getToken();
        String userId = authoredUser.getUserId();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("userId", userId);
        requestMap.put("target", target);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(USER_FUNCTIONAL);
        try {
            log.info("query user functional param:{}", JSONObject.toJSONString(requestEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<JSONObject>() {
                    });
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && null != respEntity.getBody()) {
                JSONObject resultJson = respEntity.getBody().getJSONObject("result");
                functionalDTOList = JSONArray.parseArray(JSONObject.toJSONString(resultJson.getJSONArray("permissions")), PermissionUserFunctionalDTO.class);
            }
        } catch (Exception ex) {
            log.error("query user functional error. url:{}, request:{}, errorMsg:{}", url, JSONObject.toJSONString(requestEntity), ex.getMessage());
            return functionalDTOList;
        }
        return functionalDTOList;
    }

    /***
     * 获取公司别厂别
     * @return
     */
    @Override
    public JSONArray getCompanyFactory() {
        JSONArray resultArray = new JSONArray();
        //header信息
        String cascadeUrl = envProperties.getIamUri() + COMPANY_FACORY;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", Utils.getUserToken());
        HttpEntity<Object> responseHttpEntity = new HttpEntity<>(headers);
        try {
            log.info("/api/iam/v2/org/cascade param{} ", JSONObject.toJSONString(responseHttpEntity));
            ResponseEntity<JSONObject> responseEntity = restTemplate.exchange(cascadeUrl, HttpMethod.GET,
                    responseHttpEntity, JSONObject.class);
            JSONObject body = (JSONObject) responseEntity.getBody();
            resultArray = body.getJSONArray("defaultOrgAspect");
        } catch (Exception e) {
            log.error("从iam获取公司别厂别， error, token:{},request:{},errorMsg:{}", Utils.getUserToken(), JSONObject.toJSONString(responseHttpEntity), e.getMessage());
            return resultArray;
        }
        return resultArray;
    }

    @Override
    public List<String> getIamEmail() {
        List<String> emailInfo = new LinkedList<>();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", MediaType.APPLICATION_JSON_VALUE);
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(USER_LOGIN_INFO);
        String emailPrefix;
        String emailSuffix;
        try {
            ResponseEntity<JSONObject> responseEntity = restTemplate.exchange(url.toString(), HttpMethod.GET,
                    requestEntity, JSONObject.class);
            JSONObject body = responseEntity.getBody();
            if (body != null) {
                emailPrefix = body.getString("email")
                        .substring(0, body.getString("email").indexOf("@"));

                emailSuffix = body.getString("email")
                        .substring(body.getString("email").indexOf("@") + 1);

                emailInfo.add(emailPrefix);
                emailInfo.add(emailSuffix);
                return emailInfo;
            }

        } catch (Exception ex) {
            log.error("从IAM获取当前登录用户的邮件详细信息,email:{}", emailInfo);
            return new LinkedList<>();
        }

        return new LinkedList<>();
    }

    @Override
    public ResponseEntity<?> checkExitAndAddSAMLCallbackUrl(String successCallbackUrl, String failCallbackUrl) {
        ResponseEntity<?> responseEntity;
        // 1.检查是否存在
        Map<String, Object> callbackExitMap = new HashMap<>();
        String url = envProperties.getIamUri() + SAML_CALLBACK_URL_EXIT;
        try {
            ResponseEntity<JSONObject> respEntity = postIamRequest(url, new HashMap<>(0));
            int respCode = respEntity.getStatusCodeValue();
            JSONObject respBody = respEntity.getBody();
            if (org.apache.http.HttpStatus.SC_OK == respCode && respBody != null) {
                Integer code = respBody.getInteger("code");
                Boolean success = respBody.getBoolean("success");
                String message = respBody.getString("message");
                String data = respBody.getString("data");
                if (org.apache.http.HttpStatus.SC_OK == code && success && "success".equals(message) && "true".equals(data)) {
                    responseEntity = ResponseEntityWrapper.wrapperOk(true);
                } else {
                    responseEntity = addSAMLCallbackUrl(successCallbackUrl, failCallbackUrl);
                }
            } else {
                log.warn("query saml callback url return empty, url:{}, request:{}, statusCode:{}, response:{}",
                        url, callbackExitMap, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
            }
        } catch (Exception ex) {
            log.error("query saml callback url occur error, url:{}, request：{}, errorMsg：{}", url, callbackExitMap, ex.getMessage());
            responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
        }
        return responseEntity;
    }

    @Override
    public ResponseEntity<?> addSamlCallbackUrl(String successCallbackUrl, String failCallbackUrl) {
        return addSAMLCallbackUrl(successCallbackUrl, failCallbackUrl);
    }

    private ResponseEntity<?> addSAMLCallbackUrl(String successCallbackUrl, String failCallbackUrl) {
        ResponseEntity<?> responseEntity;
        // 追加回调地址
        Map<String, Object> callbackAddMap = new HashMap<>();
        callbackAddMap.put("sysName", "Athena");
        callbackAddMap.put("successCallbackUrl", successCallbackUrl);
        callbackAddMap.put("failCallbackUrl", failCallbackUrl);
        // 入参加签
        String signArg = encodeSignArg(callbackAddMap);
        String url = envProperties.getIamUri() + SAML_CALLBACK_URL_ADD;
        try {
            ResponseEntity<JSONObject> respEntity = postIamRequest(url, callbackAddMap, signArg);
            log.info("add saml callback url result, url:{}, request:{}, statusCode:{}, response:{}",
                    url, callbackAddMap, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
            int respCode = respEntity.getStatusCodeValue();
            JSONObject respBody = respEntity.getBody();
            if (null == respBody) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
            }
            if (org.apache.http.HttpStatus.SC_OK == respCode && respBody != null) {
                Integer code = respBody.getInteger("code");
                Boolean success = respBody.getBoolean("success");
                String message = respBody.getString("message");
                if (org.apache.http.HttpStatus.SC_OK == code && success && "success".equals(message)) {
                    responseEntity = ResponseEntityWrapper.wrapperOk(true);
                } else {
                    responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
                }
            } else {
                log.warn("add saml callback url return result, url:{}, request:{}, statusCode:{}, response:{}",
                        url, callbackAddMap, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, respBody.getString("errorMessage"));
            }
        } catch (Exception ex) {
            log.error("add saml callback url occur error, url:{}, request：{}, errorMsg：{}", url, callbackAddMap, ex.getMessage());
            responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
        }
        return responseEntity;
    }

    @Override
    public ResponseEntity<?> queryUserTokenByAuthCode(String authCode, String ssoType) {
        ResponseEntity<?> responseEntity;
        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("code", authCode);
        if (!ObjectUtils.isEmpty(ssoType)) {
            requestMap.put("ssoType", "cas");
        }
        String url = envProperties.getIamUri() + AUTHCODE_SAML_TOKEN;
        try {
            ResponseEntity<JSONObject> respEntity = postIamRequest(url, requestMap);
            log.info("query user token return result, authCode:{}, url:{}, request:{}, statusCode:{}, response:{}", authCode,
                    url, requestMap, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
            int respCode = respEntity.getStatusCodeValue();
            JSONObject respBody = respEntity.getBody();
            if (null == respBody) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
            }
            if (org.apache.http.HttpStatus.SC_OK == respCode && respBody != null) {
                Integer code = respBody.getInteger("code");
                Boolean success = respBody.getBoolean("success");
                String message = respBody.getString("message");
                JSONObject data = respBody.getJSONObject("data");
                if (org.apache.http.HttpStatus.SC_OK == code && success && "success".equals(message)) {
                    responseEntity = ResponseEntityWrapper.wrapperOk(data);
                } else {
                    responseEntity = ResponseEntityWrapper.wrapperOk(false);
                }
            } else {
                log.warn("query userToken from iam return result, url:{}, request:{}, statusCode:{}, response:{}",
                        url, requestMap, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, respBody.getString("errorMessage"));
            }
        } catch (Exception ex) {
            log.error("query userToken from iam occur error, authCode:{}, url:{}, request：{}, errorMsg：{}", authCode,
                    url, requestMap, ex.getMessage());
            responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
        }
        return responseEntity;
    }

    private ResponseEntity<JSONObject> postIamRequest(String url, Map<String, Object> requestParam) {
        // 请求头
        HttpHeaders header = new HttpHeaders();
        header.add("Content-Type", "application/json");
        header.add("digi-middleware-auth-app", envProperties.getAppToken());
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestParam, header);
        return restTemplate.exchange(url, HttpMethod.POST, requestEntity, JSONObject.class);
    }

    private ResponseEntity<JSONObject> postIamRequest(String url, Map<String, Object> requestParam, String encode) {
        // 请求头
        HttpHeaders header = new HttpHeaders();
        header.add("Content-Type", "application/json");
        header.add("digi-middleware-auth-app", envProperties.getAppToken());
        header.add("digi-middleware-sign-arg", encode);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestParam, header);
        return restTemplate.exchange(url, HttpMethod.POST, requestEntity, JSONObject.class);
    }

    /**
     * 查询用户语言别
     *
     * @return 返回结果
     */
    @Override
    public String queryUserLangMetadata() {
        // header信息
        String token = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        String userId = AppAuthContextHolder.getContext().getAuthoredUser().getUserId();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);

        // 请求体
        Map<String, Object> params = new HashMap<>();
        params.put("catalogId", "basic");
        params.put("key", "dapAcceptLanguage");
        params.put("userId", userId);
        params.put("tenantSid", 0);
        HttpEntity httpEntity = new HttpEntity(params, headers);
        String url = envProperties.getIamUri() + USERMETADATA_USER_GET;
        try {
            log.info("start query user lang metadata. url:{}, param:{}", url, JSONObject.toJSONString(httpEntity));
            ResponseEntity<UserLangMetadataDTO> respEntity = restTemplate.postForEntity(url, httpEntity, UserLangMetadataDTO.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                UserLangMetadataDTO baseResult = respEntity.getBody();
                Optional<Object> langOpt = Optional.ofNullable(baseResult.getData()).map(data -> data.get("value"));
                // 如果用户设置了语言别，则使用用户的语言别，如果没有使用语言别，则使用默认的语言别
                return langOpt.isPresent() ? String.valueOf(langOpt.get()) : Constants.ZH_CH_LOCALE;
            } else {
                log.error("query user lang metadata fail, url：{}, param:{}, response：{}", url,
                        JSONObject.toJSONString(httpEntity), JSONObject.toJSONString(respEntity.getBody()));
                return Constants.ZH_CH_LOCALE;
            }
        } catch (Exception ex) {
            log.error("query user lang metadata error, url：{}, param:{}, message：{}", url,
                    JSONObject.toJSONString(httpEntity), ex.getMessage());
            return Constants.ZH_CH_LOCALE;
        }
    }

    /**
     * 当前租户超管批量给用户授权应用权限
     *
     * @param appId       应用id
     * @param userSidList 用户sid列表
     * @param type        操作行为，1-删除授权，0-新增授权
     * @return 返回
     */
    public boolean batchUserAuthorizeApp(String appId, List<Long> userSidList, Integer type) {
        String url = envProperties.getIamUri() + "/api/iam/v2/tenant/batch/auth/user/update";
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();

        // header
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", userToken);

        // body
        Map<String, Object> param = new HashMap<>();
        param.put("appId", appId);
        param.put("action", type);
        param.put("userSids", userSidList);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        try {
            log.info("IAMServiceImpl | invoke batch auth user interface, url：{}, param:{}", url, JSONObject.toJSONString(httpEntity));
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONObject jsonObject = respEntity.getBody();
                return jsonObject.getBoolean("success");
            } else {
                log.error("IAMServiceImpl | batch user authorize app fail, url：{}, param:{}, response：{}", url, JSONObject.toJSONString(httpEntity), JSONObject.toJSONString(respEntity.getBody()));
                return false;
            }
        } catch (Exception ex) {
            log.error("IAMServiceImpl | batch user authorize app error, url：{}, param:{}, errorMessage：{}", url, JSONObject.toJSONString(httpEntity), ex);
            return false;
        }
    }

    /**
     * 新增sso登录系统的回调地址
     *
     * @param successCallbackUrl
     * @param failCallbackUrl
     * @param logoutCallbackUrl
     * @return
     */
    @Override
    public ResponseEntity<?> addCasCallbackUrl(String successCallbackUrl, String failCallbackUrl, String logoutCallbackUrl) {
        ResponseEntity<?> responseEntity;
        // 追加回调地址
        Map<String, Object> callbackAddMap = new HashMap<>();
        callbackAddMap.put("sysName", "Athena");
        callbackAddMap.put("successCallbackUrl", successCallbackUrl);
        callbackAddMap.put("failCallbackUrl", failCallbackUrl);
        callbackAddMap.put("logoutCallbackUrl", logoutCallbackUrl);
        callbackAddMap.put("ssoType", "cas");
        // 入参加签
        String signArg = encodeSignArg(callbackAddMap);
        String url = envProperties.getIamUri() + CAS_CALLBACK_URL_ADD;
        try {
            ResponseEntity<JSONObject> respEntity = postIamRequest(url, callbackAddMap, signArg);
            log.info("add cas callback url result, url:{}, request:{}, statusCode:{}, response:{}",
                    url, callbackAddMap, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
            int respCode = respEntity.getStatusCodeValue();
            JSONObject respBody = respEntity.getBody();
            if (null == respBody) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
            }
            if (org.apache.http.HttpStatus.SC_OK == respCode && respBody != null) {
                Integer code = respBody.getInteger("code");
                Boolean success = respBody.getBoolean("success");
                String message = respBody.getString("message");
                if (org.apache.http.HttpStatus.SC_OK == code && success && "success".equals(message)) {
                    responseEntity = ResponseEntityWrapper.wrapperOk(true);
                } else {
                    responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
                }
            } else {
                log.warn("add cas callback url return result, url:{}, request:{}, statusCode:{}, response:{}",
                        url, callbackAddMap, respEntity.getStatusCodeValue(), JSONObject.toJSONString(respEntity.getBody()));
                responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, respBody.getString("errorMessage"));
            }
        } catch (Exception e) {
            log.error("add cas callback url occur error, url:{}, request：{}, errorMsg：{}", url, callbackAddMap, e.getMessage());
            responseEntity = ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage(I18NKey.INVOKE_IAM_ERROR));
        }
        return responseEntity;
    }

    @Override
    public UserInfoDTO queryUser(String userId, Long sid, String token) {
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        headers.add(GlobalConstant.IAM_APP_TOKEN, com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        //post body
        Map<String, String> param = new HashMap<>();
        if (StringUtils.isNotEmpty(userId)) {
            param.put("id", userId);
        }
        if (Objects.nonNull(sid)) {
            param.put("sid", String.valueOf(sid));
        }

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + USER;
        ResponseEntity<UserInfoDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, UserInfoDTO.class);
        return responseEntity.getBody();
    }

    /**
     * 根据邮箱获取个人租户
     * status的值
     * -1 调用接口失败
     * 0 没有个人租户
     * 1 没有授权
     * 2 有个人租户并且有授权
     *
     * @Author：SYQ
     * @Date：2021/11/24 14:20
     */
    @Override
    public Map<String, Object> getTenantAuth(AuthoredUser authoredUser, String email, String telephone, String appId, Integer tenantType) {
        Map<String, Object> resultMap = new HashMap<>();
        //邮箱手机，不能全为空
        if (StringUtils.isEmpty(email) && StringUtils.isEmpty(telephone)) {
            return null;
        }
        //默认值为Athena
        appId = StringUtils.isNotEmpty(appId) ? appId : "Athena";
        //默认值为0
        tenantType = tenantType != null ? tenantType : 0;

        //接口地址
        String url = envProperties.getIamUri() + USER_EMAIL_TENANTS_AUTH;
        //头部数据
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, authoredUser.getToken());
        //参数
        Map<String, Object> param = new HashMap<>();
        param.put("email", email);
        param.put("telephone", telephone);
        param.put("appId", appId);
        param.put("tenantType", tenantType);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        try {
            //调用接口
            ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
            //调用失败
            if (responseEntity == null || responseEntity.getStatusCode() != HttpStatus.OK) {
                resultMap.put("status", -1);
                log.error("获取用户归属的租户以及指定应用授权信息失败");
                return resultMap;
            }
            //返回值
            EocResultDTO eocResultDTO = responseEntity.getBody();
            //返回值为空
            if (eocResultDTO == null) {
                resultMap.put("status", -1);
                log.error("获取用户归属的租户以及指定应用授权信息失败");
                return resultMap;
            }

            //解析租户信息和用户信息
            BossIamTenantAuthDTO bossIamTenantAuthDTO = JsonUtils.jsonToObject(JsonUtils.objectToString(eocResultDTO.getData()), BossIamTenantAuthDTO.class);
            //租户信息
            List<BossIamTenantDTO> bossIamTenantDTOS = bossIamTenantAuthDTO.getTenants();
            //没有个人租户
            if (CollectionUtils.isEmpty(bossIamTenantDTOS)) {
                resultMap.put("status", 0);
                log.error("获取用户归属的租户以及指定应用授权信息失败：没有个人租户");
                return resultMap;
            }
            //取第一条个人租户
            BossIamTenantDTO bossIamTenantDTO = bossIamTenantDTOS.get(0);
            //判断有没有授权
            if (!bossIamTenantDTO.getIsTenantAuth() || !bossIamTenantDTO.getIsUserAuth()) {
                resultMap.put("status", 1);
                resultMap.put("userId", bossIamTenantAuthDTO.getUserId());
                resultMap.put("tenantId", bossIamTenantDTO.getId());
                resultMap.put("isTenantAuth", bossIamTenantDTO.getIsTenantAuth());
                resultMap.put("isUserAuth", bossIamTenantDTO.getIsUserAuth());
                log.error("获取用户归属的租户以及指定应用授权信息失败（没有授权）：{}", bossIamTenantAuthDTO);
                return resultMap;
            }
            //获取用户id和租户id
            resultMap.put("status", 2);
            resultMap.put("userId", bossIamTenantAuthDTO.getUserId());
            resultMap.put("tenantId", bossIamTenantDTO.getId());
            resultMap.put("isTenantAuth", bossIamTenantDTO.getIsTenantAuth());
            resultMap.put("isUserAuth", bossIamTenantDTO.getIsUserAuth());
        } catch (Exception e) {
            resultMap.put("status", -1);
            log.error("获取用户归属的租户以及指定应用授权信息失败：{}", e);
        }
        return resultMap;
    }

    @Override
    public List<AppTenantDTO> queryTenantByApp(List<String> appIds) {
        List<AppTenantDTO> tenantInfoDTOList = new ArrayList<>();
        if (CollectionUtils.isEmpty(appIds)) {
            return tenantInfoDTOList;
        }
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("appIds", appIds);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(QUERY_TENANT_BASIC_PATH);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST,
                    requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK != respEntity.getStatusCodeValue()
                    || respEntity.getBody() == null
                    || null == respEntity.getBody().get("data")) {
                return tenantInfoDTOList;
            }
            tenantInfoDTOList = JsonUtils.jsonToObject(JSONObject.toJSONString(respEntity.getBody().get("data")), new TypeReference<List<AppTenantDTO>>() {
            });
            return tenantInfoDTOList;
        } catch (Exception ex) {
            log.error("批量根据应用id查询授权的租户基本信息 error, request：{}, errorMsg：{}", JSONObject.toJSONString(requestMap), ex.getMessage());
            return tenantInfoDTOList;
        }
    }

    @Override
    public List<AppTenantRespVO> queryBuyTenantListByAppId(String appId) {
        List<AppTenantRespVO> appTenantRespVOList = Lists.newArrayList();
        if (StringUtils.isBlank(appId)) {
            return appTenantRespVOList;
        }
        HttpHeaders headers = getCommonHeaders();
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        String urlName = "查询购买了指定应用的租户";
        String url = envProperties.getIamUri() + "/api/iam/v2/tenant/apps/" + appId;
        log.info("{}请求：{}，{}", urlName, url, com.alibaba.fastjson2.JSON.toJSONString(headers.toSingleValueMap()));
        try {
            ResponseEntity<String> response = restTemplate.exchange(url, HttpMethod.GET, requestEntity, String.class);
            if (HttpStatus.OK != response.getStatusCode() || StringUtils.isBlank(response.getBody())) {
                log.info("{}请求失败：{}，{}", urlName, response.getStatusCodeValue(), response.getBody());
                return appTenantRespVOList;
            }
            String body = response.getBody();
            log.info("{}返回：{}", urlName, StringUtils.substring(body, 0, 500));
            appTenantRespVOList = com.alibaba.fastjson2.JSON.parseArray(body, AppTenantRespVO.class);

        } catch (Exception ex) {
            log.info("{}异常：{}", urlName, ex.getMessage(), ex);
        }
        return appTenantRespVOList;
    }

    @Override
    public List<TenantUserRespVO> queryUserListByTenantId(String tenantId) {
        List<TenantUserRespVO> tenantUserRespVOList = Lists.newArrayList();
        if (StringUtils.isBlank(tenantId)) {
            return tenantUserRespVOList;
        }
        HttpHeaders headers = getCommonHeaders();
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        String urlName = "获取租户下用户列表";
        String url = envProperties.getBossIamUri() + "/api/boss/v1/users/simple/" + tenantId;
        log.info("{}请求：{}，{}", urlName, url, com.alibaba.fastjson2.JSON.toJSONString(headers.toSingleValueMap()));
        try {
            ResponseEntity<String> response = restTemplate.exchange(url, HttpMethod.GET, requestEntity, String.class);
            if (HttpStatus.OK != response.getStatusCode() || StringUtils.isBlank(response.getBody())) {
                log.info("{}请求失败：{}，{}", urlName, response.getStatusCodeValue(), response.getBody());
                return tenantUserRespVOList;
            }
            String body = response.getBody();
            log.info("{}返回：{}", urlName, StringUtils.substring(body, 0, 500));
            TenantUserBaseRespVO tenantUserBaseRespVO = com.alibaba.fastjson2.JSON.parseObject(body, TenantUserBaseRespVO.class);
            tenantUserRespVOList = tenantUserBaseRespVO.getData();

        } catch (Exception ex) {
            log.info("{}异常：{}", urlName, ex.getMessage(), ex);
        }
        return tenantUserRespVOList;
    }

    @Override
    public UserAllPermissionBaseRespVO queryUserAllPermission(UserAllPermissionReqDTO reqDTO) {
        UserAllPermissionBaseRespVO userAllPermissionBaseRespVO = null;
        // header信息
        HttpHeaders headers = getCommonHeaders();
        // 请求体
        Map<String, Object> requestMap = BeanUtil.beanToMap(reqDTO, false, true);
        String urlName = "获取用户的功能权限";
        String url = envProperties.getIamUri() + "/api/iam/v2/permission/user/all";
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        log.info("{}请求：{}，{}，{}", urlName, url, com.alibaba.fastjson2.JSON.toJSONString(requestMap), com.alibaba.fastjson2.JSON.toJSONString(headers.toSingleValueMap()));
        try {
            ResponseEntity<String> response = restTemplate.exchange(url, HttpMethod.POST, requestEntity, String.class);
            if (HttpStatus.OK != response.getStatusCode() || StringUtils.isBlank(response.getBody())) {
                log.info("{}请求失败：{}，{}", urlName, response.getStatusCodeValue(), response.getBody());
                return null;
            }
            String body = response.getBody();
            log.info("{}返回：{}", urlName, StringUtils.substring(body, 0, 500));
            userAllPermissionBaseRespVO = com.alibaba.fastjson2.JSON.parseObject(body, UserAllPermissionBaseRespVO.class);

        } catch (Exception ex) {
            log.info("{}异常：{}", urlName, ex.getMessage(), ex);
        }
        return userAllPermissionBaseRespVO;
    }

    /**
     * 获取公共请求头
     *
     * @return
     */
    private HttpHeaders getCommonHeaders() {
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        header.add("digi-middleware-auth-app", envProperties.getAppToken());
        header.add("digi-middleware-auth-user", userToken);
        return header;
    }

    /**
     * 查询当前租户下所有角色列表
     *
     * @return 返回
     */
    @Override
    public List<RoleDTO> queryAllRoleList() {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        HttpEntity<?> httpEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(QUERY_TENANT_ROLE_LIST);
        try {
            ResponseEntity<List<RoleDTO>> respEntity = restTemplate.exchange(url.toString(), HttpMethod.GET, httpEntity,
                    new ParameterizedTypeReference<List<RoleDTO>>() {
                    });
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return respEntity.getBody();
            } else {
                log.error("query role list fail. url:{}, request:{}, response:{}", url, httpEntity, JSONObject.toJSONString(respEntity.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("query role list occur error. url:{}, request:{}, errorMsg:{}", url, httpEntity, ex.getMessage());
            return null;
        }
    }

    /**
     * 查询当前租户下的租户账套映射关系
     *
     * @param appId     应用id
     * @param companyId 公司别
     * @param account   账套id
     * @return 返回
     */
    public List<AccountDTO> queryTenantAccountMappingList(String appId, String companyId, String account) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 拼接查询条件
        String urlPath = "/api/iam/v2/tenantmapping/list?appId=";
        if (StringUtils.isNotBlank(appId)) {
            urlPath = urlPath + appId;
        } else if (StringUtils.isNotBlank(companyId)) {
            urlPath = urlPath + "&companyId=" + companyId;
        } else if (StringUtils.isNotBlank(account)) {
            urlPath = urlPath + "&account=" + account;
        }

        // 请求体
        HttpEntity<?> httpEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(urlPath);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.GET, httpEntity,
                    new ParameterizedTypeReference<JSONObject>() {
                    });
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONArray data = respEntity.getBody().getJSONArray("data");
                return JSONArray.parseArray(data.toJSONString(), AccountDTO.class);
            } else {
                log.error("query tenant account mapping list fail. url:{}, request:{}, response:{}", url, httpEntity, JSONObject.toJSONString(respEntity.getBody()));
                return Lists.newArrayList();
            }
        } catch (Exception ex) {
            log.error("query tenant account mapping error. url:{}, request:{}, errorMsg:{}", url, httpEntity, ex.getMessage());
            return Lists.newArrayList();
        }
    }

    @Override
    public TenantSimpleDTO queryTenantSimple() {
        TenantSimpleDTO simpleDTO = new TenantSimpleDTO();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        // 请求体
        HttpEntity<?> httpEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(TENANT_CURRENT_SIMPLE);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.GET, httpEntity,
                    new ParameterizedTypeReference<JSONObject>() {
                    });
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                simpleDTO = JSONObject.parseObject(JSONObject.toJSONString(respEntity.getBody().getJSONObject("data")), TenantSimpleDTO.class);
                return simpleDTO;
            } else {
                log.error("/api/iam/v2/tenant/current/simple fail. url:{}, request:{}, response:{}", url, httpEntity, JSONObject.toJSONString(respEntity.getBody()));
                return simpleDTO;
            }
        } catch (Exception ex) {
            log.error("/api/iam/v2/tenant/current/simple error. url:{}, request:{}, errorMsg:{}", url, httpEntity, ex.getMessage());
            return simpleDTO;
        }
    }

    @Override
    public List<OrgAspectDTO> getOrgAspect() {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("id", "defaultOrgAspect");
        requestMap.put("catalogId", "defaultOrgCatalog");
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(ORG_ASPECT);
        try {
            ResponseEntity<List<OrgAspectDTO>> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST,
                    requestEntity, new ParameterizedTypeReference<List<OrgAspectDTO>>() {
                    });
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue()) {
                return respEntity.getBody();
            }
        } catch (Exception ex) {
            log.error("/api/iam/v2/org/aspect error. url:{}, request:{}, errorMsg:{}", url, requestEntity, ex.getMessage());
            return null;
        }
        return null;
    }

    /**
     * 查询当前租户下三方账号映射的用户信息
     *
     * @param userToken    userToken
     * @param appCode      应用code
     * @param verifyUserId 三方账号
     * @return 返回
     */
    @Override
    public String queryVerifyMappingUser(String userToken, String appCode, String verifyUserId) {
        if (StringUtils.isBlank(appCode) || StringUtils.isBlank(verifyUserId)) {
            return null;
        }
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("identityId", appCode);
        requestMap.put("verifyUserId", verifyUserId);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(QUERY_USER_BY_VERIFY_USER_ID);

        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                String userId = respEntity.getBody().getString("verifyUserId");
                return userId;
            } else {
                log.info("query by verify userId return empty, url:{}, request:{}, response:{}", url, requestEntity, JSONObject.toJSONString(respEntity.getBody()));
                return null;
            }
        } catch (Exception ex) {
            log.error("query by verify userId error, url:{}, request:{}, errMsg:{}", url, requestEntity, ex.getMessage(), ex);
            return null;
        }
    }

    /**
     * 查询当前租户下三方账号映射的用户信息
     *
     * @param userToken    userToken
     * @param appCode      应用code
     * @return 返回
     */
    @Override
    public List<AppUserDTO> queryAllVerifyMappingUser(String userToken, String appCode) {
        if (StringUtils.isBlank(appCode)) {
            return null;
        }
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("identityId", appCode);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(QUERY_All_VERIFY_MAPPING_USER);

        try {
            ResponseEntity<JSONArray> respEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, requestEntity, JSONArray.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                return JSONArray.parseArray(respEntity.getBody().toString(), AppUserDTO.class);
            } else {
                log.info("query all verify mapping user return empty, url:{}, request:{}, response:{}", url, requestEntity, JSONObject.toJSONString(respEntity.getBody()));
                return Lists.newArrayList();
            }
        } catch (Exception ex) {
            log.error("query all verify mapping user error, url:{}, request:{}, errMsg:{}", url, requestEntity, ex.getMessage(), ex);
            return Lists.newArrayList();
        }
    }
}