package com.digiwin.athena.semc.proxy.tripartite.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.dto.portal.LabelSystemPreReq;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.tripartite.service.TripartiteService;
import com.digiwin.athena.semc.util.DataTypeAdapter;
import com.digiwin.athena.semc.util.Utils;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 非混合云方式请求三方系统相关接口
 */
@Service
@Slf4j
public class TripartiteServiceImpl implements TripartiteService {

    @Resource
    private RestTemplate restTemplate;

    @Resource
    EnvProperties envProperties;

    private static final String BODY_STD_DATA = "std_data";

    /**
     * 三方应用业务待办类型请求url
     */
    private static final String THIRD_GROUP_BUSINESS_TODO_CONFIG_URL = "/third/business/todo/queryGroupList";

    /**
     * 三方应用业务待办数量请求url
     */
    private static final String THIRD_GROUP_BUSINESS_TODO_VALUE_URL = "/third/business/todo/queryGroupValueList";

    @Override
    public Map<String, Object> selectDataPagePost(String url, LabelSystemPreReq bodyParam,
                                                  String appToken) {
        Map<String, Object> result = new HashMap<>();
        String token = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-userToken", token);
        //com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA
        headers.add("digi-appToken", StringUtils.isBlank(appToken) ? envProperties.getAppToken() : appToken);
        // 请求体
        // 创建查询条件
        Map<String, Object> bodyMap = new HashMap<>();
        // 包装分页信息
        if (bodyParam.getPageSize() != null) {
            bodyMap.put("page_size", bodyParam.getPageSize());
        }
        if (bodyParam.getPageNum() != null) {
            bodyMap.put("page_no", bodyParam.getPageNum());
        }
        // 包装查询条件
        if (StringUtils.isNotEmpty(bodyParam.getQryCondition())) {
            bodyMap.put("query_condition", bodyParam.getQryCondition());
        }
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(bodyMap, headers);
        try {
            log.info("请求三方系统start url:{},req:{}", url, bodyMap);
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST,
                    requestEntity, JSONObject.class);
            log.info("请求三方系统结果 url:{},req:{},resp:{}", url, bodyMap,
                    JSON.toJSONString(respEntity.getBody()));
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue()
                    && respEntity.getBody() != null) {
                Gson gson = new GsonBuilder().registerTypeAdapter(new TypeToken<Map<String, Object>>() {
                }.getType(), new DataTypeAdapter()).create();
                result = gson.fromJson(JSON.toJSONString(respEntity.getBody().getJSONObject("response")),
                        (new TypeToken<Map<String, Object>>() {
                        }).getType());
            }
        } catch (Exception e) {
            log.error("请求三方系统异常,,req:{},url:{},errorMsg:{}", bodyMap, url, e);
            return result;
        }
        return result;
    }

    /**
     * 查询三方应用统计类型数据
     *
     * @param countCode 统计分类
     * @param contentCodeList 统计分类下类型
     * @param url 请求url
     * @param locale 语系
     * @return 返回
     */
    @Override
    public Map<String, Object> selectDataCountListPost(String countCode, List<String> contentCodeList, String url, String locale) {
        Map<String, Object> result = new HashMap<>();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        // 请求体
        Map<String, Object> bodyMap = new HashMap<>();
        if (StringUtils.isNotEmpty(countCode) && CollectionUtils.isNotEmpty(contentCodeList)) {
            bodyMap.put("countType", countCode);
            bodyMap.put("countContent", contentCodeList);
        }
        bodyMap.put("locale", locale);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(bodyMap, headers);
        try {
            log.info("start query third count data. url:{}, req:{}", url, bodyMap);
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url,
                    HttpMethod.POST, requestEntity, JSONObject.class);
            log.info("start query third count data end. url:{}, req:{}, resp:{}", url, bodyMap, JSON.toJSONString(respEntity.getBody()));
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                Gson gson = new GsonBuilder().registerTypeAdapter(new TypeToken<Map<String, Object>>() {
                }.getType(), new DataTypeAdapter()).create();
                result = gson.fromJson(JSON.toJSONString(respEntity.getBody().getJSONObject("response")), (new TypeToken<Map<String, Object>>() {
                }).getType());
            }
        } catch (Exception e) {
            log.error("query third count data occur error. url:{}, req:{}", url, bodyMap, e);
            return result;
        }
        return result;
    }

    /**
     * 查询异构系统消息
     *
     * @param url      请求url
     * @param pageNo   当前页
     * @param pageSize 页大小
     * @return
     */
    @Override
    public Map<String, Object> queryThirdMessageDataPage(String url, Integer pageNo, Integer pageSize) {
        Map<String, Object> result = new HashMap<>();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        // 请求体
        // 创建查询条件
        Map<String, Object> bodyMap = new HashMap<>();
        // 包装分页信息
        if (pageNo != null) {
            bodyMap.put("page_no", pageNo);
        }
        if (pageSize != null) {
            bodyMap.put("page_size", pageSize);
        }
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(bodyMap, headers);
        try {
            log.info("start query third message data page. url:{}, req:{}", url, bodyMap);
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, JSONObject.class);
            log.info("query third message data page over. url:{},req:{},resp:{}", url, bodyMap, JSON.toJSONString(respEntity.getBody()));
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                Gson gson = new GsonBuilder().registerTypeAdapter(new TypeToken<Map<String, Object>>() {
                }.getType(), new DataTypeAdapter()).create();
                result = gson.fromJson(JSON.toJSONString(respEntity.getBody().getJSONObject("response")), (new TypeToken<Map<String, Object>>() {
                }).getType());
            } else {
                log.info("query third message data page fail. url:{}, param:{}, resp:{}", url, bodyMap, JSON.toJSONString(respEntity.getBody()));
            }
        } catch (Exception e) {
            log.error("query third message data page error, url:{}, param:{}, errorMsg:{}", bodyMap, url, e);
            return result;
        }
        return result;
    }

    @Override
    public JSONArray queryThirdData(String url, String appToken) {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-appToken", StringUtils.isBlank(appToken) ? envProperties.getAppToken() : appToken);
        // 请求体
        Map<String, Object> bodyMap = new HashMap<>();
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(bodyMap, headers);
        try {
            log.info("请求三方系统数据 start url:{},req:{}", url, bodyMap);
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST,
                    requestEntity, JSONObject.class);
            log.info("请求三方系统数据 结果 url:{},req:{},resp:{}", url, bodyMap, JSON.toJSONString(respEntity.getBody()));
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                JSONArray data = respEntity.getBody().getJSONArray("response");
                return data;
            }
        } catch (Exception e) {
            log.error("请求三方系统数据 异常,,req:{},url:{},errorMsg:{}", bodyMap, url, e);
            return null;
        }
        return null;
    }

    @Override
    public Map<String, Object> queryThirdData(String url, String appToken, Map<String, Object> bodyMap) {
        Map<String, Object> result = new HashMap<>();
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-appToken", StringUtils.isBlank(appToken) ? envProperties.getAppToken() : appToken);
        // 请求体
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(bodyMap, headers);
        try {
            log.info("请求三方系统数据 start url:{},req:{}", url, bodyMap);
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST,
                    requestEntity, JSONObject.class);
            log.info("请求三方系统数据 结果 url:{},req:{},resp:{}", url, bodyMap, JSON.toJSONString(respEntity.getBody()));
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                Gson gson = new GsonBuilder().registerTypeAdapter(new TypeToken<Map<String, Object>>() {
                }.getType(), new DataTypeAdapter()).create();
                result = gson.fromJson(JSON.toJSONString(respEntity.getBody().getJSONObject("response")), (new TypeToken<Map<String, Object>>() {
                }).getType());
                return result;
            }
        } catch (Exception e) {
            log.error("请求三方系统数据 异常,,req:{},url:{},errorMsg:{}", bodyMap, url, e);
            return null;
        }
        return null;
    }

    /**
     * 查询三方应用业务待办类型
     *
     * @param restUrl 请求地址
     * @return 返回
     */
    @Override
    public Map<String, Object> queryThirdBusinessTodoGroupList(String restUrl) {
        Map<String, Object> result = new HashMap<>();

        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        // 请求体
        Map<String, Object> bodyMap = new HashMap<>();
        bodyMap.put("locale", LocaleContextHolder.getLocale().toString());
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(bodyMap, headers);

        // url
        String url = restUrl + THIRD_GROUP_BUSINESS_TODO_CONFIG_URL;
        try {
            log.info("start query third business todo group list. url:{}, param:{}", url, requestEntity);
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, JSONObject.class);
            log.info("query third business todo group list end. url:{}, param:{}, result:{}", restUrl, bodyMap,
                    JSON.toJSONString(respEntity.getBody()));
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                Gson gson = new GsonBuilder().registerTypeAdapter(new TypeToken<Map<String, Object>>() {
                }.getType(), new DataTypeAdapter()).create();
                result = gson.fromJson(JSON.toJSONString(respEntity.getBody().getJSONObject("response")),
                        (new TypeToken<Map<String, Object>>() {
                        }).getType());
            } else {
                log.info("query third business todo group list error fail. url:{}, param:{}, resp:{}", url, bodyMap, JSON.toJSONString(respEntity.getBody()));
            }
        } catch (Exception e) {
            log.error("query third business todo group list error. url:{}, param:{}, errorMsg:{}", url, bodyMap, e);
            return result;
        }
        return result;
    }

    /**
     * 查询三方应用业务待办数量
     *
     * @param labelSystemData 请求参数
     * @param appToken        应用appToken
     * @return 返回
     */
    @Override
    public Map<String, Object> queryThirdBusinessTodoValueList(LabelSystemData labelSystemData, String appToken) {
        Map<String, Object> result = new HashMap<>();

        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-userToken", Utils.getUserToken());
        headers.add("digi-appToken", appToken);

        // 请求体
        Map<String, Object> bodyMap = new HashMap<>();
        bodyMap.put("locale", LocaleContextHolder.getLocale().toString());

        // 如果应用只有业务待办类型，没有下级，则按条件值传参查询
        if (StringUtils.isBlank(labelSystemData.getCountContentCode())) {
            bodyMap.put("groupConditionCode", labelSystemData.getCountCode());
        } else {
            bodyMap.put("groupCode", labelSystemData.getCountCode());
            bodyMap.put("groupConditionCode", labelSystemData.getCountContentCode());
        }
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(bodyMap, headers);

        // url
        String url = labelSystemData.getRestUrl() + THIRD_GROUP_BUSINESS_TODO_VALUE_URL;
        try {
            log.info("start query third business todo group value list. url:{}, param:{}", url, requestEntity);
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, JSONObject.class);
            log.info("query third business todo group value list end. url:{}, param:{}, result:{}", url, bodyMap,
                    JSON.toJSONString(respEntity.getBody()));
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                Gson gson = new GsonBuilder().registerTypeAdapter(new TypeToken<Map<String, Object>>() {
                }.getType(), new DataTypeAdapter()).create();
                result = gson.fromJson(JSON.toJSONString(respEntity.getBody().getJSONObject("response")),
                        (new TypeToken<Map<String, Object>>() {
                        }).getType());
            } else {
                log.info("query third business todo group value list error fail. url:{}, param:{}, resp:{}", url, bodyMap, JSON.toJSONString(respEntity.getBody()));
            }
        } catch (Exception e) {
            log.error("query third business todo group value list error. url:{}, param:{}, errorMsg:{}", url, bodyMap, e);
            return result;
        }
        return result;
    }
}