package com.digiwin.athena.semc.proxywrapper.impl;


import com.alibaba.fastjson2.JSON;
import com.digiwin.athena.semc.common.RedisConstant;
import com.digiwin.athena.semc.proxy.eoc.service.EocService;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxywrapper.IamUserService;
import com.digiwin.athena.semc.proxywrapper.dto.IamUserAuthInfo;
import com.digiwin.athena.semc.service.cache.CommonConfigService;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.util.FormatUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReference;

import io.vavr.Tuple2;
import io.vavr.Tuple3;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@RequiredArgsConstructor
@Service
@Slf4j
public class IamUserServiceImpl implements IamUserService {

    @Qualifier("semcAsyncExecutor")
    private final Executor asyncTaskExecutor;


    private final IamService iamService;


    private final EocService eocService;


    private final ICacheService cacheService;

    private final CommonConfigService commonConfigService;


    @Override
    public Tuple3<List<Long>, List<Long>, List<Long>> getUserAuthInfo(String userId, String userToken) {
        AtomicReference<List<Long>> userOrgInfoDTO = new AtomicReference<>();
        AtomicReference<List<Long>> roleDTOList = new AtomicReference<>();
        AtomicReference<List<Long>> userInfoDetailDTO = new AtomicReference<>();
        // 查询用户所在组织
        CompletableFuture<List<Long>> completableFutureOrg = CompletableFuture.supplyAsync(() -> iamService.qryUserOrgSidList(userId, userToken), asyncTaskExecutor);
        // 查询用户所在角色
        CompletableFuture<List<Long>> completableFutureRole = CompletableFuture.supplyAsync(() -> iamService.queryUserRolesSidList(userId, userToken), asyncTaskExecutor);
        // 查询用户所在部门
        CompletableFuture<List<Long>> completableFutureDept = CompletableFuture.supplyAsync(() -> eocService.queryUserDeptSidList(userId), asyncTaskExecutor);
        CompletableFuture.allOf(completableFutureOrg, completableFutureRole, completableFutureDept).join();
        completableFutureOrg.handle((result, ex) -> {
            if (ex != null) {
                log.error("查询用户所在组织异常：",ex);
            } else {
                userOrgInfoDTO.set(result);
            }
            return result;
        });
        completableFutureRole.handle((result, ex) -> {
            if (ex != null) {
                log.error("查询用户所在角色异常：", ex);
            } else {
                roleDTOList.set(result);
            }
            return result;
        });
        completableFutureDept.handle((result, ex) -> {
            if (ex != null) {
                log.error("查询用户所在部门异常：", ex);
            } else {
                userInfoDetailDTO.set(result);
            }
            return result;
        });
        return new Tuple3<>(userOrgInfoDTO.get(), roleDTOList.get(), userInfoDetailDTO.get());
    }

    private IamUserAuthInfo getNoCacheUserAuthSids(String userId, String userToken) {
        Tuple3<List<Long>, List<Long>, List<Long>> tuple3 = getUserAuthInfo(userId, userToken);
        List<Long> orgSidList = tuple3._1;
        List<Long> roleSidList = tuple3._2;
        List<Long> deptSidList = tuple3._3;
        IamUserAuthInfo iamUserAuthInfo = new IamUserAuthInfo();
        boolean hasAuth = false;
        if (CollectionUtils.isNotEmpty(orgSidList)) {
            hasAuth = true;
            iamUserAuthInfo.setOrgSids(orgSidList);
        }
        if (CollectionUtils.isNotEmpty(roleSidList)) {
            hasAuth = true;
            iamUserAuthInfo.setRoleSids(roleSidList);
        }
        if (CollectionUtils.isNotEmpty(deptSidList)) {
            hasAuth = true;
            iamUserAuthInfo.setDeptSids(deptSidList);
        }
        if (hasAuth) {
            return iamUserAuthInfo;
        }
        return null;
    }


    @Override
    public IamUserAuthInfo getUserAuthSids(String userId, String tenantId,String userToken) {
        Tuple2<Boolean, Integer> tuple2 = commonConfigService.getIamUserAuthInfoConfig();
        if (Boolean.TRUE.equals(tuple2._1)) {
            String cacheKey = FormatUtil.format(RedisConstant.IAM_USER_AUTH_INFO_KEY, userId, tenantId);
            String cacheValue = cacheService.getValue(cacheKey);
            if (StringUtils.isNotBlank(cacheValue)) {
                return JSON.parseObject(cacheValue, IamUserAuthInfo.class);
            } else {
                IamUserAuthInfo iamUserAuthInfo = getNoCacheUserAuthSids(userId, userToken);
                if (iamUserAuthInfo != null) {
                    cacheService.cache(cacheKey, JSON.toJSONString(iamUserAuthInfo), Duration.ofSeconds(tuple2._2));
                }
                return iamUserAuthInfo;
            }
        }
        return getNoCacheUserAuthSids(userId, userToken);


    }

    /**
     * 删除iam用户权限信息sid集合
     *
     * @param userId 用户id
     * @param tenantId 租户id
     */
    @Override
    public void removeUserAuthInfo(String userId, String tenantId) {
        String cacheKey = FormatUtil.format(RedisConstant.IAM_USER_AUTH_INFO_KEY, userId, tenantId);
        cacheService.delete(cacheKey);
    }

    /**
     * 删除是否是超管标识缓存
     *
     * @param tenantId 租户id
     * @param userId 用户id
     */
    @Override
    public void removeSuperAdminFlagCache(String tenantId, String userId) {
        String cacheKey = FormatUtil.format(RedisConstant.SUPER_ADMIN_FLAG, tenantId, userId);
        cacheService.delete(cacheKey);
    }
}
