
package com.digiwin.athena.semc.quartz;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.enums.EAIServiceNameEnum;
import com.digiwin.athena.semc.dto.mq.MessageDO;
import com.digiwin.athena.semc.entity.message.ThirdMessageInfo;
import com.digiwin.athena.semc.event.ThirdMessageWecomEvent;
import com.digiwin.athena.semc.event.dto.ThirdMessageWecomEventDTO;
import com.digiwin.athena.semc.mapper.message.ThirdMessageInfoMapper;
import com.digiwin.athena.semc.mapper.message.ThirdSystemMessageMapper;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.tripartite.service.TripartiteService;
import com.digiwin.athena.semc.service.message.ThirdMessageInfoService;
import com.digiwin.athena.semc.service.mq.MessageSendService;
import com.digiwin.athena.semc.service.sso.IErpSsoInfoService;
import com.digiwin.athena.semc.util.DateUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.quartz.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.scheduling.quartz.QuartzJobBean;
import org.springframework.scheduling.quartz.SchedulerFactoryBean;
import org.springframework.stereotype.Component;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @Author: sungqz
 * @Date: 2024/3/5
 * @Version 1.0
 * @Description quartz任务接收执行 异构系统消息拉取定时任务
 */

@Component
@Slf4j
public class ThirdMessagePullTaskJob extends QuartzJobBean {

    @Resource
    private SchedulerFactoryBean schedulerFactoryBean;


    @Autowired
    private TripartiteService tripartiteService;

    @Autowired
    private ESPService espService;

    @Autowired
    private ThirdMessageInfoMapper thirdMessageInfoMapper;

    @Autowired
    private ThirdMessageInfoService thirdMessageInfoService;

    @Autowired
    private IErpSsoInfoService erpSsoInfoService;

    @Autowired
    private  MessageSendService messageSendService;

    @Resource
    private IamService iamService;

    @Resource
    private ApplicationEventPublisher eventPublisher;

    @Override
    protected void executeInternal(JobExecutionContext context) throws JobExecutionException {
        try {
            List<Map<String, Object>> messageList = new ArrayList<>();
            Map<String, Object> params = (Map<String, Object>) context.getJobDetail().getJobDataMap().get("params");
            Integer appAccessModel = (Integer) params.get("appAccessModel");
            String id=ObjectUtils.isNotEmpty(params.get("id")) ? params.get("id").toString() : "";
            String tenantId=ObjectUtils.isNotEmpty(params.get("tenantsid")) ? params.get("tenantsid").toString() : "";

            //混合云
            if (appAccessModel == 0) {
                AuthoredUser athenaUser = new AuthoredUser();
                athenaUser.setTenantId(tenantId);
                AppAuthContextHolder.getContext().setAuthoredUser(athenaUser);
                //默认设置中文简体
                LocaleContextHolder.setLocale(Locale.SIMPLIFIED_CHINESE);
                String middleSystemName = params.get("middleSystemName").toString();
                String middleSystemUid = params.get("middleSystemUid").toString();
                Map<String, String> extHeader = new HashMap<>();
                extHeader.put("digi-userToken", "");
                extHeader.put("digi-appToken", "");
                // 通过ESP查询三方系统的消息数据
                for (int i = 1; ; i++) {
                    Map<String,Object>parameter=new HashMap<>();
                    parameter.put("page_no",i);
                    parameter.put("page_size",100);
                    parameter.put("tenant_id",tenantId);
                    Map<String, Object> result = espService.queryByEsp(middleSystemName,
                            middleSystemUid, EAIServiceNameEnum.MESSAGE_PULL.getServiceName(), extHeader, parameter, null,
                            null);
                    // 没有显示字段时，返回异常
                    if (Objects.isNull(result.get("message_list"))) {
                        System.out.println("混合云未查询到消息数据");
                        break;
                    }
                    messageList = (List<Map<String, Object>>) result.get("message_list");
                        System.out.println("混合云消息数据"+messageList);
                        //插入消息数据
                        handleMessageData(params, messageList);
                        System.out.println("混合云消息任务:"+id+"执行成功！");
                    if (!(Boolean)result.get("has_next")) {
                        break;
                    }
                }
            }
            //非混合云
            if (appAccessModel == 1) {
                for (int i = 1; ; i++) {
                    String domain = params.get("domain").toString();
                    Map<String, Object> bodyMap = new HashMap<>();
                    bodyMap.put("page_no", i);
                    bodyMap.put("page_size", 100);
                    bodyMap.put("tenant_id",tenantId);
                    Map<String, Object> jobData = tripartiteService.queryThirdData(domain + Constants.QUERY_MESSAGEJOB_LIST_URL, "", bodyMap);
                    if (null == jobData) {
                        System.out.println("非混合云未查询到消息数据");
                        break;
                    }
                    messageList=(List<Map<String, Object>>)jobData.get("message_list");
                    System.out.println("非混合云消息数据"+messageList);
                    //插入消息数据
                    handleMessageData(params, messageList);
                    System.out.println("非混合云消息任务:"+id+"执行成功！");
                    if (!(Boolean)jobData.get("has_next")) {
                        break;
                    }
                }
            }
        }catch (Exception e){
            System.out.println(e.getMessage());
            System.out.println("消息任务执行失败");
        }
    }

    /**
     * @Author xuzd
     * @Date 2024/5/9 17:52
     * @Description: 添加定时任务
     */
    public void addJob(Map<String, Object> params) {
        try {
            String id=params.get("id").toString();
            Integer pullPeriod = (Integer) params.get("pullPeriod");
            //pullPeriod=pullPeriod*3600;
            //1:创建调度器
            Scheduler scheduler = schedulerFactoryBean.getScheduler();
            //2:创建任务实例
            JobDetail jobDetail = JobBuilder.newJob(ThirdMessagePullTaskJob.class).
                    withIdentity("ThirdMessagePullTaskJob", "MessageJob_" + id).storeDurably(true).
                    build();
            //设置参数，在定时任务执行时可以动态获取这些参数
            jobDetail.getJobDataMap().put("params", params);
            //3:创建触发器Trigger
            Trigger trigger = TriggerBuilder.newTrigger().
                    withIdentity("ThirdMessagePullTaskJob", "MessageTrigger_" + id).
                    withSchedule(SimpleScheduleBuilder.simpleSchedule().withIntervalInHours(pullPeriod).repeatForever()).
                    build();
            //4:触发器和任务关联
            scheduler.scheduleJob(jobDetail, trigger);
            //5:启动任务调度器
            scheduler.start();
        } catch (SchedulerException e) {
            e.printStackTrace();
        }
    }

    /**
     * @Author xuzd
     * @Date 2024/5/10 9:59
     * @Description: 暂停定时任务
     */
    public void pauseJob(String name, String group) throws SchedulerException {
        Scheduler scheduler = schedulerFactoryBean.getScheduler();
        JobKey jobKey = new JobKey(name, group);
        JobDetail jobDetail = scheduler.getJobDetail(jobKey);
        if (jobDetail == null) {
            return;
        }
        scheduler.pauseJob(jobKey);
    }


    /**
     * @Author xuzd
     * @Date 2024/5/10 10:01
     * @Description: 恢复定时任务
     */
    public void resumeJob(String name, String group) throws SchedulerException {
        Scheduler scheduler = schedulerFactoryBean.getScheduler();
        JobKey jobKey = new JobKey(name, group);
        JobDetail jobDetail = scheduler.getJobDetail(jobKey);
        if (jobDetail == null) {
            return;
        }
        scheduler.resumeJob(jobKey);
    }


    /**
     * @Author xuzd
     * @Date 2024/5/10 10:02
     * @Description: 删除定时任务
     */
    public void deleteJob(String name, String group) throws Exception {
        Scheduler scheduler = schedulerFactoryBean.getScheduler();
        JobKey jobKey = new JobKey(name, group);
        JobDetail jobDetail = scheduler.getJobDetail(jobKey);
        if (jobDetail == null) {
            return;
        }
        scheduler.deleteJob(jobKey);
    }


    /**
     * @Author xuzd
     * @Date 2024/5/13 14:45
     * @Description: 消息数据处理
     */
    public void handleMessageData(Map<String, Object> params, List<Map<String, Object>> messageList){
        String messageAppCode = ObjectUtils.isNotEmpty(params.get("appCode")) ? params.get("appCode").toString() : "";
        String appId = ObjectUtils.isNotEmpty(params.get("appId")) ? params.get("appId").toString() : "";
        String tenantId = ObjectUtils.isNotEmpty(params.get("tenantsid")) ? params.get("tenantsid").toString() : "";
        String sendChannel = ObjectUtils.isNotEmpty(params.get("sendChannel")) ? params.get("sendChannel").toString() : "";

        // 是否全员可见标识
        boolean allSeeFlag = Boolean.FALSE;
        List<ThirdMessageInfo> messageInfos = new ArrayList<>();
        List<ThirdMessageInfo> newMessageInfoList = new ArrayList<>();
        List<String> messageUserList = new ArrayList<>();

        QueryWrapper<ThirdMessageInfo> condition = new QueryWrapper<ThirdMessageInfo>()
                .eq("message_app_id", appId);
        List<ThirdMessageInfo> thirdMessageInfoList = thirdMessageInfoService.list(condition);
        Map<String, ThirdMessageInfo> existMessageMap = CollectionUtils.isNotEmpty(thirdMessageInfoList) ?
                thirdMessageInfoList.stream().collect(Collectors.toMap(ThirdMessageInfo::getMessageId, a -> a, (k1, k2) -> k1)) : Maps.newHashMap();

        for(Map<String,Object>map:messageList){
            ThirdMessageInfo messageInfo = new ThirdMessageInfo();
            messageInfo.setMessageAppCode(messageAppCode);
            messageInfo.setMessageAppId(appId);
            messageInfo.setMessageId(map.get("message_id").toString());
            messageInfo.setMessageTitle(map.get("message_title").toString());
            messageInfo.setMessageUrl(map.get("message_url").toString());
            messageInfo.setMessagePublishTime(map.get("message_publish_time").toString());
            messageInfo.setMessageContent(map.get("message_content").toString());
            messageInfo.setMessageOwner(map.get("message_owner").toString());
            String accountId = MapUtils.getString(map, "account_id");
            if(StringUtils.isNotBlank(accountId)){
                messageInfo.setAccountId(accountId);
                messageInfo.setAccountName(MapUtils.getString(map, "account_name"));
            }
            messageInfo.setCreateTime(DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER));
            messageInfo.setMessageAllStaffRead(Integer.parseInt(map.get("message_all_staff_read").toString()));
            messageInfo.setTenantId(tenantId);
            messageInfo.setSideType(MapUtils.getInteger(map,"sideType",null));
            messageInfo.setMobileMessageUrl(MapUtils.getString(map,"mobileMessageUrl",null));
            // 更新的消息,更新消息的可查看用户列表
            if (existMessageMap.containsKey(messageInfo.getMessageId())) {
                ThirdMessageInfo existInfo = existMessageMap.get(messageInfo.getMessageId());
                messageInfo.setId(existInfo.getId());
                messageInfo.setModifyTime(DateUtils.getNowTime(null));
                // 全员可见存储为"all"
                if (Constants.MESSAGE_ALL_STAFF_FLAG.equals(messageInfo.getMessageAllStaffRead())) {
                    messageInfo.setMessageOwner("all");
                    allSeeFlag = Boolean.TRUE;
                } else {
                    messageInfo.setMessageOwner(existInfo.getMessageOwner().equals("all") ? messageInfo.getMessageOwner() : existInfo.getMessageOwner() + "," + messageInfo.getMessageOwner());
                    messageUserList.addAll(Arrays.asList(messageInfo.getMessageOwner().split(",")));
                }
            } else { // 新增的消息
                messageInfo.setCreateTime(DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER));
                if (Constants.MESSAGE_ALL_STAFF_FLAG.equals(messageInfo.getMessageAllStaffRead())) {
                    messageInfo.setMessageOwner("all");
                    allSeeFlag = Boolean.TRUE;
                } else {
                    messageUserList.addAll(Arrays.asList(messageInfo.getMessageOwner().split(",")));
                    newMessageInfoList.add(messageInfo);
                }
            }
            messageInfos.add(messageInfo);
        }
        if (CollectionUtils.isNotEmpty(messageInfos)) {
            thirdMessageInfoService.saveOrUpdateBatch(messageInfos);
        }

        List<String> tenantIdList = erpSsoInfoService.queryTenantIdByAppId(appId);
        // mqtt通知前端
        if (Boolean.TRUE.equals(allSeeFlag)) {
            MessageDO payload = new MessageDO();
            payload.setTenantId(CollectionUtils.isNotEmpty(tenantIdList) ? tenantIdList.get(0) : "");
            payload.setAllStaffNoticeFlag(Boolean.TRUE);
            payload.setType(2);
            payload.setAppId(appId);
            messageSendService.sendToClient(null, payload, Constants.THIRD_MESSAGE_NOTICE_USER_TOPIC);
            messageSendService.sendToClient(null, payload,Constants.THIRD_MESSAGE_MOBILE_NOTICE_USER_TOPIC);
        } else {
            List<String> distinctList = messageUserList.stream().distinct().collect(Collectors.toList());
            for (String verifyUserId : distinctList) {
                MessageDO payload = new MessageDO();
                payload.setTenantId(CollectionUtils.isNotEmpty(tenantIdList) ? tenantIdList.get(0) : "");
                payload.setUserId(verifyUserId);
                payload.setType(2);
                payload.setAppId(appId);
                messageSendService.sendToClient(verifyUserId, payload, Constants.THIRD_MESSAGE_NOTICE_USER_TOPIC);
                messageSendService.sendToClient(verifyUserId, payload,Constants.THIRD_MESSAGE_MOBILE_NOTICE_USER_TOPIC);
            }
        }

        // 异步发送企微
        if (StringUtils.isNotBlank(sendChannel) && sendChannel.contains(Constants.MessageChannelEnum.WECOM.getFlag()) && CollectionUtils.isNotEmpty(newMessageInfoList)) {
            // 三方应用调用该接口，无法得知token，所以需要获取集成token
            String userToken = iamService.queryIntegrationUserToken(tenantId);
            if (StringUtils.isNotBlank(userToken)) {
                ThirdMessageWecomEventDTO thirdMessageWecomEventDTO = new ThirdMessageWecomEventDTO();
                thirdMessageWecomEventDTO.setNewMessageInfoList(newMessageInfoList);
                thirdMessageWecomEventDTO.setTenantId(tenantId);
                thirdMessageWecomEventDTO.setUserToken(userToken);
                ThirdMessageWecomEvent thirdMessageWecomEvent = new ThirdMessageWecomEvent(this, thirdMessageWecomEventDTO);
                eventPublisher.publishEvent(thirdMessageWecomEvent);
            }
        }
    }

    /**
     * 获取查询待办数据QueryWrapper
     *
     * @return
     */
    public QueryWrapper getMessageWrapepr(String appId, String Id) {
        QueryWrapper queryWrapper = new QueryWrapper<ThirdMessageInfo>();
        //应用查询
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("message_app_id", appId);
        }
        //外部待办唯一id
        if (StringUtils.isNotEmpty(Id)) {
            queryWrapper.eq("message_id", Id);
        }
        return queryWrapper;
    }
}
