package com.digiwin.athena.semc.service.applink.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.semc.util.Utils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErpSsoConstants;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.BizObjectTypeEnum;
import com.digiwin.athena.semc.common.enums.ErpAppNameEnum;
import com.digiwin.athena.semc.common.enums.MaycurCodeEnum;
import com.digiwin.athena.semc.common.enums.PreinstalledApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.ProtocolTypeEnum;
import com.digiwin.athena.semc.common.enums.SsoSourceTypeEnum;
import com.digiwin.athena.semc.dto.erpsso.AuthorizeDTO;
import com.digiwin.athena.semc.dto.erpsso.ErpSsoSpecialDTO;
import com.digiwin.athena.semc.dto.erpsso.GetSsoUrlReq;
import com.digiwin.athena.semc.dto.erpsso.GetSsoUrlResp;
import com.digiwin.athena.semc.dto.erpsso.PreErpSsoDTO;
import com.digiwin.athena.semc.dto.erpsso.QueryDisplayReq;
import com.digiwin.athena.semc.dto.erpsso.ThirdSsoInfoDto;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.applink.AppLinkGroupDTO;
import com.digiwin.athena.semc.entity.applink.AppLinkListDTO;
import com.digiwin.athena.semc.entity.applink.TenantAppDTO;
import com.digiwin.athena.semc.entity.common.BizObjAuthRel;
import com.digiwin.athena.semc.entity.common.BizObjCommonUse;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplicationInstance;
import com.digiwin.athena.semc.entity.sso.ErpSsoInfo;
import com.digiwin.athena.semc.entity.sso.MobileSsoInfo;
import com.digiwin.athena.semc.entity.sso.ThirdSsoInfo;
import com.digiwin.athena.semc.mapper.mongo.AppLinkMongoMapper;
import com.digiwin.athena.semc.mapper.portal.PreinstalledApplicationInstanceMapper;
import com.digiwin.athena.semc.mapper.portal.PreinstalledApplicationMapper;
import com.digiwin.athena.semc.proxy.cac.service.CacService;
import com.digiwin.athena.semc.proxy.eoc.service.EocService;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.maycur.MayCurService;
import com.digiwin.athena.semc.proxy.mdc.MdcService;
import com.digiwin.athena.semc.proxy.mdc.model.TenantProductOperationDTO;
import com.digiwin.athena.semc.proxy.trans.service.TranslateService;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.common.IBizObjAuthRelService;
import com.digiwin.athena.semc.service.portal.IPreinstalledApplicationService;
import com.digiwin.athena.semc.service.sso.IErpSsoInfoService;
import com.digiwin.athena.semc.service.sso.IMobileSsoInfoService;
import com.digiwin.athena.semc.service.sso.IThirdSsoInfoService;
import com.digiwin.athena.semc.util.DateUtils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.Base64Utils;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;
import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;

import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;

/**
 * @date 2022/12/23
 */
@Service
@Slf4j
public class AppLinkServiceImpl implements AppLinkService {

    @Autowired
    private AppLinkMongoMapper appLinkMongoMapper;

    @Resource
    private PreinstalledApplicationInstanceMapper preinstalledApplicationInstanceMapper;

    @Resource
    private IamService iamService;

    @Resource
    private CacService cacService;

    @Resource
    private IPreinstalledApplicationService preinstalledApplicationService;

    @Autowired
    IErpSsoInfoService erpSsoInfoService;

    @Autowired
    IThirdSsoInfoService thirdSsoInfoService;

    @Autowired
    IMobileSsoInfoService mobileSsoInfoService;

    @Resource
    private IBizObjAuthRelService bizObjAuthRelService;

    @Resource
    private TranslateService translateService;

    @Resource
    private PreinstalledApplicationMapper preinstalledApplicationMapper;

    @Autowired
    MayCurService mayCurService;

    @Autowired
    EocService eocService;

    @Autowired
    MdcService mdcService;

    @Resource(name = "semcAsyncExecutor")
    private Executor asyncTaskExecutor;

    private static final String PLATFORM_CODE_ATHENA_LCDP = "athena-lcdp";

    public static final int RESENT_COUNT = 9;

    private static final String SSO_LOGIN_URL = "%s/sso-login?userToken=%s&dwLang=%s";

    /**
     * 记录一次访问友情链接的操作
     *
     * @param appId 应用Id
     */
    @Override
    public void accessLink(String appId) {
        appLinkMongoMapper.access(appId);
    }

    /**
     * 获取友情链接
     *
     * @param user 用户信息
     * @return
     */
    @Override
    public AppLinkListDTO queryLinkList(AuthoredUser user) {
        AppLinkListDTO result = new AppLinkListDTO();
        List<AppLinkGroupDTO> appLinkList = Lists.newArrayList();

        // 获取IAM的应用列表和预设的应用列表，作为一组
        AppLinkGroupDTO defaultGroup = new AppLinkGroupDTO();
        defaultGroup.setGroupId("default");
        defaultGroup.setGroupName("鼎捷应用");
        // IAM应用列表,即鼎捷雅典娜应用
        List<AppLinkDTO> iamAppLinkList = queryDefaultGroup(user);
        // 过滤出已经购买的应用
        List<AppLinkDTO> filteredAppLinkList = filterBSAppList(user.getTenantId(), iamAppLinkList);

        // 预设的应用列表
        List<AppLinkDTO> preAppLinkList = queryPerInstallAppGroup(user);
        filteredAppLinkList.addAll(preAppLinkList);
        defaultGroup.setLinks(filteredAppLinkList);
        appLinkList.add(defaultGroup);

        // 获取手动配置的稳态敏态应用列表
        AppLinkGroupDTO ssoAppGroup = new AppLinkGroupDTO();
        ssoAppGroup.setGroupId("sso");
        ssoAppGroup.setGroupName("其他应用");
        List<AppLinkDTO> ssoAppLinkList = queryStaticAppLinks();
        ssoAppGroup.setLinks(ssoAppLinkList);
        appLinkList.add(ssoAppGroup);

        // 返回的所有应用列表
        result.setAll(appLinkList);

        // 10-20个时，额外呈现用户最近点击使用过的3个链接地址
        List<AppLinkDTO> allAppLinkList = Lists.newArrayList();
        allAppLinkList.addAll(filteredAppLinkList);
        allAppLinkList.addAll(ssoAppLinkList);
        if (allAppLinkList.size() > RESENT_COUNT) {
            Map<String, AppLinkDTO> dic = buildAppMap(allAppLinkList);
            List<Map> latestApps = appLinkMongoMapper.getLatestAccesses();
            List<AppLinkDTO> latest = Lists.newArrayList();
            for (Map app : latestApps) {
                // 最新使用的链接必须存在于IAM中
                if (dic.containsKey(app.get("_id"))) {
                    latest.add(dic.get(app.get("_id")));
                }
            }
            result.setLatest(latest);
        }
        return result;
    }

    @Override
    public List<AppLinkGroupDTO> getAppLinkList(AuthoredUser user) {
        List<AppLinkGroupDTO> appLinkList = Lists.newArrayList();
        // 获取IAM的应用列表和预设的应用列表，作为一组
        AppLinkGroupDTO defaultGroup = new AppLinkGroupDTO();
        defaultGroup.setGroupId("default");
        defaultGroup.setGroupName("鼎捷应用");
        // IAM应用列表,即鼎捷雅典娜应用
        List<AppLinkDTO> iamAppLinkList = queryDefaultGroup(user);
        // 过滤出已经购买的应用
        List<AppLinkDTO> filteredAppLinkList = filterBSAppList(user.getTenantId(), iamAppLinkList);

        // 预设的应用列表
        List<AppLinkDTO> preAppLinkList = queryPerInstallAppGroup(user);
        filteredAppLinkList.addAll(preAppLinkList);
        defaultGroup.setLinks(filteredAppLinkList);
        appLinkList.add(defaultGroup);

        // 获取手动配置的稳态敏态应用列表
        AppLinkGroupDTO ssoAppGroup = new AppLinkGroupDTO();
        ssoAppGroup.setGroupId("sso");
        ssoAppGroup.setGroupName("其他应用");
        List<AppLinkDTO> ssoAppLinkList = queryStaticAppLinks();
        ssoAppGroup.setLinks(ssoAppLinkList);
        appLinkList.add(ssoAppGroup);

        return appLinkList;
    }

    /**
     * 首页，查询友情链接全部应用列表，返回当前登录用户权限内可查看的列表
     *
     * @param user 用户信息
     * @return
     */
    @Override
    public List<AppLinkDTO> queryDisplayList(AuthoredUser user) {
        List<AppLinkDTO> appList = Lists.newArrayList();
        // 查询当前用户 + 当前角色 + 当前组织的所有应用
        List<BizObjAuthRel> bizObjAuthRelList = bizObjAuthRelService.qryBizObjAuthRel(BizObjectTypeEnum.APPlICATION_MANAGEMENT.getValue());
        if (CollectionUtils.isEmpty(bizObjAuthRelList)) {
            return appList;
        }

        // b/s和c/s应用配置
        List<AppLinkDTO> ssoAppList = queryStaticAppLinks();
        // 预设的应用列表
        List<AppLinkDTO> preAppList = queryPerInstallAppGroup(user);
        // 获取IAM应用列表,即鼎捷雅典娜应用
        List<AppLinkDTO> iamAppLinkList = queryDefaultGroup(user);
        // 过滤出已经购买的应用
        List<AppLinkDTO> filteredAppLinkList = filterBSAppList(user.getTenantId(), iamAppLinkList);
        ssoAppList.addAll(preAppList);
        ssoAppList.addAll(filteredAppLinkList);

        // 根据应用子类型,应用主键id分类
        Map<String, Map<String, AppLinkDTO>> typeAppMap = ssoAppList.stream().collect(Collectors.groupingBy(AppLinkDTO::getType, Collectors.toMap(AppLinkDTO::getPrimaryId, Function.identity())));
        // 构建返回应用列表，按照应用类型 + 应用id先去重
        bizObjAuthRelList = bizObjAuthRelList.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(
                () -> new TreeSet<>(Comparator.comparing(BizObjAuthRel::getBizObjSubType).thenComparing(BizObjAuthRel::getBizObjId))), ArrayList::new)
        );
        bizObjAuthRelList.forEach(x -> {
            Map<String, AppLinkDTO> idMap = typeAppMap.get(String.valueOf(x.getBizObjSubType()));
            if (MapUtils.isNotEmpty(idMap) && idMap.get(x.getBizObjId()) != null) {
                AppLinkDTO appLinkDTO = idMap.get(x.getBizObjId());
                appList.add(appLinkDTO);
            }
        });

        // 名称翻译与排序
        for (AppLinkDTO appLinkDTO : appList) {
            appLinkDTO.setName(translateService.translateTextCache(appLinkDTO.getName(), StringUtils.EMPTY));
        }
        appList.sort(Comparator.comparing(AppLinkDTO::getName));
        return appList;
    }

    @Override
    public AppLinkDTO queryDisplay(AuthoredUser user, QueryDisplayReq req) throws Exception {
        return queryStaticAppLink(user, req);
    }

    /**
     * 首页友情链接，通过应用id查询应用列表
     *
     * @param commonUseList 常用列表id
     * @return
     */
    @Override
    public List<AppLinkDTO> queryCommonList(List<BizObjCommonUse> commonUseList, AuthoredUser user) {
        List<AppLinkDTO> appList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(commonUseList)) {
            return appList;
        }
        // b/s和c/s应用配置
        List<AppLinkDTO> ssoAppList = queryStaticAppLinks();
        // 预设的应用列表
        List<AppLinkDTO> preAppList = queryPerInstallAppGroup(user);
        // 获取租户购买的应用列表，即鼎捷雅典娜应用
        List<AppLinkDTO> iamAppLinkList = queryDefaultGroup(user);
        ssoAppList.addAll(preAppList);
        ssoAppList.addAll(iamAppLinkList);

        // 当前登录用户有权限看到的应用
        List<BizObjAuthRel> bizObjAuthRelList = bizObjAuthRelService.qryBizObjAuthRel(BizObjectTypeEnum.APPlICATION_MANAGEMENT.getValue());
        List<String> appIdList = bizObjAuthRelList.stream().map(BizObjAuthRel::getBizObjId).collect(Collectors.toList());
        commonUseList = commonUseList.stream().filter(x -> appIdList.contains(x.getBizObjId())).collect(Collectors.toList());

        // 根据应用子类型,应用主键id分类
        Map<String, Map<String, AppLinkDTO>> typeAppMap = ssoAppList.stream().collect(Collectors.groupingBy(AppLinkDTO::getType, Collectors.toMap(AppLinkDTO::getPrimaryId, Function.identity())));
        commonUseList.forEach(x -> {
            Map<String, AppLinkDTO> idMap = typeAppMap.get(String.valueOf(x.getBizObjSubType()));
            if (MapUtils.isNotEmpty(idMap) && idMap.get(String.valueOf(x.getBizObjId())) != null) {
                AppLinkDTO appLinkDTO = idMap.get(x.getBizObjId());
                appList.add(appLinkDTO);
            }
        });
        // 名称翻译
        for (AppLinkDTO appLinkDTO : appList) {
            appLinkDTO.setName(translateService.translateTextCache(appLinkDTO.getName(), StringUtils.EMPTY));
        }
        return appList;
    }

    /**
     * 管理后台，查询所有b/s应用、c/s应用、预设应用
     *
     * @param user 用户信息
     * @return
     */
    @Override
    public List<AppLinkDTO> queryManageList(AuthoredUser user) {
        // b/s和c/s应用配置
        List<AppLinkDTO> ssoAppList = queryStaticAppLinks();
        // 预设的应用列表
        List<AppLinkDTO> preAppList = queryPerInstallAppGroup(user);
        // 获取IAM应用列表,即鼎捷雅典娜应用
        List<AppLinkDTO> iamAppLinkList = queryDefaultGroup(user);
        // 过滤出已经购买的应用
        List<AppLinkDTO> filteredAppLinkList = filterBSAppList(user.getTenantId(), iamAppLinkList);
        ssoAppList.addAll(preAppList);
        ssoAppList.addAll(filteredAppLinkList);
        // 名称翻译与排序
        // ssoAppList.forEach(x -> x.setName(translateService.translateTextCache(x.getName(), StringUtils.EMPTY)));
        ssoAppList.sort(Comparator.comparing(AppLinkDTO::getName));
        return ssoAppList;
    }

    /**
     * 管理后台，根据权限id查询有权限的应用列表
     *
     * @param authId 权限id
     * @return
     */
    @Override
    public List<AppLinkDTO> queryAuthorizedList(String authId) {
        // 获取当前权限对象的应用列表
        List<AppLinkDTO> authAppList = Lists.newArrayList();
        QueryWrapper<BizObjAuthRel> condition = new QueryWrapper<BizObjAuthRel>()
                .eq("biz_obj_type", BizObjectTypeEnum.APPlICATION_MANAGEMENT.getValue()).eq("auth_id", authId);
        List<BizObjAuthRel> bizObjAuthRelList = bizObjAuthRelService.getBaseMapper().selectList(condition);

        // 返回
        if (CollectionUtils.isEmpty(bizObjAuthRelList)) {
            return authAppList;
        }
        authAppList = bizObjAuthRelList.stream().map(x -> {
            AppLinkDTO appLinkDTO = new AppLinkDTO();
            appLinkDTO.setPrimaryId(x.getBizObjId());
            appLinkDTO.setType(String.valueOf(x.getBizObjSubType()));
            return appLinkDTO;
        }).collect(Collectors.toList());
        return authAppList;
    }

    /**
     * 调用鼎捷云接口判断是否购买
     *
     * @param tenantId       租户id
     * @param iamAppLinkList 应用列表
     * @return
     */
    private List<AppLinkDTO> filterBSAppList(String tenantId, List<AppLinkDTO> iamAppLinkList) {
        if (StringUtils.isBlank(tenantId) || CollectionUtils.isEmpty(iamAppLinkList)) {
            return iamAppLinkList;
        }
        Map<String, AppLinkDTO> iamAppLinkMap = iamAppLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getId, Function.identity(), (a, b) -> a));
        CompletableFuture[] completableFutures = new CompletableFuture[iamAppLinkList.size()];
        List<AppLinkDTO> appLinkList = Lists.newArrayList();
        for (int i = 0; i < iamAppLinkList.size(); i++) {
            AppLinkDTO x = iamAppLinkList.get(i);
            completableFutures[i] = CompletableFuture.supplyAsync(() -> cacService.verifyAppAuthorizedResult(tenantId, x.getId()), asyncTaskExecutor);
        }
        CompletableFuture.allOf(completableFutures).join();
        for (CompletableFuture<Tuple2<String, Boolean>> completableFuture : completableFutures) {
            try {
                Tuple2<String, Boolean> tuple2 = completableFuture.get();
                if (Boolean.TRUE.equals(tuple2._2)) {
                    appLinkList.add(iamAppLinkMap.get(tuple2._1()));
                }
            } catch (Exception e) {
                log.error("filterBSAppListEx：{}", e.getMessage(), e);
            }
        }
        return appLinkList;
    }

    /**
     * 获取租户购买的应用列表，即鼎捷雅典娜应用
     *
     * @return
     */
    @Override
    public List<AppLinkDTO> queryDefaultGroup(AuthoredUser user) {
        AppAuthContextHolder.getContext().setAuthoredUser(user);
        List<AppLinkDTO> iamAppLinkList = Lists.newArrayList();
        List<TenantAppDTO> allTenantApps = iamService.queryTenantApps();
        for (TenantAppDTO tenantApp : allTenantApps) {
            AppLinkDTO appLink = new AppLinkDTO();
            appLink.setPrimaryId(tenantApp.getId());
            appLink.setType(PreinstalledApplicationTypeEnum.IAM_APPLICATION.getType());
            appLink.setSystemType(ApplicationTypeEnum.BS_APPLICATION.getType());
            appLink.setDataSource(ApplicationTypeEnum.IAM_APPLICATION.getType());
            appLink.setId(tenantApp.getId());
            appLink.setCode(tenantApp.getId());
            appLink.setName(tenantApp.getName());
            appLink.setCloudwebsite(tenantApp.getCloudwebsite());
            appLink.setNeedAuthCodeFlag(Constants.NEED_AUTH_CODE_FLAG_NO);
            // 如果关联是athena-lcdp，设定AppToken，由前端拼接url
            if (tenantApp.getPlatformCodes() != null && tenantApp.getPlatformCodes().contains(PLATFORM_CODE_ATHENA_LCDP)) {
                appLink.setAppToken(tenantApp.getAppToken());
            }
            // 特殊处理：IAM同账套体系应用，需要拼接userToken，实现自动登录
            appLink.setCloudwebsite(tenantApp.getCloudwebsite() + "/sso-login?userToken=" + AppAuthContextHolder.getContext().getAuthoredUser().getToken());
            iamAppLinkList.add(appLink);
        }
        return iamAppLinkList;
    }

    @Override
    public AppLinkDTO getLinkSsoUrl(AppLinkDTO req) {
        AppLinkDTO appLinkDTO = new AppLinkDTO();

        PreinstalledApplication preinstalled = preinstalledApplicationMapper.selectById(req.getPrimaryId());
        if (null == preinstalled) {
            return appLinkDTO;
        }
        //如果是每刻云票和每刻报销
        if (MaycurCodeEnum.MAYCUR_CLOUD.getType().equals(preinstalled.getApplicationCode()) || MaycurCodeEnum.MAYCUR_APPROVALS.getType().equals(preinstalled.getApplicationCode())) {
            PreinstalledApplicationInstance preinstalledIns = preinstalledApplicationService.getPreINstance(Long.parseLong(req.getPrimaryId()), "", "");
            if (null == preinstalledIns || null == preinstalledIns.getApplicationExtConfig()) {
                return appLinkDTO;
            }
            //取回调地址
            String callBackUrl = preinstalledIns.getApplicationExtConfig().getString("callBackUrl");
            String appId = preinstalledIns.getApplicationExtConfig().getString("appId");
            String appSecret = preinstalledIns.getApplicationExtConfig().getString("appSecret");
            //获取工号
            String userId = eocService.getEmpJobNum();
            if (StringUtils.isEmpty(userId)) {
                return appLinkDTO;
            }
            long timestamp = System.currentTimeMillis();
            //获取ssoToken
            String ssoToken = mayCurService.getSsoToken(userId, appSecret, timestamp);
            String ssoUrl = mayCurService.getHomeSsoUrl(callBackUrl, appId, userId, ssoToken, timestamp);
            appLinkDTO.setCloudwebsite(ssoUrl);
        }
        return appLinkDTO;
    }

    @Override
    public List<AppLinkDTO> querySsoList(AuthoredUser user) {
        // b/s和c/s应用配置
        List<AppLinkDTO> ssoAppList = queryStaticAppLinks();
        // 预设的应用列表
        List<AppLinkDTO> preAppList = queryPerInstallAppGroup(user);
        ssoAppList.addAll(preAppList);
        ssoAppList.sort(Comparator.comparing(AppLinkDTO::getName));
        return ssoAppList;
    }

    /**
     * 获取预设的应用列表
     *
     * @return
     */
    @Override
    public List<AppLinkDTO> queryPerInstallAppGroup(AuthoredUser user) {
        AppAuthContextHolder.getContext().setAuthoredUser(user);
        List<AppLinkDTO> preInstallAppLinkList = Lists.newArrayList();
        List<PreinstalledApplication> preInstalledAppList = preinstalledApplicationService.queryPreinstalledAppList(user);
        if (CollectionUtils.isEmpty(preInstalledAppList)) {
            return preInstallAppLinkList;
        }
        Map<String, List<PreinstalledApplication>> preinstalledApplicationMap = preInstalledAppList.stream()
                .collect(Collectors.groupingBy(PreinstalledApplication::getApplicationType));

        // 构建C/S应用链接
        List<AppLinkDTO> csSsoLinkList = buildCsAppLink(preinstalledApplicationMap);
        // 构建B/S应用链接
        List<AppLinkDTO> bsSsoLinkList = buildBsAppLink(preinstalledApplicationMap);
        // 构建特殊类型的应用链接
        List<AppLinkDTO> t100LinkList = buildSpecialAppLink(user, preinstalledApplicationMap);
        preInstallAppLinkList.addAll(csSsoLinkList);
        preInstallAppLinkList.addAll(bsSsoLinkList);
        preInstallAppLinkList.addAll(t100LinkList);
        preInstallAppLinkList.sort(Comparator.comparing(AppLinkDTO::getName));
        return preInstallAppLinkList;
    }

    /**
     * 组装T100的应用
     *
     * @param user                       用户信息
     * @param preinstalledApplicationMap 应用信息
     * @return
     */
    private List<AppLinkDTO> buildSpecialAppLink(AuthoredUser user, Map<String, List<PreinstalledApplication>> preinstalledApplicationMap) {
        List<PreinstalledApplication> specialPreAppList = preinstalledApplicationMap.get(PreinstalledApplicationTypeEnum.SPECIAL_APPLICATION.getValue());
        if (CollectionUtils.isEmpty(specialPreAppList)) {
            return Lists.newArrayList();
        }
        // 查询应用的动态参数
        List<Long> idList = specialPreAppList.stream().map(PreinstalledApplication::getId).collect(Collectors.toList());
        QueryWrapper<PreinstalledApplicationInstance> queryWrapper = new QueryWrapper<>();
        queryWrapper.in("preinstalled_application_id", idList);
        queryWrapper.eq("tenant_id", user.getTenantId());
        List<PreinstalledApplicationInstance> applicationInstanceList = preinstalledApplicationInstanceMapper.selectList(queryWrapper);
        applicationInstanceList = applicationInstanceList.stream().filter(x -> ObjectUtils.isNotEmpty(x.getApplicationExtConfig())).collect(Collectors.toList());
        Map<Long, net.sf.json.JSONObject> id2ConfigMap = applicationInstanceList.stream().collect(Collectors.toMap(PreinstalledApplicationInstance::getPreinstalledApplicationId, PreinstalledApplicationInstance::getApplicationExtConfig));

        List<AppLinkDTO> specialAppLinkList = Lists.newArrayList();
        // T100应用
        List<AppLinkDTO> T100AppLinkList = parseT100AppLink(specialPreAppList, id2ConfigMap);
        specialAppLinkList.addAll(T100AppLinkList);
        // TIPTOP应用
        List<AppLinkDTO> TTLinkAppList = parseTTAppLink(specialPreAppList, id2ConfigMap);
        specialAppLinkList.addAll(TTLinkAppList);
        return specialAppLinkList;
    }

    /**
     * 构建T100应用链接
     *
     * @param specialPreAppList 应用列表
     * @param id2ConfigMap      应用的动态配置
     * @return
     */
    private List<AppLinkDTO> parseT100AppLink(List<PreinstalledApplication> specialPreAppList, Map<Long, net.sf.json.JSONObject> id2ConfigMap) {
        // T100应用
        List<PreinstalledApplication> T100AppList = specialPreAppList.stream().filter(x -> Constants.T100_PRODUCT_TYPE.equals(x.getProductCode())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(T100AppList)) {
            return Lists.newArrayList();
        }
        return T100AppList.stream().map(x -> {
            AppLinkDTO appLinkDTO = new AppLinkDTO();
            appLinkDTO.setProductCode(x.getProductCode());
            appLinkDTO.setPrimaryId(String.valueOf(x.getId()));
            appLinkDTO.setId(Constants.CODE_PREFFIX_WT + x.getApplicationName());
            appLinkDTO.setCode(x.getApplicationCode());
            appLinkDTO.setName(x.getApplicationName());
            appLinkDTO.setAppId(x.getAppId());
            appLinkDTO.setType(PreinstalledApplicationTypeEnum.SPECIAL_APPLICATION.getValue());
            appLinkDTO.setSystemType(x.getSystemType());
            appLinkDTO.setDataSource(ApplicationTypeEnum.PRESET_APPLICATION.getType());
            appLinkDTO.setUserBindFlag(x.getUserBindFlag());
            appLinkDTO.setApplicationAppId(x.getAppId());
            appLinkDTO.setProtocolType(x.getProtocolType());
            appLinkDTO.setNeedAuthCodeFlag(x.getNeedAuthCodeFlag());
            // 拼接T100的链接
            ErpSsoSpecialDTO erpSsoSpecialDTO = Optional.ofNullable(JSONObject.parseObject(x.getApplicationConfig(), ErpSsoSpecialDTO.class)).orElse(new ErpSsoSpecialDTO());
            appLinkDTO.setAppToken(erpSsoSpecialDTO.getAppToken());
            net.sf.json.JSONObject jsonObject = id2ConfigMap.get(x.getId());
            String protocolHeader = Optional.ofNullable(jsonObject).filter(object -> object.containsKey("protocolHeader")).map(object -> object.getString("protocolHeader")).orElse("");
            String ip = Optional.ofNullable(jsonObject).filter(object -> object.containsKey("ip")).map(object -> object.getString("ip")).orElse("");
            String companyCode = Optional.ofNullable(jsonObject).filter(object -> object.containsKey("companyCode")).map(object -> object.getString("companyCode")).orElse("");

            // 租户配置的区域别（erpSsoSpecialDTO.getEnvironment()是兜底数据）
            String environment = Optional.ofNullable(jsonObject).filter(object -> object.containsKey("environment")).map(object -> object.getString("environment")).orElse(erpSsoSpecialDTO.getEnvironment());
            environment = StringUtils.isNotBlank(environment) ? "/w" + environment : "";

            String linkStr = protocolHeader + ip + environment + "/wa/r/app/gdc_azzi000?Arg=" + companyCode + "&Arg=" + erpSsoSpecialDTO.getArg() + "&Arg=QVRIRU5B&Arg=";

            // 查询云端账号绑定的地端员工账号
            String loginUserId = queryVerifyUserId(x.getApplicationCode());
            // 参数hashKey，时间戳 + trust key + 登录账号，base64加密
            String hashKey = DateUtils.getNowTime(DateUtils.DATE_TIME_FORMATTER) + Constants.T100_TRUST_KEY + loginUserId;
            String encoderHashKey = Base64.getEncoder().encodeToString(hashKey.getBytes(StandardCharsets.UTF_8));
            encoderHashKey = encoderHashKey.replaceAll("\\+", "_plus_");
            String fullLink = linkStr + encoderHashKey;
            appLinkDTO.setCloudwebsite(fullLink);
            appLinkDTO.setDesc(x.getApplicationDescription());
            return appLinkDTO;
        }).collect(Collectors.toList());
    }

    /**
     * 查询云端账号查询对应的稳态地端用户账号
     *
     * @param appId appId
     * @return
     */
    private String queryVerifyUserId(String appId) {
        String empId = iamService.queryMappingEmpId(appId);
        empId = StringUtils.isNotBlank(empId) ? empId : "";
        return empId;
    }

    /**
     * 构建TIPTOP应用链接
     *
     * @param specialPreAppList 应用列表
     * @param id2ConfigMap      应用的动态配置
     * @return
     */
    private List<AppLinkDTO> parseTTAppLink(List<PreinstalledApplication> specialPreAppList, Map<Long, net.sf.json.JSONObject> id2ConfigMap) {
        List<PreinstalledApplication> TTAppList = specialPreAppList.stream().filter(x -> Constants.TIPTOP_PRODUCT_TYPE.equals(x.getProductCode())).collect(Collectors.toList());
        List<AppLinkDTO> TTLinkAppList = TTAppList.stream().map(x -> {
            AppLinkDTO appLinkDTO = new AppLinkDTO();
            appLinkDTO.setProductCode(x.getProductCode());
            appLinkDTO.setPrimaryId(String.valueOf(x.getId()));
            appLinkDTO.setId(Constants.CODE_PREFFIX_WT + x.getApplicationName());
            appLinkDTO.setCode(x.getApplicationCode());
            appLinkDTO.setName(x.getApplicationName());
            appLinkDTO.setAppId(x.getAppId());
            appLinkDTO.setType(PreinstalledApplicationTypeEnum.SPECIAL_APPLICATION.getValue());
            appLinkDTO.setSystemType(x.getSystemType());
            appLinkDTO.setDataSource(ApplicationTypeEnum.PRESET_APPLICATION.getType());
            appLinkDTO.setUserBindFlag(x.getUserBindFlag());
            appLinkDTO.setApplicationAppId(x.getAppId());
            appLinkDTO.setProtocolType(x.getProtocolType());
            appLinkDTO.setNeedAuthCodeFlag(x.getNeedAuthCodeFlag());
            // 拼接TT的链接
            ErpSsoSpecialDTO erpSsoSpecialDTO = Optional.ofNullable(JSONObject.parseObject(x.getApplicationConfig(), ErpSsoSpecialDTO.class)).orElse(new ErpSsoSpecialDTO());
            appLinkDTO.setAppToken(erpSsoSpecialDTO.getAppToken());
            net.sf.json.JSONObject jsonObject = id2ConfigMap.get(x.getId());
            String ip = Optional.ofNullable(jsonObject).map(object -> object.getString("ip")).orElse("");
            // 查询云端账号绑定的地端员工账号
            String loginUserId = queryVerifyUserId(x.getApplicationCode());
            // 获取tokenKey
            String tokenKey = encryptionTokenKey(erpSsoSpecialDTO.getSystemCode(), loginUserId);
            String linkStr = erpSsoSpecialDTO.getProtocolHeader() + ip + "?Arg=" + erpSsoSpecialDTO.getArg() + "&Arg="
                    + AppAuthContextHolder.getContext().getAuthoredUser().getUserId() + "&Arg=" + tokenKey + "&Arg=_blank_&Arg=" + erpSsoSpecialDTO.getJobCode() + "&Arg=" + LocaleContextHolder.getLocale();
            appLinkDTO.setCloudwebsite(linkStr);
            appLinkDTO.setDesc(x.getApplicationDescription());
            return appLinkDTO;
        }).collect(Collectors.toList());
        return TTLinkAppList;
    }

    /**
     * 获取tokenKey
     *
     * @param systemCode       平台系统编号
     * @param outerLoginUserId 外部系统账号
     * @return
     */
    public String encryptionTokenKey(String systemCode, String outerLoginUserId) {
        String encode;
        final String base64String = "28682266";
        try {
            JSONObject info = new JSONObject();
            info.put("System", systemCode);    // 平台系统编號 ex:ATHENA
            info.put("Account", outerLoginUserId); // 外部系统帳號 ex:tiptop
            info.put("Date", DateUtils.getNowTime(null));     // 時間搓記
            String token = info.toString();

            // 加密演算法BlowFish
            byte[] keyData = Base64Utils.encode(base64String.getBytes());
            SecretKeySpec spec = new SecretKeySpec(keyData, Constants.BLOW_FISH);
            Cipher cipher = Cipher.getInstance(Constants.BLOW_FISH);
            cipher.init(Cipher.ENCRYPT_MODE, spec);

            // 開始加密
            byte[] hasil = cipher.doFinal(token.getBytes());
            encode = new String(Base64Utils.encode(hasil));
            encode = encode.replaceAll("\\+", "_plus_");
        } catch (Exception e) {
            log.error("encrypt token key occur error", e);
            return null;
        }
        return encode;
    }

    /**
     * 获取动态参数
     *
     * @param applicationId
     * @return
     */
    public String getPreInstanceCallback(Long applicationId) {
        // 查询应用的动态参数
        PreinstalledApplicationInstance applicationInstance = preinstalledApplicationService.getPreINstance(applicationId, "", "");
        if (null == applicationInstance) {
            return "";
        }
        net.sf.json.JSONObject jsonObject = applicationInstance.getApplicationExtConfig();
        String callBackUrl = jsonObject == null ? "" : jsonObject.getString("callBackUrl");
        return callBackUrl;
    }


    /**
     * 组装B/S应用
     *
     * @param preinstalledApplicationMap 应用信息
     * @return
     */
    private List<AppLinkDTO> buildBsAppLink(Map<String, List<PreinstalledApplication>> preinstalledApplicationMap) {
        List<PreinstalledApplication> bsPreAppList = preinstalledApplicationMap.get(PreinstalledApplicationTypeEnum.BS_APPLICATION.getValue());
        if (CollectionUtils.isEmpty(bsPreAppList)) {
            return Lists.newArrayList();
        }
        return bsPreAppList.stream().map(x -> {
            AppLinkDTO appLinkDTO = new AppLinkDTO();
            appLinkDTO.setCasServerUrl(x.getCasServerUrl());
            appLinkDTO.setProtocolType(x.getProtocolType());
            appLinkDTO.setAppSecret(x.getAppSecret());
            //每刻报销和每刻云票
            if (MaycurCodeEnum.MAYCUR_CLOUD.getType().equals(x.getApplicationCode()) ||
                    MaycurCodeEnum.MAYCUR_APPROVALS.getType().equals(x.getApplicationCode())) {
                appLinkDTO.setIsNeedSsoUrl(1);
                //取回调地址
                String callBackUrl = getPreInstanceCallback(x.getId());
                appLinkDTO.setCallBackUrl(callBackUrl);
            }
            if (ProtocolTypeEnum.OAUTH.getType().equals(x.getProtocolType())) {
                appLinkDTO.setPrimaryId(String.valueOf(x.getId()));
                appLinkDTO.setId(Constants.CODE_PREFFIX_MT + x.getApplicationName());
                appLinkDTO.setCode(x.getApplicationCode());
                appLinkDTO.setName(x.getApplicationName());
                appLinkDTO.setAppId(x.getAppId());
                appLinkDTO.setType(PreinstalledApplicationTypeEnum.BS_APPLICATION.getValue());
                appLinkDTO.setSystemType(x.getSystemType());
                appLinkDTO.setDataSource(ApplicationTypeEnum.PRESET_APPLICATION.getType());
                appLinkDTO.setUserBindFlag(x.getUserBindFlag());
                appLinkDTO.setApplicationAppId(x.getAppId());
                ThirdSsoInfoDto thirdSsoInfoDto = Optional.ofNullable(JSON.toJavaObject(JSONObject.parseObject(x.getApplicationConfig()), ThirdSsoInfoDto.class)).orElse(new ThirdSsoInfoDto());
                appLinkDTO.setAppToken(thirdSsoInfoDto.getAppToken());
                appLinkDTO.setCallBackUrl(thirdSsoInfoDto.getCallBackUrl());
                if (Arrays.asList(Constants.BindFlagEnum.AUTO.getFlag(), Constants.BindFlagEnum.HAND.getFlag()).contains(x.getUserBindFlag())) {
                    appLinkDTO.setNeedAuthCodeFlag(x.getNeedAuthCodeFlag());
                }
                String linkStr;
                // 鼎捷云账套体系应用只要传userToken
                if (Constants.BindFlagEnum.NO.getFlag().equals(x.getUserBindFlag())) {
                    linkStr = StringUtils.stripToEmpty(thirdSsoInfoDto.getCallBackUrl()) + "?userToken=" + AppAuthContextHolder.getContext().getAuthoredUser().getToken();
                    // 定制化需求
                    if (x.getApplicationName().equals("开发平台")) {
                        linkStr = linkStr + "&routerLink=/home";
                    }
                } else {
                    linkStr = StringUtils.stripToEmpty(thirdSsoInfoDto.getCallBackUrl()) +
                            (thirdSsoInfoDto.getCallBackUrl().contains("?") ? "&" : "?") +
                            "appToken=" + thirdSsoInfoDto.getAppToken() +
                            "&tenantId=" + AppAuthContextHolder.getContext().getAuthoredUser().getTenantId() +
                            "&appCode=" + x.getApplicationCode() +
                            "&appId=" + thirdSsoInfoDto.getAppId() +
                            "&bindMode=" + x.getUserBindFlag();
                }
                appLinkDTO.setCloudwebsite(linkStr);
            }
            if (ProtocolTypeEnum.CAS.getType().equals(x.getProtocolType())) {
                appLinkDTO.setPrimaryId(String.valueOf(x.getId()));
                appLinkDTO.setId(Constants.CODE_PREFFIX_MT + x.getApplicationName());
                appLinkDTO.setCode(x.getApplicationCode());
                appLinkDTO.setName(x.getApplicationName());
                appLinkDTO.setType(PreinstalledApplicationTypeEnum.BS_APPLICATION.getValue());
                appLinkDTO.setApplicationAppId(x.getAppId());
                ThirdSsoInfoDto thirdSsoInfoDto = Optional.ofNullable(JSON.toJavaObject(JSONObject.parseObject(x.getApplicationConfig()), ThirdSsoInfoDto.class)).orElse(new ThirdSsoInfoDto());
                appLinkDTO.setAppToken(thirdSsoInfoDto.getAppToken());
                appLinkDTO.setCallBackUrl(thirdSsoInfoDto.getCallBackUrl());
                appLinkDTO.setNeedAuthCodeFlag(x.getNeedAuthCodeFlag());
                appLinkDTO.setCloudwebsite(thirdSsoInfoDto.getCallBackUrl());
            }
            if (ProtocolTypeEnum.SAML.getType().equals(x.getProtocolType())) {
                appLinkDTO.setPrimaryId(String.valueOf(x.getId()));
                appLinkDTO.setId(Constants.CODE_PREFFIX_MT + x.getApplicationName());
                appLinkDTO.setCode(x.getApplicationCode());
                appLinkDTO.setName(x.getApplicationName());
                appLinkDTO.setType(PreinstalledApplicationTypeEnum.BS_APPLICATION.getValue());
                appLinkDTO.setSystemType(x.getSystemType());
                appLinkDTO.setDataSource(ApplicationTypeEnum.PRESET_APPLICATION.getType());
                ThirdSsoInfoDto thirdSsoInfoDto = Optional.ofNullable(JSON.toJavaObject(JSONObject.parseObject(x.getApplicationConfig()), ThirdSsoInfoDto.class)).orElse(new ThirdSsoInfoDto());
                appLinkDTO.setCallBackUrl(thirdSsoInfoDto.getCallBackUrl());
                appLinkDTO.setCloudwebsite(thirdSsoInfoDto.getCallBackUrl());
            }
            appLinkDTO.setDesc(x.getApplicationDescription());
            return appLinkDTO;
        }).collect(Collectors.toList());
    }

    /**
     * 组装C/S应用
     *
     * @param preinstalledApplicationMap 应用信息
     * @return
     */
    private List<AppLinkDTO> buildCsAppLink(Map<String, List<PreinstalledApplication>> preinstalledApplicationMap) {
        List<PreinstalledApplication> csPreAppList = preinstalledApplicationMap.get(PreinstalledApplicationTypeEnum.CS_APPLICATION.getValue());
        if (CollectionUtils.isEmpty(csPreAppList)) {
            return Lists.newArrayList();
        }
        return csPreAppList.stream().map(x -> {
            AppLinkDTO appLinkDTO = new AppLinkDTO();
            if (x.getProtocolType().equals(1)) {
                appLinkDTO.setPrimaryId(String.valueOf(x.getId()));
                appLinkDTO.setId(Constants.CODE_PREFFIX_WT + x.getApplicationName());
                appLinkDTO.setCode(x.getApplicationCode());
                appLinkDTO.setName(x.getApplicationName());
                appLinkDTO.setAppId(x.getAppId());
                appLinkDTO.setType(PreinstalledApplicationTypeEnum.CS_APPLICATION.getValue());
                appLinkDTO.setSystemType(x.getSystemType());
                appLinkDTO.setDataSource(ApplicationTypeEnum.PRESET_APPLICATION.getType());
                appLinkDTO.setUserBindFlag(x.getUserBindFlag());
                appLinkDTO.setNeedAuthCodeFlag(x.getNeedAuthCodeFlag());
                PreErpSsoDTO preErpSsoDTO = Optional.ofNullable(JSON.toJavaObject(JSONObject.parseObject(x.getApplicationConfig()), PreErpSsoDTO.class)).orElse(new PreErpSsoDTO());
                appLinkDTO.setAppToken(preErpSsoDTO.getAppToken());
                // 拼接链接
                StringBuilder linkSb = new StringBuilder();
                linkSb.append(StringUtils.stripToEmpty(preErpSsoDTO.getProtocolHeader())).append(StringUtils.stripToEmpty(preErpSsoDTO.getClientConfiguration()));
                if (preErpSsoDTO.getGetClientMode() != null && Constants.PRESCRIBED_ROUTE.equals(preErpSsoDTO.getGetClientMode())) {
                    linkSb.append("?");
                }
                // 拼接链接参数
                StringBuilder linkParamSb = new StringBuilder();
                linkParamSb.append("appToken").append("=").append(preErpSsoDTO.getAppToken()).append("&");
                linkParamSb.append("appCode").append("=").append(x.getApplicationCode()).append("&");
                if (Constants.BindFlagEnum.NO.getFlag().equals(x.getUserBindFlag()) || StringUtils.isBlank(preErpSsoDTO.getAppId())) { // 兼容已集成的应用
                    linkParamSb.append("userToken").append("=").append(AppAuthContextHolder.getContext().getAuthoredUser().getToken());
                } else {
                    linkParamSb.append("tenantId").append("=").append(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId()).append("&");
                    linkParamSb.append("callBackUrl").append("=").append(String.format(Constants.CS_APP_CALLBACK_URL, preErpSsoDTO.getApplicationCode())).append("&")
                            .append("appId").append("=").append(preErpSsoDTO.getAppId()).append("&").append("bindMode").append("=").append(x.getUserBindFlag());
                }
                JSONObject dynamicParameter = JSON.parseObject(preErpSsoDTO.getDynamicParameter());
                Optional.ofNullable(dynamicParameter).ifPresent(param -> param.forEach((k, v) -> {
                    if (k != null && v != null) {
                        linkParamSb.append("&").append(k).append("=").append(v);
                    }
                }));
                // 参数是否base64加密
                String linkParamStr = linkParamSb.toString();
                if (Constants.ERP_URL_NEEDBASE64.equals(preErpSsoDTO.getParamEncryptionMethod())) {
                    linkParamStr = Base64.getEncoder().encodeToString(linkParamStr.getBytes(StandardCharsets.UTF_8));

                }
                String fullLinkStr = linkSb + linkParamStr;
                appLinkDTO.setCloudwebsite(fullLinkStr);
                appLinkDTO.setProtocolType(x.getProtocolType());
                appLinkDTO.setCasServerUrl(x.getCasServerUrl());
                appLinkDTO.setCallBackUrl(String.format(Constants.CS_APP_CALLBACK_URL, preErpSsoDTO.getApplicationCode()));
                appLinkDTO.setApplicationAppId(x.getAppId());
            }
            if (x.getProtocolType().equals(2)) {
                appLinkDTO.setPrimaryId(String.valueOf(x.getId()));
                appLinkDTO.setId(Constants.CODE_PREFFIX_WT + x.getApplicationName());
                appLinkDTO.setCode(x.getApplicationCode());
                appLinkDTO.setName(x.getApplicationName());
                appLinkDTO.setAppId(x.getAppId());
                appLinkDTO.setType(PreinstalledApplicationTypeEnum.CS_APPLICATION.getValue());
                appLinkDTO.setSystemType(x.getSystemType());
                appLinkDTO.setDataSource(ApplicationTypeEnum.PRESET_APPLICATION.getType());
                PreErpSsoDTO preErpSsoDTO = Optional.ofNullable(JSON.toJavaObject(JSONObject.parseObject(x.getApplicationConfig()), PreErpSsoDTO.class)).orElse(new PreErpSsoDTO());
                appLinkDTO.setAppToken(preErpSsoDTO.getAppToken());
                appLinkDTO.setNeedAuthCodeFlag(x.getNeedAuthCodeFlag());
                appLinkDTO.setCloudwebsite(String.format(Constants.CS_APP_CALLBACK_URL, preErpSsoDTO.getApplicationCode()));
                appLinkDTO.setProtocolType(x.getProtocolType());
                appLinkDTO.setCasServerUrl(x.getCasServerUrl());
                appLinkDTO.setApplicationAppId(x.getAppId());
                appLinkDTO.setCallBackUrl(String.format(Constants.CS_APP_CALLBACK_URL, preErpSsoDTO.getApplicationCode()));
            }
            return appLinkDTO;
        }).collect(Collectors.toList());
    }

    /**
     * 获取手动配置的稳态和敏态的应用列表
     *
     * @return
     */
    private List<AppLinkDTO> queryStaticAppLinks() {
        AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
        List<AppLinkDTO> erpSSOList = queryErpSsoInfoAppLinks(user);
        List<AppLinkDTO> thirdSSOList = queryThirdSsoInfoAppLinks(user);
        erpSSOList.addAll(thirdSSOList);
        return erpSSOList;
    }

    /**
     * 获取稳态配置列表后台拼接参数
     *
     * @param req
     * @return
     * @throws Exception
     */
    private AppLinkDTO queryStaticAppLink(AuthoredUser user, QueryDisplayReq req) throws Exception {
        AppLinkDTO appLink = null;
        // 将云端eoc公司 转换成地端公司别
        convertCompanyId(user, req);
        // WFGP、E10
        if (ErpAppNameEnum.WFGP.getValue().equalsIgnoreCase(req.getErpAppName()) || ErpAppNameEnum.E10.getValue().equalsIgnoreCase(req.getErpAppName())) {
            ErpSsoInfo erpSsoInfo = erpSsoInfoService.erpSsoInfoWithParams(user, req);
            if (null == erpSsoInfo) {
                return null;
            }
            appLink = buildAppLinkDTO(erpSsoInfo.getCallBackUrl(), erpSsoInfo.getAppId(), erpSsoInfo.getAppToken(), erpSsoInfo.getId(), erpSsoInfo.getCode(), erpSsoInfo.getName(), null, erpSsoInfo.getExt(), ApplicationTypeEnum.CS_APPLICATION.getType(), erpSsoInfo.getProtocolType(), erpSsoInfo.getCasServerUrl(), erpSsoInfo.getUserBindFlag());
        } else if (ErpAppNameEnum.T100.getValue().equalsIgnoreCase(req.getErpAppName())) {
            // T100的处理
            appLink = dealT100(user, req);
        } else {
            log.warn("erpAppName not match,erpAppName:{}", req.getErpAppName());
        }
        // 添加授权码
        addAuthorize(appLink);
        return appLink;
    }

    /**
     * T100的动态参数替换处理
     *
     * @param user
     * @param req
     * @return
     */
    private AppLinkDTO dealT100(AuthoredUser user, QueryDisplayReq req) {
        List<AppLinkDTO> preAppList = queryPerInstallAppGroup(user);
        Optional<AppLinkDTO> appLinkOpt = preAppList.stream().filter(appLinkDTO -> Constants.T100_PRODUCT_TYPE.equals(appLinkDTO.getProductCode())).findFirst();
        if (!appLinkOpt.isPresent()) {
            return null;
        }
        AppLinkDTO appLink = appLinkOpt.get();
        // 替换动态参数的报表
        Map.Entry<String, String> pridEntry = getDynamicParamValue(ErpSsoConstants.DYNAMIC_PARAM_PRID, req.getDynamicParams());
        if (null != pridEntry) {
            String prid = pridEntry.getValue();
            appLink.setCloudwebsite(dynamicParamsReplace(appLink.getCloudwebsite(), prid, ErpSsoConstants.T100_PRID, ErpSsoConstants.T100_COMPANY_ID));
        }

        // 替换动态参数中的公司别
        Map.Entry<String, String> companyIdEntry = getDynamicParamValue(ErpSsoConstants.DYNAMIC_PARAM_COMPANY_ID, req.getDynamicParams());
        if (null != companyIdEntry) {
            String companyId = companyIdEntry.getValue();
            appLink.setCloudwebsite(dynamicParamsReplace(appLink.getCloudwebsite(), companyId, ErpSsoConstants.T100_COMPANY_ID, ErpSsoConstants.T100_COMPANY_ID_SUFFIX));
        }
        return appLink;
    }

    /**
     * 动态参数替换处理
     *
     * @param originalStr 原始的url
     * @param replaceStr  需要替换的字符串
     * @param beginStr    替换的起始标识符
     * @param endStr      替换的结束标识符
     * @return
     */
    private String dynamicParamsReplace(String originalStr, String replaceStr, String beginStr, String endStr) {
        log.info("dynamicParamsReplace originalStr:{}, replaceStr:{}", originalStr, replaceStr);
        int beginStrIndex = originalStr.indexOf(beginStr);
        if (beginStrIndex > 0) {
            String str1 = originalStr.substring(0, beginStrIndex + beginStr.length());
            String str2 = originalStr.substring(beginStrIndex + beginStr.length() + 1);
            int endStrIndex = str2.indexOf(endStr);
            String str3 = str2.substring(endStrIndex);
            String newStr = str1 + replaceStr + str3;
            log.info("dynamicParamsReplace newStr:{}", newStr);
            return newStr;
        }
        return originalStr;
    }

    /**
     * 将云端eoc公司 转换成地端公司别
     *
     * @param user
     * @param req
     */
    private void convertCompanyId(AuthoredUser user, QueryDisplayReq req) {
        Map.Entry<String, String> entry = getDynamicParamValue("companyId", req.getDynamicParams());
        if (null == entry) {
            return;
        }
        String companyId = entry.getValue();
        List<TenantProductOperationDTO> tenantProductOperationDTOList = mdcService.getTenantProductOperationList(user.getTenantId(), req.getErpAppName());
        if (com.baomidou.mybatisplus.core.toolkit.CollectionUtils.isEmpty(tenantProductOperationDTOList)) {
            return;
        }

        Optional<TenantProductOperationDTO> tenantProductOpt = tenantProductOperationDTOList.stream().filter(item -> item.getEocCompanyId().equals(companyId)).findFirst();
        if (tenantProductOpt.isPresent()) {
            req.getDynamicParams().put(entry.getKey(), tenantProductOpt.get().getOmCompanyId());
        }
    }

    private Map.Entry<String, String> getDynamicParamValue(String key, Map<String, String> dynamicParams) {
        Optional<Map.Entry<String, String>> keyOpt = dynamicParams.entrySet().stream().filter(param -> param.getKey().equalsIgnoreCase(key)).findFirst();
        if (!keyOpt.isPresent()) {
            return null;
        }
        return keyOpt.get();
    }

    /**
     * 添加授权码
     *
     * @param appLink
     * @throws Exception
     */
    private void addAuthorize(AppLinkDTO appLink) throws Exception {
        if (appLink == null) {
            return;
        }
        // 获取授权信息
        AuthorizeDTO authorizeDTO = iamService.queryAuthorizeInfo(appLink.getApplicationAppId(), appLink.getCallBackUrl());
        if (null == authorizeDTO) {
            return;
        }
        appLink.setCloudwebsite(appLink.getCloudwebsite() + "&code=" + authorizeDTO.getCode());
    }

    /**
     * 组装稳态或敏态的应用
     *
     * @param appToken  应用appToken
     * @param primaryId 应用主键
     * @param name      应用名称
     * @param desc      应用描述
     * @param ext       应用链接
     * @param type      应用类型 1：b/s 2：c/s 3：预设
     * @return
     */
    public AppLinkDTO buildAppLinkDTO(String callBackUrl, String appId, String appToken, Long primaryId, String code, String name, String desc, Map<String, Object> ext, Integer type, Integer protocolType, String casServerUrl, Integer userBindFlag) {
        AppLinkDTO appLink = new AppLinkDTO();
        appLink.setPrimaryId(String.valueOf(primaryId));
        appLink.setId(code);
        appLink.setCode(code);
        appLink.setName(name);
        appLink.setDesc(desc);
        appLink.setAppToken(appToken);
        appLink.setType(type + "");
        appLink.setSystemType(type);
        appLink.setDataSource(type);
        appLink.setCloudwebsite(ObjectUtils.isNotEmpty(ext) && ObjectUtils.isNotEmpty(ext.get("_fullpath")) ? ext.get("_fullpath").toString() : "");
        appLink.setProtocolType(protocolType);
        appLink.setCasServerUrl(casServerUrl);
        //appLink.setAppId(appId);
        appLink.setApplicationAppId(appId);
        appLink.setCallBackUrl(callBackUrl);
        appLink.setUserBindFlag(userBindFlag);
        if (Arrays.asList(Constants.BindFlagEnum.AUTO.getFlag(), Constants.BindFlagEnum.HAND.getFlag()).contains(userBindFlag)) {
            appLink.setNeedAuthCodeFlag(Constants.NEED_AUTH_CODE_FLAG_YES);
        }
        return appLink;
    }

    private Map<String, AppLinkDTO> buildAppMap(List<AppLinkDTO> defaultAppLinkList) {
        Map<String, AppLinkDTO> result = Maps.newHashMap();
        for (AppLinkDTO link : defaultAppLinkList) {
            result.put(link.getId(), link);
        }
        return result;
    }

    private List<AppLinkDTO> queryErpSsoInfoAppLinks(AuthoredUser user) {
        AppAuthContextHolder.getContext().setAuthoredUser(user);
        List<AppLinkDTO> ssoLinkList = Lists.newArrayList();
        List<ErpSsoInfo> erpSsoInfoList = erpSsoInfoService.erpSsoInfoWithParams();
        if (CollectionUtils.isNotEmpty(erpSsoInfoList)) {
            for (ErpSsoInfo erpSsoInfo : erpSsoInfoList) {
                AppLinkDTO appLink = buildAppLinkDTO(erpSsoInfo.getCallBackUrl(), erpSsoInfo.getAppId(), erpSsoInfo.getAppToken(), erpSsoInfo.getId(), erpSsoInfo.getCode(), erpSsoInfo.getName(), erpSsoInfo.getAppDesc(), erpSsoInfo.getExt(), ApplicationTypeEnum.CS_APPLICATION.getType(), erpSsoInfo.getProtocolType(), erpSsoInfo.getCasServerUrl(), erpSsoInfo.getUserBindFlag());
                ssoLinkList.add(appLink);
            }
        }
        return ssoLinkList;
    }


    private List<AppLinkDTO> queryThirdSsoInfoAppLinks(AuthoredUser user) {
        AppAuthContextHolder.getContext().setAuthoredUser(user);
        List<AppLinkDTO> ssoLinkList = Lists.newArrayList();
        List<ThirdSsoInfoDto> infosThirdPartWeb = thirdSsoInfoService.erpSsoInfoWithParams();
        if (CollectionUtils.isNotEmpty(infosThirdPartWeb)) {
            for (ThirdSsoInfoDto thirdSsoInfo : infosThirdPartWeb) {
                AppLinkDTO appLink = buildAppLinkDTO(thirdSsoInfo.getCallBackUrl(), thirdSsoInfo.getAppId(), thirdSsoInfo.getAppToken(), thirdSsoInfo.getId(), thirdSsoInfo.getCode(), thirdSsoInfo.getName(), thirdSsoInfo.getAppDesc(), thirdSsoInfo.getExt(), ApplicationTypeEnum.BS_APPLICATION.getType(), thirdSsoInfo.getProtocolType(), thirdSsoInfo.getCasServerUrl(), thirdSsoInfo.getUserBindFlag());
                ssoLinkList.add(appLink);
            }
        }
        return ssoLinkList;
    }

    /**
     * 查询移动端应用列表，包括跳转链接
     *
     * @return 返回
     */
    private List<AppLinkDTO> queryMobileAppLink(AuthoredUser user) {
        AppAuthContextHolder.getContext().setAuthoredUser(user);
        List<AppLinkDTO> appLinkList = Lists.newArrayList();
        List<MobileSsoInfo> mobileSsoInfoList = mobileSsoInfoService.queryMobileLinkSsoList(null);
        if (CollectionUtils.isEmpty(mobileSsoInfoList)) {
            return appLinkList;
        }

        for (MobileSsoInfo mobileSsoInfo : mobileSsoInfoList) {
            AppLinkDTO appLink = new AppLinkDTO();
            appLink.setPrimaryId(String.valueOf(mobileSsoInfo.getId()));
            appLink.setId(mobileSsoInfo.getAppCode());
            appLink.setCode(mobileSsoInfo.getAppCode());
            appLink.setName(mobileSsoInfo.getAppName());
            appLink.setDesc(mobileSsoInfo.getAppDesc());
            appLink.setAppToken(mobileSsoInfo.getAppToken());
            appLink.setCloudwebsite(mobileSsoInfo.getMobileLinkUrl());
            appLink.setAppId(mobileSsoInfo.getAppId());
            appLink.setApplicationAppId(mobileSsoInfo.getAppId());
            appLink.setCallBackUrl(mobileSsoInfo.getCallBackUrl());
            appLink.setUserBindFlag(mobileSsoInfo.getUserBindFlag());
            appLink.setSystemType(ApplicationTypeEnum.BS_APPLICATION.getType());
            if (Arrays.asList(Constants.BindFlagEnum.AUTO.getFlag(), Constants.BindFlagEnum.HAND.getFlag()).contains(mobileSsoInfo.getUserBindFlag())) {
                appLink.setNeedAuthCodeFlag(Constants.NEED_AUTH_CODE_FLAG_YES);
            }
            appLinkList.add(appLink);
        }
        return appLinkList;
    }


    @Override
    public List<AppLinkDTO> queryManageListSync(AuthoredUser user) {
        try {
            CompletableFuture<List<AppLinkDTO>> erpSsoInfo = CompletableFuture.supplyAsync(() -> queryErpSsoInfoAppLinks(user), asyncTaskExecutor);
            CompletableFuture<List<AppLinkDTO>> queryThirdSsoInfo = CompletableFuture.supplyAsync(() -> queryThirdSsoInfoAppLinks(user), asyncTaskExecutor);
            CompletableFuture<List<AppLinkDTO>> queryPerInstallApp = CompletableFuture.supplyAsync(() -> queryPerInstallAppGroup(user), asyncTaskExecutor);
            CompletableFuture<List<AppLinkDTO>> queryDefaultGroup = CompletableFuture.supplyAsync(() -> queryDefaultGroup(user), asyncTaskExecutor);
            CompletableFuture.allOf(erpSsoInfo, queryThirdSsoInfo, queryPerInstallApp, queryDefaultGroup);

            List<AppLinkDTO> erpSsoInfoAppList = erpSsoInfo.get();
            List<AppLinkDTO> thirdSsoInfoAppList = queryThirdSsoInfo.get();
            List<AppLinkDTO> preAppList = queryPerInstallApp.get();
            List<AppLinkDTO> iamAppLinkList = queryDefaultGroup.get();
            List<AppLinkDTO> result = new ArrayList<>();
            result.addAll(erpSsoInfoAppList);
            result.addAll(thirdSsoInfoAppList);
            result.addAll(preAppList);
            // 过滤出已经购买的应用
            List<AppLinkDTO> filteredAppLinkList = filterBSAppList(user.getTenantId(), iamAppLinkList);
            result.addAll(filteredAppLinkList);
            return result;
        } catch (Exception e) {
            return new ArrayList<>();
        }
    }

    /**
     * 异步查询移动端+IAM应用列表
     *
     * @param user 用户信息
     * @return 返回
     */
    @Override
    public List<AppLinkDTO> queryMobileManageListSync(AuthoredUser user) {
        try {
            CompletableFuture<List<AppLinkDTO>> mobileSsoAppList = CompletableFuture.supplyAsync(() -> queryMobileAppLink(user), asyncTaskExecutor);
            CompletableFuture<List<AppLinkDTO>> iamAppList = CompletableFuture.supplyAsync(() -> queryDefaultGroup(user), asyncTaskExecutor);
            CompletableFuture.allOf(mobileSsoAppList, iamAppList);

            List<AppLinkDTO> mobileSsoList = mobileSsoAppList.get();
            List<AppLinkDTO> iamAppLinkList = iamAppList.get();

            // 过滤出已经购买的应用
            List<AppLinkDTO> result = new ArrayList<>(mobileSsoList);
            List<AppLinkDTO> filteredAppLinkList = filterBSAppList(user.getTenantId(), iamAppLinkList);
            result.addAll(filteredAppLinkList);
            return result;
        } catch (Exception e) {
            log.error("query mobile manage list sync error.", e);
            return new ArrayList<>();
        }
    }

    /**
     * 构建应用跳转的完整链接
     *
     * @param url        应用返回的链接
     * @param appLinkDTO 应用信息
     * @return 返回
     */
    public String parseJumpUrl(String url, AppLinkDTO appLinkDTO) {
        if (StringUtils.isEmpty(url)) {
            return url;
        }
        AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
        String finalUrl = null;
        if (Objects.equals(ProtocolTypeEnum.OAUTH.getType(), appLinkDTO.getProtocolType())) {
            if (!appLinkDTO.getType().equals(PreinstalledApplicationTypeEnum.SPECIAL_APPLICATION.getValue())) {
                if (Constants.APP_TYPE_BS.equals(appLinkDTO.getSystemType())) {
                    finalUrl = url + (url.contains("?") ? "&" : "?")
                            + "appToken=" + appLinkDTO.getAppToken() +
                            "&tenantId=" + user.getTenantId() +
                            "&appCode=" + appLinkDTO.getCode() +
                            "&appId=" + appLinkDTO.getApplicationAppId() +
                            "&curLocale=" + LocaleContextHolder.getLocale() +
                            "&bindMode=" + appLinkDTO.getUserBindFlag();
                }
                if (Constants.APP_TYPE_CS.equals(appLinkDTO.getSystemType())) {
                    StringBuilder cloudwebsite = new StringBuilder(appLinkDTO.getCloudwebsite());
                    JSONObject jsonObject = null;
                    try {
                        jsonObject = JSONObject.parseObject(url);
                    }catch (Exception e){
                        return null;
                    }
                    for (String key : jsonObject.keySet()) {
                        Object value = jsonObject.get(key);
                        cloudwebsite.append("&").append(key).append("=").append(value);
                    }
                    finalUrl = cloudwebsite.toString();
                }
            }
            // T100
            else if (Constants.T100_PRODUCT_TYPE.equals(appLinkDTO.getProductCode())) {
                finalUrl = appLinkDTO.getCloudwebsite();
                List<PreinstalledApplication> preInstalledAppList = preinstalledApplicationService.queryPreinstalledAppList(AppAuthContextHolder.getContext().getAuthoredUser());
                List<PreinstalledApplication> T100AppList = preInstalledAppList.stream()
                        .filter(x -> Constants.T100_PRODUCT_TYPE.equals(x.getProductCode()) && appLinkDTO.getPrimaryId().equals(String.valueOf(x.getId()))).collect(Collectors.toList());
                if (CollectionUtils.isEmpty(T100AppList)) {
                    return finalUrl;
                }
                if (StringUtils.isBlank(url) || url.split("&").length < 4) {
                    return finalUrl;
                }

                // T100消息跳转的参数拼接，"Arg=asft311&Arg=99&Arg=DSCNJ&Arg="default_wc":"(pmdsdocno = 'CT3-Y53-240600000001') OR 1<>1"
                // 原来单点登入格式，t100v1://10.40.41.146/wtopprd/wa/r/app/gdc_azzi000?Arg=&Arg=T100SSO&Arg=QVRIRU5B&Arg=MjAyNDA5MTgxNDAxMTFhdGhlbmFBYjEyMzRQV2QyODY4MjI2NlQxMDBmcmFua3RpcHRvcA==
                // 消息串接，t100v1://10.40.41.146/wtopprd/wa/r/app/gdc_asft311?Arg=99&Arg=T100SSO&Arg=QVRIRU5B&Arg=MjAyNDA5MTgxNDAxMTFhdGhlbmFBYjEyMzRQV2QyODY4MjI2NlQxMDBmcmFua3RpcHRvcA==&Arg=DSCNJ&Arg="default_wc":"(pmdsdocno = 'CT3-Y53-240600000001') OR 1<>1
                PreinstalledApplicationInstance applicationInstance = preinstalledApplicationService.getPreINstance(T100AppList.get(0).getId(), "", "");
                List<String> paramList = Arrays.asList(url.split("&"));
                // 第一个参数 作业编号
                String[] argOne = paramList.get(0).split("=");
                finalUrl = finalUrl.replace("azzi000", argOne[1]);

                // 第二个参数 企业编号
                String companyCode = Optional.ofNullable(applicationInstance.getApplicationExtConfig()).map(object -> object.getString("companyCode")).orElse("");
                finalUrl = finalUrl.replace("Arg=" + companyCode, paramList.get(1));

                // 第三个参数（据点编号），第四个参数（单号）直接拼接
                finalUrl = finalUrl + "&" + paramList.get(2) + "&" + paramList.get(3);
            }
            // TIPTOP(暂时没有消息)
            else if (Constants.TIPTOP_PRODUCT_TYPE.equals(appLinkDTO.getProductCode())) {
                String cloudwebsite = appLinkDTO.getCloudwebsite();
                List<PreinstalledApplication> preInstalledAppList = preinstalledApplicationService.queryPreinstalledAppList(AppAuthContextHolder.getContext().getAuthoredUser());
                List<PreinstalledApplication> TTAppList = preInstalledAppList.stream().filter(x -> Constants.TIPTOP_PRODUCT_TYPE.equals(x.getProductCode())).collect(Collectors.toList());

                // 动态参数
                JSONObject jsonObject = JSONObject.parseObject(url);
                StringBuilder arg = new StringBuilder();
                for (String key : jsonObject.keySet()) {
                    Object value = jsonObject.get(key);
                    arg.append("&").append(key).append("=").append(value);
                }
                for (PreinstalledApplication x : TTAppList) {
                    if (appLinkDTO.getPrimaryId().equals(String.valueOf(x.getId()))) {
                        ErpSsoSpecialDTO erpSsoSpecialDTO = Optional.ofNullable(JSONObject.parseObject(x.getApplicationConfig(), ErpSsoSpecialDTO.class)).orElse(new ErpSsoSpecialDTO());
                        finalUrl = cloudwebsite.replaceAll("&Arg=" + erpSsoSpecialDTO.getJobCode(), arg.toString());
                    }
                }
            }
        }
        return finalUrl;
    }

    /**
     * 管理后台，根据权限id查询所有有权限的应用列表
     *
     * @param authId 权限id
     * @return
     */
    @Override
    public List<AppLinkDTO> queryAllAuthorizedList(String authId) {
        // 获取当前权限对象的应用列表
        List<AppLinkDTO> authAppList = Lists.newArrayList();
        QueryWrapper<BizObjAuthRel> condition = new QueryWrapper<BizObjAuthRel>().eq("auth_id", authId);
        List<BizObjAuthRel> bizObjAuthRelList = bizObjAuthRelService.getBaseMapper().selectList(condition);

        // 返回
        if (CollectionUtils.isEmpty(bizObjAuthRelList)) {
            return authAppList;
        }
        authAppList = bizObjAuthRelList.stream().map(x -> {
            AppLinkDTO appLinkDTO = new AppLinkDTO();
            appLinkDTO.setPrimaryId(x.getBizObjId());
            appLinkDTO.setType(String.valueOf(x.getBizObjSubType()));
            return appLinkDTO;
        }).collect(Collectors.toList());
        return authAppList;
    }

    @Override
    public GetSsoUrlResp getSsoUrl(GetSsoUrlReq req) {
        GetSsoUrlResp resp = new GetSsoUrlResp();
        AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
        if(!ApplicationTypeEnum.BS_APPLICATION.getType().toString().equals(req.getAppType())){
            return resp;
        }

        String callBackUrl = "";
        // 只有SSO配置来源类型为手动则查手动配置，其他均查内置配置
        if(StringUtils.isNotEmpty(req.getSsoSourceType()) && SsoSourceTypeEnum.MANUAL.getValue().equals(req.getSsoSourceType())){
            QueryWrapper<ThirdSsoInfo> queryWrapper = new QueryWrapper<>();
            //只查询有效链接
            queryWrapper.eq("valid_status", Constants.VALID_STATUS_ENABLE);
            queryWrapper.eq("app_code", req.getSsoAppCode());
            // 1、查询t_third_sso_info
            List<ThirdSsoInfo> infos = thirdSsoInfoService.list(queryWrapper);
            if (CollectionUtils.isEmpty(infos)) {
                return resp;
            }
            ThirdSsoInfo info = infos.get(0);
            callBackUrl = info.getCallBackUrl();
        } else {
            List<PreinstalledApplication> preinstalledAppList = preinstalledApplicationService.queryPreinstalledAppList(AppAuthContextHolder.getContext().getAuthoredUser());
            if(CollectionUtils.isEmpty(preinstalledAppList)){
                return resp;
            }
            PreinstalledApplication preinstalledApp = preinstalledAppList.stream().filter(x -> req.getSsoAppCode().equals(x.getApplicationCode())).findFirst().orElse(null);
            if(null == preinstalledApp || StringUtils.isEmpty(preinstalledApp.getApplicationConfig())){
                return resp;
            }
            ThirdSsoInfoDto thirdSsoInfoDto = Optional.ofNullable(JSON.toJavaObject(JSONObject.parseObject(preinstalledApp.getApplicationConfig()), ThirdSsoInfoDto.class)).orElse(new ThirdSsoInfoDto());
            callBackUrl = thirdSsoInfoDto.getCallBackUrl();
        }
        // 不需要免登，则返回全路径普通地址
        if(null == req.getSsoLogin() || !req.getSsoLogin()){
            resp.setAddress(callBackUrl);
            return resp;
        }

        // 2、拼接sso-login
        String ssoAddress = String.format(SSO_LOGIN_URL,callBackUrl, user.getToken(), LocaleContextHolder.getLocale());
        resp.setSsoAddress(ssoAddress);
        if(MapUtils.isEmpty(req.getDynamicParams())){
            return resp;
        }

        // 3、拼接动态参数
        Iterator<Map.Entry<String,String>> iterator = req.getDynamicParams().entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String,String> entry = iterator.next();
            String key = entry.getKey();
            String value = entry.getValue();
            if(StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)){
                ssoAddress+="&"+key+"="+value;
                resp.setSsoAddress(ssoAddress);
            }
        }
        return resp;
    }

    /**
     * 根据应用来源和应用主键id查询应用信息
     *
     * @param dataType     应用来源
     * @param appPrimaryId 应用主键id
     * @param user         用户信息
     * @return 返回
     */
    @Override
    public AppLinkDTO queryApplicationByDataType(Integer dataType, String appPrimaryId, AuthoredUser user) {
        List<AppLinkDTO> appLinkDTOList;
        if (ApplicationTypeEnum.CS_APPLICATION.getType().equals(dataType)) {
            appLinkDTOList = queryErpSsoInfoAppLinks(user);
        } else if (ApplicationTypeEnum.BS_APPLICATION.getType().equals(dataType)) {
            appLinkDTOList = queryThirdSsoInfoAppLinks(user);
        } else if (ApplicationTypeEnum.PRESET_APPLICATION.getType().equals(dataType)) {
            appLinkDTOList = queryPerInstallAppGroup(user);
        } else {
            appLinkDTOList = queryDefaultGroup(user);
            appLinkDTOList = filterBSAppList(user.getTenantId(), appLinkDTOList);
        }
        return appLinkDTOList.stream().filter(x -> appPrimaryId.equals(x.getPrimaryId())).findFirst().orElse(null);
    }

    /**
     * 查询sso配置列表
     *
     * @param appIdList appId
     * @return 返回
     */
    @Override
    public List<AppLinkDTO> querySsoConfigListSync(List<String> appIdList) {
        try {
            // CS应用配置
            CompletableFuture<List<ErpSsoInfo>> erpSsoInfoList = CompletableFuture.supplyAsync(() -> {
                LambdaQueryWrapper<ErpSsoInfo> wrapper = new LambdaQueryWrapper<>();
                wrapper.in(ErpSsoInfo::getAppId, appIdList);
                return erpSsoInfoService.getBaseMapper().selectList(wrapper);
            }, asyncTaskExecutor);
            // BS应用配置
            CompletableFuture<List<ThirdSsoInfo>> thirdSsoInfoList = CompletableFuture.supplyAsync(() -> {
                LambdaQueryWrapper<ThirdSsoInfo> wrapper = new LambdaQueryWrapper<>();
                wrapper.in(ThirdSsoInfo::getAppId, appIdList);
                return thirdSsoInfoService.getBaseMapper().selectList(wrapper);
            }, asyncTaskExecutor);
            // 预设应用配置
            CompletableFuture<List<PreinstalledApplicationInstance>> preSsoInfoList = CompletableFuture.supplyAsync(() -> {
                LambdaQueryWrapper<PreinstalledApplicationInstance> wrapper = new LambdaQueryWrapper<>();
                wrapper.in(PreinstalledApplicationInstance::getAppId, appIdList);
                return preinstalledApplicationInstanceMapper.getPreInsBy(Utils.getTenantId(), null, appIdList);
            }, asyncTaskExecutor);
            CompletableFuture.allOf(erpSsoInfoList, thirdSsoInfoList, preSsoInfoList).join();
            List<ErpSsoInfo> csAppList = erpSsoInfoList.get();
            List<ThirdSsoInfo> bsAppList = thirdSsoInfoList.get();
            List<PreinstalledApplicationInstance> preAppList = preSsoInfoList.get();

            List<AppLinkDTO> resultList = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(csAppList)) {
                csAppList.forEach(x -> {
                    AppLinkDTO appLinkDTO = new AppLinkDTO();
                    appLinkDTO.setApplicationAppId(x.getAppId());
                    appLinkDTO.setName(x.getName());
                    resultList.add(appLinkDTO);
                });
            }
            if (CollectionUtils.isNotEmpty(bsAppList)) {
                bsAppList.forEach(x -> {
                    AppLinkDTO appLinkDTO = new AppLinkDTO();
                    appLinkDTO.setApplicationAppId(x.getAppId());
                    appLinkDTO.setName(x.getAppName());
                    resultList.add(appLinkDTO);
                });
            }
            if (CollectionUtils.isNotEmpty(preAppList)) {
                preAppList.forEach(x -> {
                    AppLinkDTO appLinkDTO = new AppLinkDTO();
                    appLinkDTO.setApplicationAppId(x.getAppId());
                    appLinkDTO.setName(x.getPreinstalledApplicationName());
                    resultList.add(appLinkDTO);
                });
            }
            return resultList;
        } catch (Exception e) {
            log.error("query sso config list error. appIdList:{}", appIdList, e);
            return new ArrayList<>();
        }
    }
}
