package com.digiwin.athena.semc.service.cache.impl;

import com.digiwin.athena.semc.service.cache.ICacheService;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.util.Set;

/**
 * @description: 缓存服务
 * @author: sunyfa
 */
@Service
public class CacheServiceImpl implements ICacheService {
    public final static String KEY_PREFIX = "semc:cache:";

    public final static Duration DEFAULT_DURATION = Duration.ofDays(7);

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    @Override
    public Long incr(String key) {
        return stringRedisTemplate.opsForValue().increment(KEY_PREFIX + key);
    }

    @Override
    public boolean hasKey(String key) {
        return stringRedisTemplate.hasKey(KEY_PREFIX + key);
    }

    @Override
    public boolean hasKey(String serviceName, String functionName, String paraStr) {
        return hasKey(getFormatKey(serviceName, functionName, paraStr));
    }

    @Override
    public void delete(String key) {
        stringRedisTemplate.delete(KEY_PREFIX + key);
    }

    @Override
    public void deleteByPrex(String prex) {
        Set<String> keys = stringRedisTemplate.keys(KEY_PREFIX + prex);
        if (!CollectionUtils.isEmpty(keys)) {
            stringRedisTemplate.delete(keys);
        }
    }

    @Override
    public void cache(String key, String value) {
        stringRedisTemplate.boundValueOps(KEY_PREFIX + key).set(value);
    }

    @Override
    public void cache(String key, String value, Duration duration) {
        stringRedisTemplate.boundValueOps(KEY_PREFIX + key).set(value, duration);
    }

    @Override
    public void cache(String serviceName, String functionName, String paraStr, String value) {
        cache(getFormatKey(serviceName, functionName, paraStr), value);
    }

    @Override
    public void cache(String serviceName, String functionName, String paraStr, String value, Duration duration) {
        cache(getFormatKey(serviceName, functionName, paraStr), value, duration);
    }

    @Override
    public Object get(String key) {
        key = KEY_PREFIX + key;
        return stringRedisTemplate.boundValueOps(key).get();
    }

    @Override
    public String getValue(String key) {
        key = KEY_PREFIX + key;
        return stringRedisTemplate.boundValueOps(key).get();
    }

    @Override
    public Object get(String serviceName, String functionName, String paraStr) {
        return get(getFormatKey(serviceName, functionName, paraStr));
    }

    private String getFormatKey(String serviceName, String functionName, String paraStr) {
        return String.format("%s.%s(%s)", serviceName, functionName, paraStr);
    }

    @Override
    public void batchDelete(String pattern) {
        Set<String> keys = stringRedisTemplate.keys(KEY_PREFIX + pattern);
        stringRedisTemplate.delete(keys);
    }

    @Override
    public void cacheNoPrefix(String key, String value) {
        stringRedisTemplate.opsForValue().set(key, value);
    }
}
