package com.digiwin.athena.semc.service.cache.impl;


import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.dto.config.MobilePortalCacheConfigParam;
import com.digiwin.athena.semc.dto.fineReport.FineReportConfigDTO;
import com.digiwin.athena.semc.service.cache.CommonConfigService;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.service.common.EnvService;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.util.Arrays;

import io.vavr.Tuple2;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Service
@RequiredArgsConstructor
@Slf4j
public class CommonConfigServiceImpl implements CommonConfigService {

    private final ICacheService cacheService;

    private final EnvService envService;

    public static final String PREFIX_KEY = "cc:";

    public static final String MOBILE_PORTAL_CACHE_CONFIG_PARAM_KEY = PREFIX_KEY + "MobilePortalCacheConfigParam";

    public static final String NEWS_PUBLISH_EDIT_FLAG = PREFIX_KEY + "NewsPublishEditFlag";

    public static final String IAM_USER_AUTH_INFO_CONFIG = PREFIX_KEY + "IamUserAuthInfoConfig";

    /**
     * 是否查询超管缓存 开关
     */
    public static final String SUPER_ADMIN_FLAG_SWITCH = PREFIX_KEY + "SuperAdminFlagSwitch";


    public static final String BACKEND_MENU_OPERATE_FLAG = PREFIX_KEY + "BackendMenuOperateFlag";

    public static final String BACKEND_MENU_OPERATE_USER = "zhouyxa@digiwin.com,chendda@digiwin.com,chenzlc@digiwin.com,dengkw@digiwin.com,hongyana@digiwin.com,liujxb@digiwin.com,liuyanb@digiwin.com,liypb@digiwin.com,songcla@digiwin.com,sungqz@digiwin.com,sunztf@digiwin.com,xueah@digiwin.com,yufei@digiwin.com,xiaozm@digiwin.com";

    /**
     * 公告发送工作提醒人员数量
     */
    public static final String NEWS_SEND_NOTICE_USER_COUNT = PREFIX_KEY + "NewsSendNoticeUserCount";

    /**
     * 公告是否给外部用户发送工作提醒
     */
    public static final String NEWS_SEND_NOTICE_OUTER_USER_FLAG = PREFIX_KEY + "NewsSendNoticeOuterUserFlag";

    /**
     * 帆软应用参数配置缓存key
     */
    public static final String FINE_REPORT_CACHE_CONFIG_KEY = PREFIX_KEY + "FineReportCacheConfig:";


    @Override
    public String getCommonConfig(String key) {
        try {
            return cacheService.getValue(key);
        } catch (Exception e) {
            log.error("getCommonConfigEx", e);
        }
        return "";
    }

    @Override
    public boolean setCommonConfig(String key, String value) {
        return setCommonConfig(key, value, CacheServiceImpl.DEFAULT_DURATION);
    }

    @Override
    public boolean setCommonConfig(String key, String value, Duration duration) {
        try {
            if (duration != null) {
                cacheService.cache(key, value, duration);
            } else {
                cacheService.cache(key, value);
            }
            return true;
        } catch (Exception e) {
            log.error("setCommonConfigEx", e);
        }
        return false;
    }

    @Override
    public boolean removeCommonConfig(String key) {
        try {
            cacheService.delete(key);
            return true;
        } catch (Exception e) {
            log.error("removeCommonConfigEx", e);
        }
        return false;
    }

    @Override
    public MobilePortalCacheConfigParam getMobilePortalCacheConfigParam() {
        String value = getCommonConfig(MOBILE_PORTAL_CACHE_CONFIG_PARAM_KEY);
        if (StringUtils.isBlank(value)) {
            return null;
        }
        return JSON.parseObject(value, MobilePortalCacheConfigParam.class);
    }

    @Override
    public boolean setMobilePortalCacheConfigParam(MobilePortalCacheConfigParam param) {
        if (param == null) {
            return false;
        }
        String value = JSON.toJSONString(param);
        return setCommonConfig(MOBILE_PORTAL_CACHE_CONFIG_PARAM_KEY, value);
    }

    @Override
    public boolean getNewsPublishEditFlag() {
        String value = getCommonConfig(NEWS_PUBLISH_EDIT_FLAG);
        if (StringUtils.isBlank(value)) {
            return false;
        }
        return Boolean.TRUE.toString().equals(value);
    }

    @Override
    public Tuple2<Boolean, Integer> getIamUserAuthInfoConfig() {
        String value = getCommonConfig(IAM_USER_AUTH_INFO_CONFIG);
        if (StringUtils.isBlank(value)) {
            return new Tuple2<>(true, 900);
        }
        JSONObject jsonObject = JSON.parseObject(value);
        return new Tuple2<>(jsonObject.getBoolean("switch"), jsonObject.getInteger("expire"));
    }

    /**
     * 是否查询超管缓存 开关配置
     *
     * @return 是否开启缓存和缓存时间（分钟）
     * {
     * "switch": true,
     * "expire": 900
     * }
     */
    @Override
    public Tuple2<Boolean, Integer> querySuperAdminFlagSwitch() {
        String value = getCommonConfig(SUPER_ADMIN_FLAG_SWITCH);
        if (StringUtils.isBlank(value)) {
            return new Tuple2<>(true, 900);
        }
        JSONObject jsonObject = JSON.parseObject(value);
        return new Tuple2<>(jsonObject.getBoolean("switch"), jsonObject.getInteger("expire"));
    }

    @Override
    public boolean getBackendMenuOperateFlag() {
        //下地部署环境默认具有
        if (!envService.isOnline()) {
            return true;
        }
        String value = getCommonConfig(BACKEND_MENU_OPERATE_FLAG);
        if (StringUtils.isBlank(value)) {
            value = BACKEND_MENU_OPERATE_USER;
        }
        String userId = AppAuthContextHolder.getContext().getAuthoredUser().getUserId();
        return Arrays.asList(value.split(",")).contains(userId);
    }

    /**
     * 获取公告发送工作提醒人员数量
     *
     * @return 返回
     */
    @Override
    public Integer querySendNewsNoticeUserCount() {
        String count = getCommonConfig(NEWS_SEND_NOTICE_USER_COUNT);
        if (StringUtils.isBlank(count)) {
            return 200;
        }
        return Integer.parseInt(count);
    }

    /**
     * 获取公告是否给外部用户发送工作提醒 开关配置
     *
     * @return 返回
     */
    @Override
    public boolean querySendNewsNoticeOuterFlag() {
        String value = getCommonConfig(NEWS_SEND_NOTICE_OUTER_USER_FLAG);
        if (StringUtils.isBlank(value)) {
            return true;
        }
        return Boolean.TRUE.toString().equals(value);
    }

    /**
     * 查询帆软应用参数配置
     *
     * @param tenantId 租户id
     * @return 返回
     */
    @Override
    public FineReportConfigDTO queryFineReportConfigCache(String tenantId) {
        String value = getCommonConfig(FINE_REPORT_CACHE_CONFIG_KEY + tenantId);
        if (StringUtils.isBlank(value)) {
            return null;
        }
        return JSON.parseObject(value, FineReportConfigDTO.class);
    }

    /**
     * 设置帆软应用参数配置
     *
     * @param fineReportConfigDTO 配置值
     * @param tenantId 租户id
     */
    @Override
    public void setFineReportConfigCache(FineReportConfigDTO fineReportConfigDTO, String tenantId) {
        if (ObjectUtils.isEmpty(fineReportConfigDTO)) {
            return;
        }
        String value = JSON.toJSONString(fineReportConfigDTO);
        setCommonConfig(FINE_REPORT_CACHE_CONFIG_KEY + tenantId, value);
    }
}
