package com.digiwin.athena.semc.service.common.impl;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.enums.BizAuthTypeEnum;
import com.digiwin.athena.semc.common.enums.BizObjectTypeEnum;
import com.digiwin.athena.semc.entity.common.BizObjAuthRel;
import com.digiwin.athena.semc.entity.portal.KnowledgeApplicationRef;
import com.digiwin.athena.semc.mapper.common.BizObjAuthRelMapper;
import com.digiwin.athena.semc.mapper.portal.KnowledgeApplicationMapper;
import com.digiwin.athena.semc.proxy.eoc.service.EocService;
import com.digiwin.athena.semc.proxy.eoc.service.model.UserDeptInfoDTO;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.OrgRoleUserDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.UserDTO;
import com.digiwin.athena.semc.proxywrapper.IamUserService;
import com.digiwin.athena.semc.proxywrapper.dto.IamUserAuthInfo;
import com.digiwin.athena.semc.service.common.IBizObjAuthRelService;
import com.digiwin.athena.semc.vo.common.BizObjAuthRelBatchForLinkReq;
import com.digiwin.athena.semc.vo.common.BizObjAuthRelBatchVo;
import com.digiwin.athena.semc.vo.common.BizObjAuthRelVo;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;

/**
 * @description: 业务对象权限关系实现类
 * @createDate: 2023/5/26 13:53
 * @author: sunyfa
 */
@Slf4j
@Service
public class BizObjAuthRelServiceImpl extends ServiceImpl<BizObjAuthRelMapper, BizObjAuthRel> implements IBizObjAuthRelService {

    @Resource
    private IamService iamService;

    @Resource
    private EocService eocService;

    @Autowired
    private KnowledgeApplicationMapper knowledgeApplicationMapper;

    @Resource
    BizObjAuthRelMapper bizObjAuthRelMapper;

    @Autowired
    private IamUserService iamUserService;

    @Resource
    private Converter converter;

    /**
     * @param
     * @return
     * @description: 新增/修改业务对象权限关系
     * @author: sunyfa
     */
    @Override
    @Transactional
    public void addBizObjectAuthRel(BizObjAuthRelBatchVo bizObjAuthRelBatchVo, int bizObjectType) {
        QueryWrapper<BizObjAuthRel> condition = new QueryWrapper<>();
        condition.eq("biz_obj_type", bizObjectType);
        condition.in("biz_obj_id", bizObjAuthRelBatchVo.getIds());

        // 删除业务对象权限关系
        baseMapper.delete(condition);
        // 插入业务对象权限关系
        List<BizObjAuthRelVo> allBizObjAuthRelVoList = new ArrayList<>();
        allBizObjAuthRelVoList.addAll(bizObjAuthRelBatchVo.getOrg() == null ? Lists.newArrayList() : bizObjAuthRelBatchVo.getOrg());
        allBizObjAuthRelVoList.addAll(bizObjAuthRelBatchVo.getRole() == null ? Lists.newArrayList() : bizObjAuthRelBatchVo.getRole());
        allBizObjAuthRelVoList.addAll(bizObjAuthRelBatchVo.getUser() == null ? Lists.newArrayList() : bizObjAuthRelBatchVo.getUser());
        if (CollectionUtils.isNotEmpty(allBizObjAuthRelVoList)) {
            List<BizObjAuthRel> bizObjAuthRelList = new ArrayList<>();
            allBizObjAuthRelVoList.forEach(x -> bizObjAuthRelBatchVo.getIds().forEach(id -> {
                BizObjAuthRel bizObjAuthRel = converter.convert(x, BizObjAuthRel.class);
                bizObjAuthRel.setBizObjId(String.valueOf(id));
                bizObjAuthRelList.add(bizObjAuthRel);
            }));
            saveBatch(bizObjAuthRelList);
        }
    }

    /**
     * 管理后台，保存应用的查看权限对象
     *
     * @param bizObjAuthRelBatchForLinkReq 请求入参
     */
    @Override
    @Transactional
    public void saveBizObjectAuthRelForLinkApp(BizObjAuthRelBatchForLinkReq bizObjAuthRelBatchForLinkReq) {
        List<BizObjAuthRelBatchForLinkReq.AuthVO> authList = bizObjAuthRelBatchForLinkReq.getAuthList();
        List<Long> authIdList = authList.stream().map(BizObjAuthRelBatchForLinkReq.AuthVO::getAuthId).collect(Collectors.toList());

        QueryWrapper<BizObjAuthRel> condition = new QueryWrapper<>();
        condition.in("auth_id", authIdList);
        condition.eq("biz_obj_type", BizObjectTypeEnum.APPlICATION_MANAGEMENT.getValue());

        // 先查询表里现有的权限关联的应用
        List<BizObjAuthRel> currentRelList = bizObjAuthRelMapper.selectList(condition);
        if (CollectionUtils.isNotEmpty(currentRelList)) {
            // 删除权限对象关联的应用
            baseMapper.delete(condition);
        }

        // 插入权限对象绑定的应用
        List<BizObjAuthRelBatchForLinkReq.AppVO> appList = bizObjAuthRelBatchForLinkReq.getAppList();
        List<BizObjAuthRel> bizObjAuthRelList = new ArrayList<>();
        authList.forEach(authVO -> {
            for (BizObjAuthRelBatchForLinkReq.AppVO appVO : appList) {
                BizObjAuthRel bizObjAuthRel = new BizObjAuthRel();
                bizObjAuthRel.setAuthId(authVO.getAuthId());
                bizObjAuthRel.setAuthType(authVO.getAuthType());
                bizObjAuthRel.setAuthName(authVO.getAuthName());
                bizObjAuthRel.setBizObjId(appVO.getPrimaryId());
                bizObjAuthRel.setBizObjSubType(appVO.getBizObjSubType());
                bizObjAuthRel.setBizObjType(BizObjectTypeEnum.APPlICATION_MANAGEMENT.getValue());
                bizObjAuthRelList.add(bizObjAuthRel);
            }
        });
        // 500个一组分批存储，防止一次数据过大
        List<List<BizObjAuthRel>> partList = Lists.partition(bizObjAuthRelList, 500);
        for (List<BizObjAuthRel> tempList : partList) {
            saveBatch(tempList);
        }

        // 给用户授权或取消授权应用
        authorizeUserApp(currentRelList, bizObjAuthRelList, authList.get(0));
    }

    /**
     * 给用户授权或取消授权应用
     *
     * @param currentRelList    当前的应用权限
     * @param bizObjAuthRelList 最终的应用权限
     * @param authVO            权限对象
     */
    private void authorizeUserApp(List<BizObjAuthRel> currentRelList, List<BizObjAuthRel> bizObjAuthRelList, BizObjAuthRelBatchForLinkReq.AuthVO authVO) {
        log.info("BizObjAuthRelServiceImpl | batch user authorize app, currentRelList:{}, bizObjAuthRelList:{}, auth:{}", currentRelList, bizObjAuthRelList, authVO);
        // 判断出删除的权限对象和新增的权限对象
        List<String> bizIdList = bizObjAuthRelList.stream().map(x -> x.getBizObjSubType() + "-" + x.getBizObjId()).collect(Collectors.toList());
        List<BizObjAuthRel> deleteList = currentRelList.stream().filter(x -> !bizIdList.contains(x.getBizObjSubType() + "-" + x.getBizObjId())).collect(Collectors.toList());

        List<String> idList = currentRelList.stream().map(x -> x.getBizObjSubType() + "-" + x.getBizObjId()).collect(Collectors.toList());
        List<BizObjAuthRel> addList = bizObjAuthRelList.stream().filter(x -> !idList.contains(x.getBizObjSubType() + "-" + x.getBizObjId())).collect(Collectors.toList());

        // 查询应用是否关联启用的知识库
        List<BizObjAuthRel> list = Lists.newArrayList();
        list.addAll(deleteList);
        list.addAll(addList);
        List<KnowledgeApplicationRef> kgAppList = knowledgeApplicationMapper.queryKgList(list);
        log.info("BizObjAuthRelServiceImpl | query kg list, param:{}, result:{}", list, kgAppList);
        if (CollectionUtils.isEmpty(kgAppList)) {
            return;
        }

        // 查询应用关联的所有权限对象，后续进一步获取用户，用于给用户授权(或取消授权)应用
        List<BizObjAuthRel> authList = queryAuthListByAppId(kgAppList);
        log.info("BizObjAuthRelServiceImpl | query auth list by appId, param:{}, result:{}", kgAppList, authList);
        // 查询组织或角色下用户列表
        Map<Long, List<UserDTO>> orgSidMap = buildOrgRoleMap(authList, authVO);
        Map<Long, List<UserDTO>> roleSidMap = buildOrgRoleMap(authList, authVO);
        Map<Long, List<UserDTO>> deptSidMap = buildOrgRoleMap(authList, authVO);

        // 取消授权的应用
        List<KnowledgeApplicationRef> delAppList = Lists.newArrayList();
        // 添加授权的应用
        List<KnowledgeApplicationRef> addAppList = Lists.newArrayList();
        kgAppList.forEach(x -> {
            deleteList.forEach(y -> {
                if (x.getApplicationType().equals(y.getBizObjSubType()) && x.getAppPrimaryId().equals(y.getBizObjId())) {
                    delAppList.add(x);
                }
            });
            addList.forEach(y -> {
                if (x.getApplicationType().equals(y.getBizObjSubType()) && x.getAppPrimaryId().equals(y.getBizObjId())) {
                    addAppList.add(x);
                }
            });
        });

        log.info("BizObjAuthRelServiceImpl | delAppList:{}, addAppList:{}", delAppList, addAppList);
        // 取消授权应用
        if (CollectionUtils.isNotEmpty(delAppList)) {
            Map<String, List<BizObjAuthRel>> appIdAuthMap = authList.stream().collect(Collectors.groupingBy(x -> x.getBizObjSubType() + "-" + x.getBizObjId()));
            for (KnowledgeApplicationRef knowledgeApplicationRef : delAppList) {
                // 如果取消的权限对象是用户，则判断该用户是否存在于该应用所关联的组织或角色下
                if (BizAuthTypeEnum.USER.getValue().equals(authVO.getAuthType()) || BizAuthTypeEnum.EXTERNAL_USER.getValue().equals(authVO.getAuthType())) {
                    List<Long> userSidList = Lists.newArrayList();

                    // 当前应用关联的组织和角色下所有用户
                    List<BizObjAuthRel> authRelList = appIdAuthMap.get(knowledgeApplicationRef.getApplicationType() + "-" + knowledgeApplicationRef.getAppPrimaryId());
                    if (CollectionUtils.isNotEmpty(authRelList)) {
                        List<BizObjAuthRel> orgList = authRelList.stream().filter(x -> BizAuthTypeEnum.ORG.getValue().equals(x.getAuthType())).collect(Collectors.toList());
                        for (BizObjAuthRel bizObjAuthRel : orgList) {
                            if (ObjectUtils.isEmpty(orgSidMap.get(bizObjAuthRel.getAuthId()))) {
                                continue;
                            }
                            userSidList.addAll(orgSidMap.get(bizObjAuthRel.getAuthId()).stream().map(UserDTO::getSid).collect(Collectors.toList()));
                        }
                        List<BizObjAuthRel> roleList = authRelList.stream().filter(x -> BizAuthTypeEnum.ROLE.getValue().equals(x.getAuthType())).collect(Collectors.toList());
                        for (BizObjAuthRel bizObjAuthRel : roleList) {
                            if (ObjectUtils.isEmpty(roleSidMap.get(bizObjAuthRel.getAuthId()))) {
                                continue;
                            }
                            userSidList.addAll(roleSidMap.get(bizObjAuthRel.getAuthId()).stream().map(UserDTO::getSid).collect(Collectors.toList()));
                        }
                        List<BizObjAuthRel> deptList = authRelList.stream().filter(x -> BizAuthTypeEnum.DEPARTMENT.getValue().equals(x.getAuthType())).collect(Collectors.toList());
                        for (BizObjAuthRel bizObjAuthRel : deptList) {
                            if (ObjectUtils.isEmpty(deptSidMap.get(bizObjAuthRel.getAuthId()))) {
                                continue;
                            }
                            userSidList.addAll(deptSidMap.get(bizObjAuthRel.getAuthId()).stream().map(UserDTO::getSid).collect(Collectors.toList()));
                        }
                    }
                    if (userSidList.contains(authVO.getAuthId())) {
                        continue;
                    }
                    // 取消授权应用
                    batchUserAuthorize(knowledgeApplicationRef, Collections.singletonList(authVO.getAuthId()), Constants.IamAuthorizeTypeEnum.DELETE.getVal());
                } else if (BizAuthTypeEnum.ORG.getValue().equals(authVO.getAuthType())
                        || BizAuthTypeEnum.ROLE.getValue().equals(authVO.getAuthType())
                        || BizAuthTypeEnum.DEPARTMENT.getValue().equals(authVO.getAuthType())) {
                    // 没有查询到组织(或角色)，则跳过该应用
                    List<UserDTO> curOrgUserList = orgSidMap.get(authVO.getAuthId());
                    List<UserDTO> curRoleUserList = roleSidMap.get(authVO.getAuthId());
                    List<UserDTO> curDeptUserList = deptSidMap.get(authVO.getAuthId());
                    if (CollectionUtils.isEmpty(curOrgUserList) && CollectionUtils.isEmpty(curRoleUserList) && CollectionUtils.isEmpty(curDeptUserList)) {
                        continue;
                    }

                    // 汇总关联的其他组织(或角色)下的所有用户
                    List<Long> allUserSidList = Lists.newArrayList();
                    List<BizObjAuthRel> authRelList = appIdAuthMap.get(knowledgeApplicationRef.getApplicationType() + "-" + knowledgeApplicationRef.getAppPrimaryId());
                    if (CollectionUtils.isNotEmpty(authRelList)) {
                        for (BizObjAuthRel bizObjAuthRel : authRelList) {
                            if (bizObjAuthRel.getAuthId().equals(authVO.getAuthId())) {
                                continue;
                            }
                            // 组织
                            List<UserDTO> orgUserList = ObjectUtils.isEmpty(orgSidMap.get(bizObjAuthRel.getAuthId())) ? Lists.newArrayList() : orgSidMap.get(bizObjAuthRel.getAuthId());
                            allUserSidList.addAll(orgUserList.stream().map(UserDTO::getSid).collect(Collectors.toList()));
                            // 角色
                            List<UserDTO> roleUserList = ObjectUtils.isEmpty(roleSidMap.get(bizObjAuthRel.getAuthId())) ? Lists.newArrayList() : roleSidMap.get(bizObjAuthRel.getAuthId());
                            allUserSidList.addAll(roleUserList.stream().map(UserDTO::getSid).collect(Collectors.toList()));
                            // 部门
                            List<UserDTO> deptUserList = ObjectUtils.isEmpty(deptSidMap.get(bizObjAuthRel.getAuthId())) ? Lists.newArrayList() : deptSidMap.get(bizObjAuthRel.getAuthId());
                            allUserSidList.addAll(deptUserList.stream().map(UserDTO::getSid).collect(Collectors.toList()));
                            // 用户
                            if (BizAuthTypeEnum.USER.getValue().equals(bizObjAuthRel.getAuthType()) || BizAuthTypeEnum.EXTERNAL_USER.getValue().equals(bizObjAuthRel.getAuthType())) {
                                allUserSidList.add(bizObjAuthRel.getAuthId());
                            }
                        }
                    }
                    List<Long> userSidList = Lists.newArrayList();
                    // 过滤掉当前组织（或角色）下存在于其他组织或角色下的用户，剩下的用户做取消授权操作
                    if (CollectionUtils.isNotEmpty(curOrgUserList)) {
                        userSidList = curOrgUserList.stream().map(UserDTO::getSid).filter(sid -> !allUserSidList.contains(sid)).collect(Collectors.toList());
                    } else if (CollectionUtils.isNotEmpty(curRoleUserList)) {
                        userSidList = curRoleUserList.stream().map(UserDTO::getSid).filter(sid -> !allUserSidList.contains(sid)).collect(Collectors.toList());
                    } else if (CollectionUtils.isNotEmpty(curDeptUserList)) {
                        userSidList = curDeptUserList.stream().map(UserDTO::getSid).filter(sid -> !allUserSidList.contains(sid)).collect(Collectors.toList());
                    }
                    // 取消授权应用
                    batchUserAuthorize(knowledgeApplicationRef, userSidList, Constants.IamAuthorizeTypeEnum.DELETE.getVal());
                }
            }
        }
        // 授权应用
        if (CollectionUtils.isNotEmpty(addAppList)) {
            batchUserAuthorizeAddApp(orgSidMap, roleSidMap, deptSidMap, authVO, addAppList, Constants.IamAuthorizeTypeEnum.INSERT.getVal());
        }
    }

    /**
     * 构建组织或角色用户列表
     *
     * @param authList 权限列表
     * @param authVO   权限对象
     * @return 返回
     */
    private Map<Long, List<UserDTO>> buildOrgRoleMap(List<BizObjAuthRel> authList, BizObjAuthRelBatchForLinkReq.AuthVO authVO) {
        Map<Long, List<UserDTO>> orgRoleMap = Maps.newHashMap();
        // 组织
        if (BizAuthTypeEnum.ORG.getValue().equals(authVO.getAuthType())) {
            List<Long> orgSidList = authList.stream().filter(x -> BizAuthTypeEnum.ORG.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).distinct().collect(Collectors.toList());
            orgSidList.add(authVO.getAuthId()); // 补上当前删除的组织
            if (CollectionUtils.isNotEmpty(orgSidList)) {
                List<OrgRoleUserDTO> orgUserList = iamService.queryUserByAllOrgSidList(orgSidList);
                log.info("BizObjAuthRelServiceImpl | query org user by orgSidList, param:{}, result:{}", JSONObject.toJSONString(orgSidList), JSONObject.toJSONString(orgUserList));
                orgRoleMap = orgUserList.stream().collect(Collectors.toMap(OrgRoleUserDTO::getOrgSid, OrgRoleUserDTO::getUsers));
            }
        }
        // 角色
        if (BizAuthTypeEnum.ROLE.getValue().equals(authVO.getAuthType())) {
            List<Long> roleSidList = authList.stream().filter(x -> BizAuthTypeEnum.ROLE.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).distinct().collect(Collectors.toList());
            roleSidList.add(authVO.getAuthId()); // 补上当前删除的角色
            if (CollectionUtils.isNotEmpty(roleSidList)) {
                List<OrgRoleUserDTO> roleUserList = iamService.queryUserByAllRoleSidList(roleSidList);
                orgRoleMap = roleUserList.stream().collect(Collectors.toMap(OrgRoleUserDTO::getRoleSid, OrgRoleUserDTO::getUsers));
            }
        }
        // 部门
        if (BizAuthTypeEnum.DEPARTMENT.getValue().equals(authVO.getAuthType())) {
            List<Long> deptSidList = authList.stream().filter(x -> BizAuthTypeEnum.DEPARTMENT.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).distinct().collect(Collectors.toList());
            deptSidList.add(authVO.getAuthId()); // 补上当前删除的部门
            if (CollectionUtils.isNotEmpty(deptSidList)) {
                List<UserDeptInfoDTO> deptUserList = eocService.queryUserByDeptSidList(deptSidList);
                log.info("BizObjAuthRelServiceImpl | query user by deptSidList, param:{}, result:{}", JSONObject.toJSONString(deptSidList), JSONObject.toJSONString(deptUserList));
                orgRoleMap = deptUserList.stream().collect(Collectors.toMap(UserDeptInfoDTO::getDeptSid, x -> {
                    List<UserDTO> userDTOList = Lists.newArrayList();
                    x.getEmps().forEach(empInfo -> {
                        if (ObjectUtils.isNotEmpty(empInfo.getEmpUserSid())) {
                            UserDTO userDTO = new UserDTO();
                            userDTO.setSid(empInfo.getEmpUserSid());
                            userDTO.setId(empInfo.getEmpUserId());
                            userDTO.setName(empInfo.getEmpUserName());
                            userDTOList.add(userDTO);
                        }
                    });
                    return userDTOList;
                }));
            }
        }
        return orgRoleMap;
    }

    /**
     * 给用户授权应用
     *
     * @param orgSidMap  组织下用户信息
     * @param roleSidMap 角色下用户信息
     * @param authVO     授权的对象
     * @param addAppList 授权的应用
     */
    private void batchUserAuthorizeAddApp(Map<Long, List<UserDTO>> orgSidMap, Map<Long, List<UserDTO>> roleSidMap, Map<Long, List<UserDTO>> deptSidMap, BizObjAuthRelBatchForLinkReq.AuthVO authVO,
                                          List<KnowledgeApplicationRef> addAppList, Integer actionType) {
        for (KnowledgeApplicationRef knowledgeApplicationRef : addAppList) {
            if (BizAuthTypeEnum.USER.getValue().equals(authVO.getAuthType()) || BizAuthTypeEnum.EXTERNAL_USER.getValue().equals(authVO.getAuthType())) {
                batchUserAuthorize(knowledgeApplicationRef, Collections.singletonList(authVO.getAuthId()), Constants.IamAuthorizeTypeEnum.INSERT.getVal());
            } else if (BizAuthTypeEnum.ORG.getValue().equals(authVO.getAuthType())
                    || BizAuthTypeEnum.ROLE.getValue().equals(authVO.getAuthType())
                    || BizAuthTypeEnum.DEPARTMENT.getValue().equals(authVO.getAuthType())) {
                List<Long> userSidList = Lists.newArrayList();
                // 组织
                List<UserDTO> orgUserDTOList = ObjectUtils.isEmpty(orgSidMap.get(authVO.getAuthId())) ? Lists.newArrayList() : orgSidMap.get(authVO.getAuthId());
                if (CollectionUtils.isNotEmpty(orgUserDTOList)) {
                    userSidList = orgUserDTOList.stream().map(UserDTO::getSid).collect(Collectors.toList());
                }
                // 角色
                List<UserDTO> roleUserDTOList = ObjectUtils.isEmpty(roleSidMap.get(authVO.getAuthId())) ? Lists.newArrayList() : roleSidMap.get(authVO.getAuthId());
                if (CollectionUtils.isNotEmpty(roleUserDTOList)) {
                    userSidList = roleUserDTOList.stream().map(UserDTO::getSid).collect(Collectors.toList());
                }
                // 部门
                List<UserDTO> deptUserDTOList = ObjectUtils.isEmpty(deptSidMap.get(authVO.getAuthId())) ? Lists.newArrayList() : deptSidMap.get(authVO.getAuthId());
                if (CollectionUtils.isNotEmpty(deptUserDTOList)) {
                    userSidList = deptUserDTOList.stream().map(UserDTO::getSid).collect(Collectors.toList());
                }
                // 授权
                batchUserAuthorize(knowledgeApplicationRef, userSidList, actionType);
            }
        }
    }

    /**
     * 授权应用
     *
     * @param knowledgeApplicationRef 应用信息
     * @param userSidList             用户列表
     */
    private void batchUserAuthorize(KnowledgeApplicationRef knowledgeApplicationRef, List<Long> userSidList, Integer actionType) {
        if (CollectionUtils.isEmpty(userSidList)) {
            return;
        }
        // cac接口限制用户列表最大为100个
        List<List<Long>> userPartList = Lists.partition(userSidList, 100);
        for (List<Long> tempList : userPartList) {
            log.info("BizObjAuthRelServiceImpl | start batch user authorize app, appId:{}, userSid:{}, actionType:{}", knowledgeApplicationRef.getKnowledgeId(), tempList, actionType);
            iamService.batchUserAuthorizeApp(knowledgeApplicationRef.getKnowledgeId(), tempList, actionType);
        }
    }

    /**
     * 查询当前用户 + 当前角色 + 当前组织的所有帆软报表
     *
     * @param bizObjType 业务类型
     * @return
     */
    @Override
    public List<BizObjAuthRel> qryBizObjAuthRel(Integer bizObjType) {
        String userId = AppAuthContextHolder.getContext().getAuthoredUser().getUserId();
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        Long userSid = AppAuthContextHolder.getContext().getAuthoredUser().getSid();
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        IamUserAuthInfo iamUserAuthInfo = iamUserService.getUserAuthSids(userId, tenantId, userToken);
        QueryWrapper cond = new QueryWrapper<BizObjAuthRel>().eq("biz_obj_type", bizObjType);

        // 拼接组织条件
        StringBuilder authCond = new StringBuilder();
        if (iamUserAuthInfo!=null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getOrgSids())) {
            authCond.append("(auth_type = 1 and auth_id in(" + Joiner.on(",").join(iamUserAuthInfo.getOrgSids()) + "))");
        }

        // 拼接角色条件
        if (iamUserAuthInfo!=null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getRoleSids())) {
            if (StringUtils.isNotEmpty(authCond.toString())) {
                authCond.append("or (auth_type = 2 and auth_id in(" + Joiner.on(",").join(iamUserAuthInfo.getRoleSids()) + "))");
            } else {
                authCond.append("(auth_type = 2 and auth_id in(" + Joiner.on(",").join(iamUserAuthInfo.getRoleSids()) + "))");
            }
        }

        // 拼接部门条件
        if (iamUserAuthInfo!=null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getDeptSids())) {
            if (StringUtils.isNotEmpty(authCond.toString())) {
                authCond.append("or (auth_type = 5 and auth_id in(" + Joiner.on(",").join(iamUserAuthInfo.getDeptSids()) + "))");
            } else {
                authCond.append("(auth_type = 5 and auth_id in (" + Joiner.on(",").join(iamUserAuthInfo.getDeptSids()) + "))");
            }
        }

        // 拼接用户条件
        if (ObjectUtils.isNotEmpty(userSid)) {
            if (StringUtils.isNotEmpty(authCond.toString())) {
                authCond.append("or (auth_type = 3 and auth_id = " + userSid + ")");
            } else {
                authCond.append("(auth_type = 3 and auth_id = " + userSid + ")");
            }
        }
        if (StringUtils.isNotEmpty(authCond.toString())) {
            cond.apply("(" + authCond + ")");
        }

        List<BizObjAuthRel> bizObjAuthRelList = this.getBaseMapper().selectList(cond);
        log.debug("FineReportService query business auth relation result:{}", JSONObject.toJSONString(bizObjAuthRelList));

        return bizObjAuthRelList;
    }

    /**
     * 查询应用关联的权限对象
     *
     * @param bizObjAuthRelList 应用id
     * @return 返回
     */
    @Override
    public List<BizObjAuthRel> queryAuthListByAppId(List<KnowledgeApplicationRef> bizObjAuthRelList) {
        return bizObjAuthRelMapper.queryAuthListByAppId(bizObjAuthRelList);
    }
}
