package com.digiwin.athena.semc.service.common.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.enums.BizObjectTypeEnum;
import com.digiwin.athena.semc.common.enums.FolderLevelEnum;
import com.digiwin.athena.semc.common.enums.FolderTypeEnum;
import com.digiwin.athena.semc.entity.common.BizObjAuthRel;
import com.digiwin.athena.semc.entity.common.Folder;
import com.digiwin.athena.semc.entity.portal.FineReport;
import com.digiwin.athena.semc.mapper.common.FolderMapper;
import com.digiwin.athena.semc.service.common.IBizObjAuthRelService;
import com.digiwin.athena.semc.service.common.IFolderService;
import com.digiwin.athena.semc.service.portal.IFineReportService;
import com.digiwin.athena.semc.vo.common.FolderLevelVo;
import com.digiwin.athena.semc.vo.common.MoveFolderChildVo;
import com.digiwin.athena.semc.vo.common.MoveFolderVo;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @description: 树目录实现类
 * @createDate: 2023/5/26 13:47
 * @author: sunyfa
 */
@Slf4j
@Service
public class FolderServiceImpl extends ServiceImpl<FolderMapper, Folder> implements IFolderService {
    @Resource
    private IFineReportService fineReportService;

    @Resource
    private IBizObjAuthRelService bizObjAuthRelService;

    /**
     * @param
     * @return
     * @description: 新增/修改目录
     * @author: sunyfa
     */
    @Override
    @Transactional
    public int addOrModFolder(Folder folder) {
        // 新增目录
        if (Objects.isNull(folder.getId())) {
            // 新增目录需要计算目录顺序
            Integer maxOrder = baseMapper.queryMaxOrder(folder.getParentFolderId());
            log.debug("FolderService query maxOrder result:{}", maxOrder);

            // 新增一级目录
            if (Objects.isNull(folder.getParentFolderId())) {
                folder.setFolderOrder((maxOrder == null ? 0 : maxOrder) + 1);
            } else {    // 新增非一级目录，如果父目录下没有目录，则在父目录顺序的基础上排序
                if (Objects.isNull(maxOrder)) {
                    Folder parent = baseMapper.selectById(folder.getParentFolderId());
                    folder.setFolderOrder(parent.getFolderOrder() * 10 + 1);
                } else {
                    folder.setFolderOrder(maxOrder + 1);
                }
            }
            folder.setCreateUserId(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
            folder.setObjType(FolderTypeEnum.FINE_REPORT.getValue());
            baseMapper.insert(folder);
        } else {    // 更新目录
            updateById(folder);
        }
        return Integer.parseInt(String.valueOf(folder.getId()));
    }

    /**
     * @param
     * @return
     * @description: 删除目录
     * @author: sunyfa
     */
    @Override
    @Transactional
    public void delFolder(Long id, int folderType) {
        if (FolderTypeEnum.FINE_REPORT.getValue() == folderType) {
            // 查询当前目录及所有子目录
            List<Folder> folderList = baseMapper.qryCurrentAndChildFolderList(folderType, id);
            List<Long> folderIdList = folderList.stream().map(Folder::getId).collect(Collectors.toList());
            log.debug("FolderService folder id:{} query children result:{}", id, JSONObject.toJSONString(folderIdList));

            // 查询当前目录与子目录所有报表
            List<FineReport> fineReportList = fineReportService.getBaseMapper().selectList(new QueryWrapper<FineReport>().in("folder_id", folderIdList));
            // 存在报表
            if (CollectionUtils.isNotEmpty(fineReportList)) {
                List<Long> fineReportIdList = fineReportList.stream().map(FineReport::getId).collect(Collectors.toList());
                // 删除权限数据
                bizObjAuthRelService.getBaseMapper().delete(new QueryWrapper<BizObjAuthRel>()
                        .in("biz_obj_id", fineReportIdList).eq("biz_obj_type", BizObjectTypeEnum.FINE_REPORT.getValue()));
                // 删除报表数据
                fineReportService.getBaseMapper().deleteBatchIds(fineReportIdList);
            }
            // 删除目录
            baseMapper.deleteBatchIds(folderIdList);
        }
    }

    /**
     * 移动目录
     *
     * @param childrenFolderMap 非一级目录
     * @return ResponseEntity
     */
    @Override
    @Transactional
    public void moveFolder(List<MoveFolderVo> moveFolderVoList, Map<Long, MoveFolderChildVo> childrenFolderMap) {
        // 一级目录
        Map<Long, MoveFolderVo> firstFolderMap = Maps.newHashMap();
        moveFolderVoList.forEach(moveFolderVo -> firstFolderMap.put(moveFolderVo.getId(), moveFolderVo));

        // 查询数据库中一级目录
        List<Folder> firstFolderList = baseMapper.selectBatchIds(firstFolderMap.keySet());
        firstFolderList.forEach(firstFolder -> {
            firstFolder.setParentFolderId(null);
            firstFolder.setFolderOrder(firstFolderMap.get(firstFolder.getId()).getFolderOrder());
            firstFolder.setLevel(FolderLevelEnum.FIRST.getValue());
            // firstFolder.setPathCode(String.valueOf(firstFolder.getId()));
        });
        saveOrUpdateBatch(firstFolderList);

        // 更新非一级目录的父目录id、目录顺序
        if (CollectionUtils.isNotEmpty(childrenFolderMap.keySet())) {
            // 查询数据库中非一级目录
            List<Folder> childrenFolderList = baseMapper.selectBatchIds(childrenFolderMap.keySet());
            childrenFolderList.forEach(childFolder -> {
                Long childFolderId = childFolder.getId();
                childFolder.setParentFolderId(childrenFolderMap.get(childFolderId).getParentFolderId());
                childFolder.setFolderOrder(childrenFolderMap.get(childFolderId).getFolderOrder());
                childFolder.setLevel(childrenFolderMap.get(childFolderId).getLevel());
                // childFolder.setPathCode(parentFolderId + "/" + childFolderId);
            });
            saveOrUpdateBatch(childrenFolderList);
        }
    }

    /**
     * @param
     * @return
     * @description: 报表目录查询
     * @author: sunyfa
     */
    @Override
    public List<Folder> qryFolder(int folderType) {
        if (FolderTypeEnum.FINE_REPORT.getValue() == folderType) {
            // 查询当前树的一级目录
            List<Folder> parentList = baseMapper.selectList(
                    new QueryWrapper<Folder>()
                            .isNull("parent_folder_id")
                            .eq("obj_type", folderType)
                            .orderByAsc("folder_order"));
            log.debug("FineReporterService query parent result:{}", JSONObject.toJSONString(parentList));

            if (CollectionUtils.isNotEmpty(parentList)) {
                // 查询每个根目录的子目录
                parentList.stream().forEach(parent -> {
                    List<Folder> children = baseMapper.selectList(
                            new QueryWrapper<Folder>()
                                    .eq("parent_folder_id", parent.getId())
                                    .eq("obj_type", folderType)
                                    .orderByAsc("folder_order"));
                    log.debug("FineReporterService folder id:{} query children result:{}", parent.getId(), JSONObject.toJSONString(children));

                    parent.setChildren(children);
                });
            }

            log.debug("FineReporterService query folder tree result:{}", JSONObject.toJSONString(parentList));
            return parentList;
        }

        return null;
    }

    /**
     * @param
     * @return
     * @description: 校验是否存在子目录
     * @author: sunyfa
     */
    @Override
    public boolean isExistChild(Long parentFolderId, int folderType) {
        return exists(new QueryWrapper<Folder>().eq("obj_type", folderType).eq("parent_folder_id", parentFolderId));
    }

    /**
     * 根据目录id查询所有当前目录及父目录的层级
     *
     * @param type 业务类型
     * @param folderId 目录id
     * @return
     */
    @Override
    public List<FolderLevelVo> qryCurrentAndParentFolderList(int type, long folderId){
        return baseMapper.qryCurrentAndParentFolderList(type, folderId);
    }

    public boolean exists(Wrapper<Folder> queryWrapper) {
        Long count = baseMapper.selectCount(queryWrapper);
        return null != count && count > 0;
    }
}