package com.digiwin.athena.semc.service.device.impl;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.BizException;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.dto.device.DeviceAuthorizeVO;
import com.digiwin.athena.semc.dto.device.DeviceBindConfigDTO;
import com.digiwin.athena.semc.dto.device.DeviceDTO;
import com.digiwin.athena.semc.entity.device.DeviceInfo;
import com.digiwin.athena.semc.entity.device.UserBindDevice;
import com.digiwin.athena.semc.mapper.device.UserBindDeviceMapper;
import com.digiwin.athena.semc.service.cache.LockClient;
import com.digiwin.athena.semc.service.device.IDeviceAuthService;
import com.digiwin.athena.semc.service.device.IDeviceBindConfigService;
import com.digiwin.athena.semc.service.device.IDeviceInfoService;
import com.digiwin.athena.semc.service.device.IUserBindDeviceService;
import com.digiwin.athena.semc.util.Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.redisson.api.RLock;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @Author jf gui
 */
@Slf4j
@Service
public class DeviceAuthServiceImpl implements IDeviceAuthService {

    private static final String LOCK_KEY="try:auto:bind";

    @Resource
    private IDeviceInfoService deviceInfoService;

    @Resource
    private UserBindDeviceMapper userBindDeviceMapper;

    @Resource
    private IDeviceBindConfigService iDeviceBindConfigService;

    @Resource
    private IUserBindDeviceService iUserBindDeviceService;

    @Resource
    private LockClient lockClient;

    @Resource
    private MessageUtils messageUtils;
    /**
     * 授权设备方法
     * 根据设备信息和系统配置，决定设备是否被授权
     *
     * @param device 设备信息对象，包含设备ID等数据
     * @return 返回设备授权视图对象，包含授权状态和设备绑定配置等信息
     */
    @Override
    public DeviceAuthorizeVO authorize(DeviceDTO device) {
        // 初始化设备授权视图对象
        DeviceAuthorizeVO deviceAuthorizeVO = new DeviceAuthorizeVO();
        // 获取设备绑定配置
        DeviceBindConfigDTO config = iDeviceBindConfigService.getConfig();
        // 设置设备绑定配置到视图对象
        deviceAuthorizeVO.setBindConfig(config);

        // 如果设备绑定功能未启用，则直接授权设备
        if(!BooleanUtils.isTrue(config.getBindEnabled())){
            deviceAuthorizeVO.setDeviceAuthorize(true);
            return deviceAuthorizeVO;
        }
        // 默认设备未授权
        deviceAuthorizeVO.setDeviceAuthorize(false);
        // 获取当前用户ID
        String userId = Utils.getUserId();
        // 根据设备ID查询设备信息
        DeviceInfo deviceInfo = deviceInfoService.selectByTenantIdAndDeviceId(device.getDeviceId());
        // 判断当前用户是否已绑定该设备
        UserBindDevice userBindDevice = Objects.isNull(deviceInfo)?null: userBindDeviceMapper.findByUserAndDeviceId(userId, device.getDeviceId());
        boolean existBind = Objects.nonNull(userBindDevice) && userBindDevice.getStatus() == 0;

        // 如果已绑定，更新设备信息并返回授权视图对象
        if(existBind) {
            deviceAuthorizeVO.setDeviceAuthorize(true);
            deviceAuthorizeVO.setDeviceExistBinding(true);
            deviceAuthorizeVO.setAccountExistBinding(true);
            deviceAuthorizeVO.setDeviceExistStatus(true);
            // 更新设备信息
            deviceInfoService.updateDeviceInfo(deviceInfo.getId(),device);
            return deviceAuthorizeVO;
        }
        // 检查用户是否有其他绑定的设备
        if(Objects.nonNull(userBindDevice)){
            deviceAuthorizeVO.setAccountExistBinding(true);
            deviceAuthorizeVO.setDeviceExistStatus(userBindDevice.getStatus()==0);
        }else {
            Integer counted = userBindDeviceMapper.countByTenantIdAndUserIdOrTerminalType(userId, device.getTerminalType());
            deviceAuthorizeVO.setAccountExistBinding(Objects.nonNull(counted) && counted > 0);
        }
        Integer counted1 = userBindDeviceMapper.countByTenantIdAndDeviceId(device.getDeviceId());
        deviceAuthorizeVO.setDeviceExistBinding(Objects.nonNull(counted1) && counted1 >0);
        // 返回设备授权视图对象
        return deviceAuthorizeVO;
    }


    /**
     * 尝试自动绑定设备
     * 此方法用于在用户尝试绑定设备时，检查设备或账户是否已经绑定，并在条件允许的情况下自动绑定设备
     *
     * @param deviceAuthorize 设备授权信息对象，包含绑定结果的反馈
     * @param device 设备信息对象，包含设备的详细信息
     */
    private void tryAutoBind(DeviceAuthorizeVO deviceAuthorize,DeviceDTO device){
        if(deviceAuthorize.getAccountExistBinding()){
            return;
        }
        RLock lock = null;
        // 获取当前授权用户信息
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        // 生成锁的键值，确保同一租户同一用户的操作互斥
        String key=LOCK_KEY+authoredUser.getTenantId()+":"+authoredUser.getUserId();
        try{
            lock = lockClient.getLock(key);
            // 尝试获取锁，以防止并发绑定问题
            boolean lockResult = lock.tryLock(3, 5, TimeUnit.SECONDS);
            if (!lockResult) {
                deviceAuthorize.setDeviceExistBinding(true);
                log.error("tryAutoBind tryLock fail key:{}",key);
            }else {
                // 查询设备是否绑定到其他账户
                Integer counted1 = userBindDeviceMapper.countByTenantIdAndDeviceId(device.getDeviceId());
                boolean deviceExistBind = Objects.nonNull(counted1) && counted1 >0;
                deviceAuthorize.setDeviceExistBinding(deviceExistBind);
                if(!deviceExistBind){
                    // 如果设备和账户均未绑定，调用服务进行自动绑定
                    iUserBindDeviceService.autoBind(device);
                    deviceAuthorize.setDeviceAuthorize(true);
                    deviceAuthorize.setDeviceExistBinding(true);
                    deviceAuthorize.setAccountExistBinding(true);
                }
            }
        } catch (InterruptedException e) {
            // 日志记录中断异常
            log.error("tryAutoBind is error key:{}",key,e);
        }finally {
            // 释放锁，确保资源清理
            lockClient.unlock(lock);
        }

    }


    @Override
    public Boolean autoBind(DeviceDTO device) {
        RLock lock = null;
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String key=LOCK_KEY+authoredUser.getTenantId()+":"+authoredUser.getUserId();
        try{
            lock = lockClient.getLock(key);
            boolean lockResult = lock.tryLock(3, 5, TimeUnit.SECONDS);
            if (!lockResult) {
                log.error("autoBind tryLock fail key:{}",key);
                return false;
            }else {
                DeviceBindConfigDTO config = iDeviceBindConfigService.getConfig();
                if(BooleanUtils.isTrue(config.getBindEnabled()) && BooleanUtils.isTrue(config.getBindDeviceStatus())){
                    Integer counted = userBindDeviceMapper.countByTenantIdAndUserIdOrTerminalType(Utils.getUserId(), device.getTerminalType());
                    if(Objects.nonNull(counted) && counted>0){
                        log.error("autoBind fail Account Exist Binding key:{}",key);
                        return false;
                    }
                    iUserBindDeviceService.autoBind(device);
                    return true;
                }else {
                    log.error("autoBind fail DeviceBindConfig Not Supported key:{}",key);
                    return false;
                }
            }
        }catch (InterruptedException e) {
            log.error("autoBind fail InterruptedException key:{}",key,e);
            throw BizException.getDefaultBizException(ErrorCodeConstant.SYSTEM_ERROR,
                    messageUtils.getMessage(I18NKey.SYSTEM_ERROR));
        }finally {
            lockClient.unlock(lock);
        }
    }
}
