package com.digiwin.athena.semc.service.device.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelReader;
import com.alibaba.excel.event.SyncReadListener;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.BizException;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.enums.FolderTypeEnum;
import com.digiwin.athena.semc.dto.device.DeviceBindConfigDTO;
import com.digiwin.athena.semc.dto.device.DeviceBindImportDTO;
import com.digiwin.athena.semc.dto.device.DeviceBindImportTemplateVo;
import com.digiwin.athena.semc.entity.common.ImportRecord;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.RoleDTO;
import com.digiwin.athena.semc.proxy.mdc.MdcService;
import com.digiwin.athena.semc.service.device.DeviceBindDataWriteHandler;
import com.digiwin.athena.semc.service.device.IDeviceBindConfigService;
import com.digiwin.athena.semc.service.device.IUserBindDeviceFileService;
import com.digiwin.athena.semc.service.device.IUserBindDeviceService;
import com.digiwin.athena.semc.service.portal.IImportRecordService;
import com.digiwin.athena.semc.service.utils.ValidationMsgUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.portal.ImportExcelResp;
import com.digiwin.dap.middleware.dmc.DMC;
import com.digiwin.dap.middleware.dmc.model.ShareInfo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @Description
 * @Author cai chao
 * @Date 2025/7/24 10:58
 */
@Slf4j
@Service
public class UserBindDeviceFileServiceImpl implements IUserBindDeviceFileService {

    @Resource
    private IamService iamService;

    @Resource
    private MdcService mdcService;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private IDeviceBindConfigService iDeviceBindConfigService;

    @Resource
    private IImportRecordService importRecordService;

    @Resource
    private IUserBindDeviceService iUserBindDeviceService;

    public ImportExcelResp importRecords(DeviceBindImportDTO req) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        //查询开关配置
        DeviceBindConfigDTO config = iDeviceBindConfigService.getConfig();
        Assert.isTrue(config.getBindEnabled(), //设备未打开，操作失败
                () -> BizException.getDefaultBizException(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage("error.message.user.device.config.empty")));

        //解析文件
        List<DeviceBindImportTemplateVo> deviceBindVos = parse(req.getFileId());
        int successNum = 0;
        int failNum = 0;
        if (CollUtil.isNotEmpty(deviceBindVos)) {
            for (int i = 0; i < deviceBindVos.size(); i++) {
                DeviceBindImportTemplateVo deviceBindVo = deviceBindVos.get(i);
                try {
                    if (StrUtil.isBlank(deviceBindVo.getFailMsg())) {
                        iUserBindDeviceService.importDeviceBind(deviceBindVo, authoredUser);
                        successNum++;
                    } else {
                        failNum++;
                    }
                } catch (Exception e) {
                    log.error("user.device.import error,deviceBindVo:{}", JSONUtil.toJsonStr(deviceBindVo), e);
                    deviceBindVo.setFailMsg(e.getMessage());
                    failNum++;
                }
            }
        }
        File file = new File("设备导入模板_错误日志.xlsx");
        EasyExcel.write(file, DeviceBindImportTemplateVo.class)
                .relativeHeadRowIndex(1)
                .registerWriteHandler(Utils.initHorizontalCell())
                .registerWriteHandler(new DeviceBindDataWriteHandler())
                .sheet("设备导入").doWrite(deviceBindVos);
        ShareInfo shareInfo = importRecordService.uploadExcel(file);

        //保存上传文件记录
        ImportRecord importRecord = new ImportRecord();
        importRecord.setFileId(req.getFileId());
        importRecord.setFileName(StrUtil.isNotBlank(req.getFileName()) ? req.getFileName() : shareInfo.getFileName());
        importRecord.setSuccessCount(successNum);
        importRecord.setFailCount(failNum);
        importRecord.setObjType(FolderTypeEnum.DEVICE_REPORT.getValue());
        importRecord.setTenantId(authoredUser.getTenantId());
        importRecord.setCreateUserId(authoredUser.getUserId());
        importRecord.setCreateUserName(authoredUser.getUserName());
        importRecord.setFailFileId(shareInfo.getFileId());
        importRecord.setFailFileName(shareInfo.getFileName());
        importRecord.setFailFileUrl(shareInfo.getUrl());
        importRecordService.insertImportRecord(importRecord);

        //返回结果
        ImportExcelResp importExcelResp = new ImportExcelResp();
        importExcelResp.setSuccessCount(successNum);
        importExcelResp.setFailCount(failNum);
        importExcelResp.setFailFileId(shareInfo.getFileId());
        importExcelResp.setFailFileName(shareInfo.getFileName());
        importExcelResp.setFailFileUrl(shareInfo.getUrl());
        return importExcelResp;
    }

    private List<DeviceBindImportTemplateVo> parse(String fileId) {
        DMC dmcInstance = mdcService.buildDmc();
        List<DeviceBindImportTemplateVo> excelVoList;
        try {
            byte[] byteArray = dmcInstance.download(fileId);
            ByteArrayInputStream input = new ByteArrayInputStream(byteArray);
            SyncReadListener excelListener = new SyncReadListener();
            // 读取sheet
            ExcelReader excelReader = EasyExcelFactory.read(input).build();
            excelReader.read(EasyExcelFactory.readSheet(0).head(DeviceBindImportTemplateVo.class).headRowNumber(2).registerReadListener(excelListener).build());
            excelReader.finish();
            excelVoList = mdcService.doReadSync(excelListener);

            validateDeviceBindings(excelVoList);
        } catch (Exception e) {
            log.error("userBindDevice read excel occur error", e);
            return new ArrayList<>();
        }
        return excelVoList;
    }

    private void validateDeviceBindings(List<DeviceBindImportTemplateVo> deviceBindImportTemplateVos) {
        if (CollUtil.isNotEmpty(deviceBindImportTemplateVos)) {
            //租户下所有用户信息
            List<RoleDTO> roleDTOS = iamService.queryAllUserList();
            Map<String, String> userIdNameMap = CollUtil.emptyIfNull(roleDTOS).stream().collect(Collectors.toMap(RoleDTO::getId, RoleDTO::getName));
            //校验
            for (int i = 0; i < deviceBindImportTemplateVos.size(); i++) {
                DeviceBindImportTemplateVo deviceBindVo = deviceBindImportTemplateVos.get(i).trim();
                BaseResultDTO<String> validated = ValidationMsgUtil.validateAll(deviceBindVo, ",");
                if (!validated.isOK()) {
                    deviceBindVo.setFailMsg(StrUtil.toString(validated.getErrorMessage()));
                } else {
                    String userId = deviceBindVo.getUserId();
                    if (userIdNameMap.containsKey(userId)) {
                        deviceBindVo.setUserName(userIdNameMap.get(userId));
                    }else {
                        deviceBindVo.setFailMsg(ValidationMsgUtil.getMessage("{Athena.Account}{Not.exist}"));
                    }
                }
            }
        }
    }
}
