package com.digiwin.athena.semc.service.device.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.common.BizException;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.dto.device.*;
import com.digiwin.athena.semc.dto.mq.MessageDO;
import com.digiwin.athena.semc.entity.device.DeviceInfo;
import com.digiwin.athena.semc.entity.device.UserBindDevice;
import com.digiwin.athena.semc.mapper.device.UserBindDeviceMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.UserInfoDTO;
import com.digiwin.athena.semc.service.cache.LockClient;
import com.digiwin.athena.semc.service.device.IDeviceBindConfigService;
import com.digiwin.athena.semc.service.device.IDeviceInfoService;
import com.digiwin.athena.semc.service.device.IUserBindDeviceService;
import com.digiwin.athena.semc.service.mq.MessageSendService;
import com.digiwin.athena.semc.service.utils.ValidationMsgUtil;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author jf gui
 */
@Slf4j
@Service
public class UserBindDeviceServiceImpl extends ServiceImpl<UserBindDeviceMapper, UserBindDevice> implements IUserBindDeviceService {

    @Autowired
    private MessageUtils messageUtils;

    @Resource
    private LockClient lockClient;

    @Resource
    private IDeviceInfoService iDeviceInfoService;

    @Resource
    private IDeviceBindConfigService iDeviceBindConfigService;

    @Resource
    private MessageSendService messageSendService;

    @Resource
    private IamService iamService;

    private static final String LOCK_KEY = "try:auto:device_bind_update";

    @Override
    public PageInfoResp<DeviceBindRecordVO> list(DeviceBindQueryDTO request) {
        Page page = new Page<>(request.getPageNum(), request.getPageSize());
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        IPage<DeviceBindRecordVO> deviceBindRecordVOIPage = baseMapper.selectDeviceBindPage(page, request, tenantId);

        PageInfoResp pageInfoResp = new PageInfoResp();
        pageInfoResp.setList(deviceBindRecordVOIPage.getRecords());
        pageInfoResp.setTotalRecords((int) deviceBindRecordVOIPage.getTotal());
        pageInfoResp.setTotalPages((int) deviceBindRecordVOIPage.getPages());
        return pageInfoResp;
    }

    @Override
    @Transactional
    public DeviceBindRecordVO save(DeviceBindSaveDTO record) {
        //绑定开关判断
        DeviceBindConfigDTO config = iDeviceBindConfigService.getConfig();
        Assert.isTrue(config.getBindEnabled(), //设备未打开
                () -> BizException.getDefaultBizException(ErrorCodeConstant.SYSTEM_ERROR,
                        messageUtils.getMessage("error.message.user.device.config.empty")));

        record.setUserName(getUserNameAndCheck(record.getUserId()));
        DeviceBindRecordVO deviceBindRecordVO = saveAction(record, AppAuthContextHolder.getContext().getAuthoredUser());
        sendMqtt(deviceBindRecordVO.getTenantId(), deviceBindRecordVO.getUserId());
        return deviceBindRecordVO;
    }

    private DeviceBindRecordVO saveAction(DeviceBindSaveDTO record, AuthoredUser authoredUser) {
        Integer existBind = baseMapper.countByUserIdAndDeviceId(record.getUserId(), record.getDeviceId());
        Assert.isFalse(existBind > 0, //数据已存在
                () -> BizException.getDefaultBizException(ErrorCodeConstant.SYSTEM_ERROR,
                        messageUtils.getMessage("error.message.already.exists")));
        //新增设备信息
        DeviceInfo deviceInfo = iDeviceInfoService.selectByTenantIdAndDeviceId(record.getDeviceId());
        if (Objects.isNull(deviceInfo)) {
            deviceInfo = iDeviceInfoService.save(record);
        } else {
            deviceInfo.setTerminalType(record.getTerminalType());
            iDeviceInfoService.updateDeviceTerminalType(deviceInfo.getId(), record.getTerminalType());
        }
        UserBindDevice newUserBindDevice = new UserBindDevice();
        newUserBindDevice.setDeviceId(deviceInfo.getDeviceId());
        newUserBindDevice.setTenantId(authoredUser.getTenantId());
        newUserBindDevice.setUserId(record.getUserId());
        newUserBindDevice.setUserName(record.getUserName());
        newUserBindDevice.setStatus(record.getStatus());
        newUserBindDevice.setCreateUserId(authoredUser.getUserId());
        newUserBindDevice.setCreateUserName(authoredUser.getUserName());
        newUserBindDevice.setCreateTime(DateUtils.getNowTime());
        newUserBindDevice.setModifyUserId(authoredUser.getUserId());
        newUserBindDevice.setModifyUserName(authoredUser.getUserName());
        newUserBindDevice.setModifyTime(DateUtils.getNowTime());
        this.baseMapper.insert(newUserBindDevice);
        return DeviceBindRecordVO.build(newUserBindDevice, deviceInfo);
    }


    @Override
    @Transactional
    public DeviceBindRecordVO update(DeviceBindUpdateDTO update) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        update.setUserName(getUserNameAndCheck(update.getUserId()));

        //查询配置
        DeviceBindConfigDTO config = iDeviceBindConfigService.getConfig();
        Assert.isTrue(config.getBindEnabled(),
                () -> BizException.getDefaultBizException(ErrorCodeConstant.SYSTEM_ERROR,
                        messageUtils.getMessage("error.message.user.device.config.empty")));
        //原数据是否存在
        UserBindDevice userBindDevice = baseMapper.selectById(update.getId());
        Assert.notNull(userBindDevice,
                () -> BizException.getDefaultBizException(ErrorCodeConstant.SYSTEM_ERROR,
                        messageUtils.getMessage("error.message.not.exists")));

        String key = LOCK_KEY + userBindDevice.getId();
        RLock lock = null;
        try {
            lock = lockClient.getLock(key);
            boolean lockResult = lock.tryLock(3, 5, TimeUnit.SECONDS);
            if (!lockResult) {
                log.error("tryAutoBind tryLock fail key:{}", key);
            } else {
                if (!userBindDevice.getUserId().equals(update.getUserId()) || !userBindDevice.getDeviceId().equals(update.getDeviceId())) {
                    Integer existBind = baseMapper.countByUserIdAndDeviceId(update.getUserId(), update.getDeviceId());
                    Assert.isFalse(existBind > 0, //数据已存在
                            () -> BizException.getDefaultBizException(ErrorCodeConstant.SYSTEM_ERROR,
                                    messageUtils.getMessage("error.message.already.exists")));
                }
                DeviceInfo deviceInfo = iDeviceInfoService.selectByTenantIdAndDeviceId(update.getDeviceId());
                if (Objects.isNull(deviceInfo)) {
                    deviceInfo = iDeviceInfoService.save(update);
                } else {
                    if (!update.getTerminalType().equals(deviceInfo.getTerminalType())) {
                        deviceInfo.setTerminalType(update.getTerminalType());
                        iDeviceInfoService.updateDeviceTerminalType(deviceInfo.getId(), update.getTerminalType());
                    }
                }
                UserBindDevice newUserBindDevice = new UserBindDevice();
                newUserBindDevice.setId(update.getId());
                newUserBindDevice.setDeviceId(update.getDeviceId());
                newUserBindDevice.setUserId(update.getUserId());
                newUserBindDevice.setStatus(update.getStatus());
                newUserBindDevice.setUserName(update.getUserName());
                newUserBindDevice.setModifyUserId(authoredUser.getUserId());
                newUserBindDevice.setModifyUserName(authoredUser.getUserName());
                newUserBindDevice.setModifyTime(DateUtils.getNowTime());
                baseMapper.updateById(newUserBindDevice);
                if (!userBindDevice.getUserId().equals(newUserBindDevice.getUserId())) {
                    //通知原账号重新认证
                    sendMqtt(userBindDevice.getTenantId(), userBindDevice.getUserId());
                }
                sendMqtt(userBindDevice.getTenantId(), update.getUserId());
                return DeviceBindRecordVO.build(newUserBindDevice, deviceInfo);
            }
        } catch (InterruptedException e) {
            log.error("tryAutoBind is error key:{}", key, e);
        } finally {
            lockClient.unlock(lock);
        }
        return null;
    }

    private String getUserNameAndCheck(String userId) {
        try {
            UserInfoDTO user = iamService.queryUser(userId, null, Utils.getUserToken());
            if (Objects.isNull(user) || StringUtils.isBlank(user.getId())) {
                String key = ValidationMsgUtil.buildKeys("Athena.Account", "Not.exist");
                throw BizException.getDefaultBizException(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, ValidationMsgUtil.getMessage(key));
            }
            return user.getName();
        } catch (BizException e) {
            log.error("getUserNameAndCheck BizException userId:{}", userId, e);
            throw e;
        } catch (Exception e) {
            log.error("getUserNameAndCheck is error userId:{}", userId, e);
            throw BizException.getDefaultBizException(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, ValidationMsgUtil.getMessage(ValidationMsgUtil.buildKeys("Athena.Account", "Not.exist")));
        }
    }

    @Override
    public Boolean updateValidStatus(DeviceBindUpdateStatusDTO update) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        DeviceBindConfigDTO config = iDeviceBindConfigService.getConfig();
        Assert.isTrue(config.getBindEnabled(),
                () -> BizException.getDefaultBizException(ErrorCodeConstant.SYSTEM_ERROR,
                        messageUtils.getMessage("error.message.user.device.config.empty")));

        LambdaUpdateWrapper<UserBindDevice> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(UserBindDevice::getId, update.getId())
                .set(UserBindDevice::getStatus, update.getStatus())
                .set(UserBindDevice::getModifyUserId, authoredUser.getUserId())
                .set(UserBindDevice::getModifyUserName, authoredUser.getUserName())
                .set(UserBindDevice::getModifyTime, DateUtils.getNowTime());
        update(updateWrapper);
        UserBindDevice userBindDevice = baseMapper.selectById(update.getId());
        if (Objects.nonNull(userBindDevice)) {
            sendMqtt(userBindDevice.getTenantId(), userBindDevice.getUserId());
        }
        return Boolean.TRUE;
    }

    @Override
    public Boolean delete(List<Long> ids) {
        List<DeviceBindRecordVO> deviceBindRecordVOS = baseMapper.selectByIds(ids);
        if (CollUtil.isNotEmpty(deviceBindRecordVOS)) {
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            baseMapper.deleteBatchByIds(ids, authoredUser.getUserId());
            //推送消息
            deviceBindRecordVOS.forEach(user -> sendMqtt(user.getTenantId(), user.getUserId()));
        }
        return Boolean.TRUE;
    }

    private void sendMqtt(String tenantId, String userId) {
        MessageDO payload = new MessageDO();
        payload.setTenantId(tenantId);
        payload.setType(3);
        String topicSuffix = tenantId + "/" + userId + "/" + Constants.USER_DEVICE_STATUS_CHANGE;
        messageSendService.sendToClient(userId, payload, topicSuffix);
    }

    @Transactional
    public void importDeviceBind(DeviceBindImportTemplateVo deviceBindVo, AuthoredUser authoredUser) {
        DeviceBindSaveDTO deviceBindSave = new DeviceBindSaveDTO();
        deviceBindSave.setUserId(deviceBindVo.getUserId());
        deviceBindSave.setUserName(deviceBindVo.getUserName());
        deviceBindSave.setDeviceId(deviceBindVo.getDeviceId());
        deviceBindSave.setStatus(0);
        deviceBindSave.setTerminalType(Constants.TerminalTypeEnum.getTypeByCode(deviceBindVo.getTerminalType()));
        saveAction(deviceBindSave, authoredUser);
    }

    @Override
    public List<DeviceBindExportVO> exportReport(DeviceBindExportQueryDTO req) {
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        List<DeviceBindRecordVO> deviceBindRecords = this.baseMapper.selectExportList(req,tenantId);
        if (CollectionUtils.isNotEmpty(deviceBindRecords)) {
            return deviceBindRecords.stream().map(DeviceBindExportVO::new).collect(Collectors.toList());
        }
        return Collections.emptyList();
    }

    @Override
    public List<DeviceBindRecordVO> queryUserDevice() {
        if (!getBindEnabled()) {
            return Collections.emptyList();
        }
        return this.baseMapper.selectDeviceByUserOrDevice(Utils.getUserId(), null, null);
    }

    private boolean getBindEnabled() {
        DeviceBindConfigDTO config = iDeviceBindConfigService.getConfig();
        return BooleanUtils.isTrue(config.getBindEnabled());
    }

    @Override
    @Transactional
    public void autoBind(DeviceDTO device) {
        DeviceInfo deviceInfo = iDeviceInfoService.selectByTenantIdAndDeviceId(device.getDeviceId());
        if (Objects.isNull(deviceInfo)) {
            deviceInfo = iDeviceInfoService.save(device);
        } else {
            iDeviceInfoService.updateDeviceInfo(deviceInfo.getId(), device);
        }
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        if (Objects.isNull(authoredUser)) {
            authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        }
        String nowTime = DateUtils.getNowTime();
        UserBindDevice newUserBindDevice = new UserBindDevice();
        newUserBindDevice.setDeviceId(deviceInfo.getDeviceId());
        newUserBindDevice.setTenantId(authoredUser.getTenantId());
        newUserBindDevice.setUserId(authoredUser.getUserId());
        newUserBindDevice.setUserName(authoredUser.getUserName());
        newUserBindDevice.setStatus(0);
        newUserBindDevice.setCreateUserId(authoredUser.getUserId());
        newUserBindDevice.setCreateUserName(authoredUser.getUserName());
        newUserBindDevice.setCreateTime(nowTime);
        newUserBindDevice.setModifyUserId(authoredUser.getUserId());
        newUserBindDevice.setModifyUserName(authoredUser.getUserName());
        newUserBindDevice.setModifyTime(nowTime);
        this.baseMapper.insert(newUserBindDevice);
    }
}