package com.digiwin.athena.semc.service.eai.impl;

import com.digiwin.athena.semc.common.BizException;
import com.google.common.collect.Maps;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.digiwin.app.service.DWEAIResult;
import com.digiwin.app.service.eai.EAIService;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.DigiSrvcode;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.dto.message.ReceiveThirdMessageReq;
import com.digiwin.athena.semc.dto.news.ThirdNewsAnnouncementReq;
import com.digiwin.athena.semc.dto.portal.JobSyncReq;
import com.digiwin.athena.semc.dto.portal.TodoFieldDto;
import com.digiwin.athena.semc.dto.portal.TodoListDto;
import com.digiwin.athena.semc.dto.portal.TodoReceiveDto;
import com.digiwin.athena.semc.entity.bench.SyncJobInfo;
import com.digiwin.athena.semc.entity.bench.ThirdJobConfig;
import com.digiwin.athena.semc.entity.message.ThirdMessageConfig;
import com.digiwin.athena.semc.entity.message.ThirdMessageInfo;
import com.digiwin.athena.semc.entity.portal.SyncToDoRecord;
import com.digiwin.athena.semc.entity.portal.ThirdTodoConfig;
import com.digiwin.athena.semc.entity.portal.TodoList;
import com.digiwin.athena.semc.entity.portal.TodoRead;
import com.digiwin.athena.semc.mapper.bench.ThirdJobConfigMapper;
import com.digiwin.athena.semc.mapper.portal.SyncToDoRecordMapper;
import com.digiwin.athena.semc.mapper.portal.ThirdTodoConfigMapper;
import com.digiwin.athena.semc.service.bench.SyncJobInfoService;
import com.digiwin.athena.semc.service.cache.RedisLock;
import com.digiwin.athena.semc.service.eai.EaiService;
import com.digiwin.athena.semc.service.message.ThirdSystemMessageService;
import com.digiwin.athena.semc.service.news.ThirdNewsAnnouncementService;
import com.digiwin.athena.semc.service.portal.TodoListService;
import com.digiwin.athena.semc.service.portal.TodoReadService;
import com.digiwin.athena.semc.service.portal.impl.TodoNoticeMq;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.vo.portal.NoticeVO;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import io.vavr.Tuple3;
import lombok.extern.slf4j.Slf4j;

/**
 * EAI服务实现类
 *
 * @author: sungq
 * @date: 2024-03-14
 */
@Slf4j
@Service
public class EAIServiceImpl implements EaiService {
    @Resource
    private ThirdSystemMessageService thirdSystemMessageService;
    @Resource
    private SyncJobInfoService syncJobInfoService;
    @Resource
    private TodoListService todoListService;
    @Resource
    private TodoReadService todoReadService;


    @Resource
    private TodoNoticeMq todoNoticeMq;


    @Autowired
    private SyncToDoRecordMapper syncToDoRecordMapper;

    @Autowired
    private ThirdNewsAnnouncementService thirdNewsAnnouncementService;


    @Autowired
    private ThirdTodoConfigMapper thirdTodoConfigMapper;

    @Autowired
    private ThirdJobConfigMapper thirdJobConfigMapper;

    @Override
    @EAIService(id = "third.news.push")
    public DWEAIResult receiveThirdNews(JSONObject jsonObject) throws Exception {
        log.info("invoke esp semc service third.news.push, param:{}", jsonObject);
        Map<String, Object> parameterResult = Maps.newHashMap();
        try {
            JSONObject std_data = jsonObject.getJSONObject("std_data");
            JSONObject parameter = std_data.getJSONObject("parameter");
            String appCode = parameter.getString("appCode");
            String appId = parameter.getString("appId");
            Tuple3<String, String, ApplicationTypeEnum> tuple3 = thirdNewsAnnouncementService.checkAppCode(appId, appCode);
            String error = tuple3._1;
            String tenantId = tuple3._2;
            if (StringUtils.isNotBlank(error) || StringUtils.isBlank(tenantId)) {
                throw new Exception(error);
            }


            JSONArray jsonArray = parameter.getJSONArray("thirdNewsAnnouncementReqs");
            List<ThirdNewsAnnouncementReq> newsInfoList = JSONArray.parseArray(jsonArray.toJSONString(), ThirdNewsAnnouncementReq.class);
            String errorMsg = thirdNewsAnnouncementService.checkNewsIsValid(newsInfoList);
            if (org.springframework.util.StringUtils.hasText(errorMsg)) {
                throw new Exception(errorMsg);
            }
            thirdNewsAnnouncementService.saveNewsList(newsInfoList,appId,appCode,tenantId,tuple3._3);
        } catch (Exception e) {
            log.error("invoke esp semc service third.news.push error. param:{}", jsonObject, e);
            throw new Exception("invoke esp semc service third.news.push error, msg:" + e.getMessage());
        }
        parameterResult.put("response", true);
        return new DWEAIResult(DigiSrvcode.SUCCESS.getCode(), "0", "", parameterResult);
    }

    /**
     * 接收异构系统消息
     *
     * @param jsonObject 请求体
     * @return
     */
    @Override
    @EAIService(id = "third.message.push")
    public DWEAIResult receiveThirdMessage(JSONObject jsonObject) throws Exception {
        log.info("invoke esp semc service third.message.push, param:{}", jsonObject);
        Map<String, Object> parameterResult = Maps.newHashMap();
        try {
            JSONObject std_data = jsonObject.getJSONObject("std_data");
            JSONObject parameter = std_data.getJSONObject("parameter");
            String appCode = parameter.getString("app_code");
            String appId = parameter.getString("app_id");

            // 判断当前应用是否存在
            List<ThirdMessageConfig> thirdMessageConfigList = thirdSystemMessageService.queryConfigByAppCode(appCode,appId);
            if (CollectionUtils.isEmpty(thirdMessageConfigList)) {
                throw new Exception("当前应用不在配置中，无权推送消息");
            }
            List<ThirdMessageConfig> filteredList = thirdMessageConfigList.stream().filter(x -> Constants.VALID_STATUS_ENABLE.equals(x.getValidStatus())).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(filteredList)) {
                throw new Exception("当前应用未启用，无权推送消息");
            }

            JSONArray jsonArray = parameter.getJSONArray("message_list");
            List<ThirdMessageInfo> messageInfoList = JSONArray.parseArray(jsonArray.toJSONString(), ThirdMessageInfo.class);
            for (ThirdMessageInfo info : messageInfoList) {
                //校验消息ID
                if (StringUtils.isEmpty(info.getMessageId())) {
                    throw new Exception("消息id不能为空");
                }
                //校验时间和格式
                if (StringUtils.isEmpty(info.getMessagePublishTime())) {
                    throw new Exception("消息发布时间不能为空");
                }
                if (!DateUtils.validateDate(info.getMessagePublishTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) {
                    throw new Exception("消息发布时间格式不正确");
                }
                //校验内容
                if (StringUtils.isEmpty(info.getMessageTitle()) && StringUtils.isEmpty(info.getMessageContent())) {
                    throw new Exception("消息标题和消息内容不能同时为空");
                }
            }
            ReceiveThirdMessageReq receiveThirdMessageReq = new ReceiveThirdMessageReq();
            receiveThirdMessageReq.setAppCode(appCode);
            receiveThirdMessageReq.setAppId(appId);
            receiveThirdMessageReq.setMessageList(messageInfoList);
            String sendChannel = filteredList.get(0).getSendChannel();
            receiveThirdMessageReq.setSendChannel(sendChannel);
            thirdSystemMessageService.receiveMessage(receiveThirdMessageReq);
        } catch (Exception e) {
            log.error("invoke esp semc service third.message.push error. param:{}", jsonObject, e);
            throw new Exception("invoke esp semc service third.message.push error, msg:" + e.getMessage());
        }
        parameterResult.put("response", true);
        return new DWEAIResult(DigiSrvcode.SUCCESS.getCode(), "0", "", parameterResult);
    }

    @Override
    @EAIService(id = "third.job.push")
    public DWEAIResult receiveThirdJob(JSONObject jsonObject) throws Exception {
        log.info("third.job.push  param:{}", JSON.toJSONString(jsonObject));
        Map<String, Object> parameterResult = Maps.newHashMap();
        JobSyncReq req = new JobSyncReq();
        JSONObject std_data = jsonObject.getJSONObject("std_data");
        JSONObject parameter = std_data.getJSONObject("parameter");
        String appId = parameter.getString("appId");
        Integer syncType = parameter.getInteger("syncType");
        Integer syncModel = parameter.getInteger("syncModel");
        JSONArray jsonArrayJob = parameter.getJSONArray("jobList");
        req.setAppId(appId);
        req.setSyncType(syncType);
        if(null == syncModel){
            syncModel=0;
        }
        //查询作业集成配置，校验是否配置且启用状态
        ThirdJobConfig thirdJobConfig= thirdJobConfigMapper.getJobConfigBy("",appId);
        if (null == thirdJobConfig) {
            throw new Exception("当前应用不在配置中，无权推送作业");
        }
        if (Constants.VALID_STATUS_UNUSABLE.equals(thirdJobConfig.getValidStatus())) {
            throw new Exception("当前应用未启用，无权推送作业");
        }
        req.setSyncModel(syncModel);
        try {
            List<SyncJobInfo> jobList = JSONArray.parseArray(jsonArrayJob.toJSONString(), SyncJobInfo.class);
            //判断作业是否为空
            if (CollectionUtils.isEmpty(jobList)) {
                return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "作业信息为空", parameterResult);
            }
            req.setJobList(jobList);
        } catch (Exception e) {
            log.error("third.job.push  parseObject error", e);
            return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "入参解析异常，请检查入参是否正确", parameterResult);
        }
        try {

            //判断appid是否为空
            if (StringUtils.isEmpty(req.getAppId())) {
                return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "AppId为空", parameterResult);
            }

            // 推送作业的维度是账套维度
            boolean accountFlag = req.getJobList().stream().anyMatch(x -> StringUtils.isNotBlank(x.getAccountId()));
            if (accountFlag) {
                boolean flag = req.getJobList().stream().anyMatch(x -> StringUtils.isBlank(x.getAccountId()));
                if (flag) {
                    return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "missing accountId", parameterResult);
                }
                List<String> jobCodeList = Lists.newArrayList();
                for (SyncJobInfo syncJobInfo : req.getJobList()) {
                    String jobCode = syncJobInfo.getAccountId() + "-" + syncJobInfo.getJobCode();
                    if (jobCodeList.contains(jobCode)) {
                        return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "exist repeated accountId and jobCode", parameterResult);
                    }
                    syncJobInfo.setJobCode(jobCode);
                    jobCodeList.add(jobCode);
                }
            }
            syncJobInfoService.jobSyncSave(req);
        } catch (Exception e) {
            log.error("third.job.push  jobSyncSave error", e);
            throw new Exception("third.job.push error, msg:" + e.getMessage());
        }
        parameterResult.put("response", true);
        return new DWEAIResult(DigiSrvcode.SUCCESS.getCode(), "0", "", parameterResult);
    }

    @Override
    @EAIService(id = "third.todo.push")
    public DWEAIResult receiveThirdToDo(JSONObject jsonObject) throws Exception {
        log.info("third.todo.push  param:{}", JSON.toJSONString(jsonObject));
        Map<String, Object> parameterResult = Maps.newHashMap();
        TodoReceiveDto receiveDto = new TodoReceiveDto();
        JSONObject std_data = jsonObject.getJSONObject("std_data");
        JSONObject parameter = std_data.getJSONObject("parameter");
        String appCode = parameter.getString("app_code");
        String appIdCode = parameter.getString("app_id_code");
        receiveDto.setAppCode(appCode);
        receiveDto.setAppIdCode(appIdCode);
        JSONArray jsonArrayToDoList = parameter.getJSONArray("to_do_list");
        JSONArray jsonArrayToDoField = parameter.getJSONArray("display_field_list");
        //查询待办集成配置，校验是否配置且启用状态
        ThirdTodoConfig thirdTodoConfig= thirdTodoConfigMapper.getToDoConfigBy("",appCode);
        if (null == thirdTodoConfig) {
            throw new Exception("当前应用不在配置中，无权推送待办");
        }
        if (Constants.VALID_STATUS_UNUSABLE.equals(thirdTodoConfig.getValidStatus())) {
            throw new Exception("当前应用未启用，无权推送待办");
        }
        try {
            //判断AppCode否为空
            if (StringUtils.isEmpty(receiveDto.getAppCode())) {
                return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "AppCode为空", parameterResult);
            }
            List<TodoListDto> toDoList = JSONArray.parseArray(jsonArrayToDoList.toJSONString(), TodoListDto.class);
            //判断待办数据是否为空
            if (CollectionUtils.isEmpty(toDoList)) {
                return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "待办列表为空", parameterResult);
            }
            List<TodoFieldDto> toDoFieldList = JSONArray.parseArray(jsonArrayToDoField.toJSONString(), TodoFieldDto.class);
            //判断待办标题是否为空
            if (CollectionUtils.isEmpty(toDoFieldList)) {
                return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "待办显示字段列表为空", parameterResult);
            }
            receiveDto.setDisplayFieldList(toDoFieldList);
            receiveDto.setToDoList(toDoList);
            todoListService.toDoReceive(receiveDto);
        } catch (Exception e) {
            log.error("third.todo.push   error", e);
            throw new Exception("third.todo.push error, msg:" + e.getMessage());
        }
        parameterResult.put("response", true);
        return new DWEAIResult(DigiSrvcode.SUCCESS.getCode(), "0", "", parameterResult);
    }


    @Override
    @EAIService(id = "third.todo.notice")
    public DWEAIResult noticeThirdToDo(JSONObject jsonObject) {
        log.info("third.todo.notice  param:{}", JSON.toJSONString(jsonObject));
        Map<String, Object> map = Maps.newHashMap();
        NoticeVO noticeVO = new NoticeVO();
        JSONObject stdData = jsonObject.getJSONObject("std_data");
        JSONObject parameter = stdData.getJSONObject("parameter");
        noticeVO.setBizId(parameter.getString("biz_id"));
        noticeVO.setAppName(parameter.getString("app_name"));
        noticeVO.setAppId(parameter.getString("app_id"));
        SyncToDoRecord  toDoRecord=new SyncToDoRecord();
        toDoRecord.setAppId(parameter.getString("app_id"));
        toDoRecord.setDataModel(0);
        toDoRecord.setReqJson(JSON.toJSONString(jsonObject));
        toDoRecord.setSyncType(0);
        toDoRecord.setSyncStatus(0);
        toDoRecord.setToDoId(parameter.getString("biz_id"));
        // 参数为空校验
        if (StringUtils.isEmpty(noticeVO.getBizId())) {
            return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "bizId为空", map);
        }

        if (StringUtils.isEmpty(noticeVO.getAppId())) {
            return new DWEAIResult(DigiSrvcode.FAILURE.getCode(), "0", "appId为空", map);
        }
        RedisLock redisLock = new RedisLock("thirdTodoNotice:"+noticeVO.getBizId());
        //上锁
        try {
            if(!redisLock.lock()){
                return new DWEAIResult(DigiSrvcode.SUCCESS.getCode(), "0", "", map);
            }
        } catch (InterruptedException e) {
            return new DWEAIResult(DigiSrvcode.SUCCESS.getCode(), "0", "", map);
        }
        try {
            // 开启分布式锁
            List<TodoList> todoListDb = todoNoticeMq.selectToDoBy(noticeVO.getAppId(),noticeVO.getBizId());
            LambdaUpdateWrapper<TodoList> updateTodoListWrapper = new LambdaUpdateWrapper<>();
            updateTodoListWrapper.eq(TodoList::getAppId, noticeVO.getAppId());
            updateTodoListWrapper.eq(TodoList::getToDoId, noticeVO.getBizId());
            todoListService.remove(updateTodoListWrapper);

            LambdaUpdateWrapper<TodoRead> updateTodoReadWrapper = new LambdaUpdateWrapper<>();
            updateTodoReadWrapper.eq(TodoRead::getAppId, noticeVO.getAppId());
            updateTodoReadWrapper.eq(TodoRead::getToDoId, noticeVO.getBizId());
            todoReadService.remove(updateTodoReadWrapper);

            // 发送mqtt消息给前端
            todoNoticeMq.sendMessageToMqtt(noticeVO,todoListDb);
        } catch (Exception e) {
            toDoRecord.setSyncStatus(1);
            toDoRecord.setRespJson(e.getMessage());
            log.error("third.todo.notice   error", e);
        } finally {
            redisLock.unlock();
        }
         syncToDoRecordMapper.addRecord(toDoRecord);
        return new DWEAIResult(DigiSrvcode.SUCCESS.getCode(), "0", "", map);
    }

}
