package com.digiwin.athena.semc.service.event.impl;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.EventTrackingConstants;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.EventTypeEnum;
import com.digiwin.athena.semc.converter.event.EventTrackingConverter;
import com.digiwin.athena.semc.dto.event.EventTrackingReq;
import com.digiwin.athena.semc.dto.event.PageQueryEventTrackingDTO;
import com.digiwin.athena.semc.entity.event.EventTracking;
import com.digiwin.athena.semc.event.EventTrackingEvent;
import com.digiwin.athena.semc.mapper.event.EventTrackingMapper;
import com.digiwin.athena.semc.service.event.EventTrackingService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.vo.event.PageQueryEventTrackingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class EventTrackingServiceImpl extends ServiceImpl<EventTrackingMapper, EventTracking> implements EventTrackingService {
    private final EventTrackingMapper eventTrackingMapper;

    private final ApplicationEventPublisher eventPublisher;

    private final EventTrackingConverter eventTrackingConverter;

    @Override
    public Integer saveTracking(EventTrackingReq req) {
        AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
        Map<String, Object> common = req.getCommon();
        if (CollectionUtils.isEmpty(req.getEvents())) {
            return 0;
        }
        List<EventTracking> eventTrackingList = req.getEvents().stream()
                .map(eventDto -> {
                    EventTracking event = new EventTracking();
                    event.setTenantId(user.getTenantId());
                    event.setUserId(user.getUserId());
                    event.setTenantName(user.getTenantName());
                    event.setUserName(user.getUserName());
                    event.setEventTime(req.getTs());
                    event.setSource(Integer.parseInt(String.valueOf(common.get(Constants.SOURCE))));
                    event.setEventId(eventDto.getEventCode());
                    event.setCreateTime(DateUtils.getNowTime(""));
                    event.setContent(JSON.toJSONString(req));
                    return event;
                })
                .collect(Collectors.toList());
        // 批量插入
        int result = eventTrackingMapper.batchInsert(eventTrackingList);
        boolean isSendEvent = req.getEvents().stream().anyMatch(m ->
                EventTrackingConstants.LOGIN_OUT_CONFIRM_CLICK.equals(m.getEventCode()) ||
                        EventTrackingConstants.TENANT_SWITCH_BUTTON_CLICK.equals(m.getEventCode()));
        if (isSendEvent) {
            EventTrackingEvent eventTrackingEvent = new EventTrackingEvent(this, user.getTenantId(), user.getUserId());
            eventPublisher.publishEvent(eventTrackingEvent);
        }
        return result;
    }

    /**
     * 分页查询用户日志
     *
     * @param pageQueryEventTrackingDTO 查询参数
     * @return 响应结果
     */
    @Override
    public PageInfoResp<PageQueryEventTrackingVO> pageQuery(PageQueryEventTrackingDTO pageQueryEventTrackingDTO) {
        List<String> eventIds;
        if (CollectionUtils.isEmpty(pageQueryEventTrackingDTO.getEventTypes())) {
            eventIds = EventTypeEnum.getWorkbenchEventName();
        } else {
            eventIds = EventTypeEnum.getEventIdByEventType(pageQueryEventTrackingDTO.getEventTypes());
        }

        Page<EventTracking> result = lambdaQuery()
                .like(StrUtil.isNotBlank(pageQueryEventTrackingDTO.getEventName()), EventTracking::getContent, pageQueryEventTrackingDTO.getEventName())
                .between(StrUtil.isNotBlank(pageQueryEventTrackingDTO.getCreateDateStart()) && StrUtil.isNotBlank(pageQueryEventTrackingDTO.getCreateDateEnd()),
                        EventTracking::getCreateTime, pageQueryEventTrackingDTO.getCreateDateStart(), pageQueryEventTrackingDTO.getCreateDateEnd())
                .in(CollectionUtils.isNotEmpty(eventIds), EventTracking::getEventId, eventIds)
                .eq(EventTracking::getTenantId, AppAuthContextHolder.getContext().getAuthoredUser().getTenantId())
                .and(StrUtil.isNotBlank(pageQueryEventTrackingDTO.getUserContent()), wrapper -> wrapper
                        .like(EventTracking::getUserId, pageQueryEventTrackingDTO.getUserContent())
                        .or()
                        .like(EventTracking::getUserName, pageQueryEventTrackingDTO.getUserContent()))
                .orderByDesc(EventTracking::getCreateTime)
                .page(new Page<>(pageQueryEventTrackingDTO.getPageNo(), pageQueryEventTrackingDTO.getPageSize()));

        // 拼接操作详情
        List<PageQueryEventTrackingVO> pageQueryEventTrackingVOList = eventTrackingConverter.toPageQueryEventTrackingVOList(result.getRecords());
        return new PageInfoResp<>(pageQueryEventTrackingDTO.getPageNo(),
                pageQueryEventTrackingDTO.getPageSize(),
                Integer.parseInt(String.valueOf(result.getTotal())),
                pageQueryEventTrackingVOList);
    }

    /**
     * 导出所有按条件查出的结果
     *
     * @return 返回结果
     */
    @Override
    public List<PageQueryEventTrackingVO> exportReport(PageQueryEventTrackingDTO pageQueryEventTrackingDTO) {
        List<String> eventIds;
        if (CollectionUtils.isEmpty(pageQueryEventTrackingDTO.getEventTypes())) {
            eventIds = EventTypeEnum.getWorkbenchEventName();
        } else {
            eventIds = EventTypeEnum.getEventIdByEventType(pageQueryEventTrackingDTO.getEventTypes());
        }

        List<EventTracking> eventTrackingList = lambdaQuery()
                .like(StrUtil.isNotBlank(pageQueryEventTrackingDTO.getEventName()), EventTracking::getContent, pageQueryEventTrackingDTO.getEventName())
                .between(StrUtil.isNotBlank(pageQueryEventTrackingDTO.getCreateDateStart()) && StrUtil.isNotBlank(pageQueryEventTrackingDTO.getCreateDateEnd()),
                        EventTracking::getCreateTime, pageQueryEventTrackingDTO.getCreateDateStart(), pageQueryEventTrackingDTO.getCreateDateEnd())
                .in(CollectionUtils.isNotEmpty(eventIds), EventTracking::getEventId, eventIds)
                .eq(EventTracking::getTenantId, AppAuthContextHolder.getContext().getAuthoredUser().getTenantId())
                .and(StrUtil.isNotBlank(pageQueryEventTrackingDTO.getUserContent()), wrapper -> wrapper
                        .like(EventTracking::getUserId, pageQueryEventTrackingDTO.getUserContent())
                        .or()
                        .like(EventTracking::getUserName, pageQueryEventTrackingDTO.getUserContent()))
                .orderByDesc(EventTracking::getCreateTime)
                .list();
        return eventTrackingConverter.toPageQueryEventTrackingVOList(eventTrackingList);
    }

    /**
     * 拼接操作详情
     *
     * @param eventTracking 埋点事件
     * @return 响应结果
     */
    public static String splitEventName(EventTracking eventTracking) {
        // 登入登出不需要解析attrs
        if (ObjectUtil.equals(eventTracking.getEventId(), EventTypeEnum.LOGIN_OUT_CONFIRM_CLICK.getEventId())) {
            return String.format("用户[%s(%s)]，登出租户[%s]成功", eventTracking.getUserName(), eventTracking.getUserId(), eventTracking.getTenantName());
        } else if (ObjectUtil.equals(eventTracking.getEventId(), EventTypeEnum.LOGIN_CLICK.getEventId())) {
            return String.format("用户[%s(%s)]，登入租户[%s]成功", eventTracking.getUserName(), eventTracking.getUserId(), eventTracking.getTenantName());
        }

        // 剩下都需要解析attr中的数据
        EventTrackingReq eventTrackingReq = JSON.parseObject(eventTracking.getContent(), EventTrackingReq.class);
        Map<String, Object> attr = eventTrackingReq.getEvents().get(0).getAttrs().get(0);

        // 管理后台赋权
        if (ObjectUtil.equals(eventTracking.getEventId(), EventTypeEnum.WORK_AUTHORIZATION.getEventId())) {
            // 拼接所有作业名称
            StringBuilder labelName = new StringBuilder();
            JSONArray labelList = JSON.parseArray(JSON.toJSONString(attr.get("labelList")));
            for (Object e : labelList) {
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(e));
                String newLabelName = null == jsonObject.get("labelName") ? "" : jsonObject.get("labelName").toString();
                String labelId = null == jsonObject.get("labelId") ? "" : jsonObject.get("labelId").toString();
                if (StrUtil.isBlank(newLabelName) && StrUtil.isBlank(labelId)) {
                    continue;
                }
                labelName.append(newLabelName).append("(").append(labelId).append("),");
            }
            if (labelName.length() > 0) {
                labelName.deleteCharAt(labelName.length() - 1);
            }
            String authName = null == attr.get("authName") ? "" : attr.get("authName").toString();
            String authId = null == attr.get("authId") ? "" : attr.get("authId").toString();
            // isRelated如果为true表示授权，为false表示解除权限
            if (ObjectUtil.equals(attr.get("isRelated"), Boolean.TRUE)) {
                return String.format("用户[%s(%s)]，将管理后台的[%s]作业赋权给[%s(%s)]",
                        eventTracking.getUserName(), eventTracking.getUserId(), labelName, authName, authId);
            }
            return String.format("用户[%s(%s)]，将管理后台的[%s]作业从[%s(%s)]解除权限",
                    eventTracking.getUserName(), eventTracking.getUserId(), labelName, authName, authId);
        }

        String trackName = null == attr.get(EventTrackingConstants.TRACK_NAME) ? "" : attr.get(EventTrackingConstants.TRACK_NAME).toString();
        String trackId = null == attr.get(EventTrackingConstants.TRACK_ID) ? "" : attr.get(EventTrackingConstants.TRACK_ID).toString();
        if (ObjectUtil.equals(eventTracking.getEventId(), EventTypeEnum.THIRD_TODO_CLICK.getEventId())) {
            return String.format("用户[%s(%s)]，查看待办[%s(%s)]",
                    eventTracking.getUserName(), eventTracking.getUserId(), trackName, trackId);
        } else if (ObjectUtil.equals(eventTracking.getEventId(), EventTypeEnum.THIRD_MESSAGE_CLICK.getEventId())) {
            return String.format("用户[%s(%s)]，查看消息[%s(%s)]",
                    eventTracking.getUserName(), eventTracking.getUserId(), trackName, trackId);
        } else if (ObjectUtil.equals(eventTracking.getEventId(), EventTypeEnum.CUSTOM_PORTAL_CLICK.getEventId())) {
            return String.format("用户[%s(%s)]，使用工作台[%s(%s)]",
                    eventTracking.getUserName(), eventTracking.getUserId(), trackName, trackId);
        } else if (ObjectUtil.equals(eventTracking.getEventId(), EventTypeEnum.COMPONENT_CLICK.getEventId())) {
            return String.format("用户[%s(%s)]，点击部件[%s(%s)]",
                    eventTracking.getUserName(), eventTracking.getUserId(), trackName, trackId);
        }
        return "";
    }
}
