package com.digiwin.athena.semc.service.fineReport.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.fineReport.FineReportConfigDTO;
import com.digiwin.athena.semc.dto.fineReport.QueryFineReportListReq;
import com.digiwin.athena.semc.dto.fineReport.SaveFineReportAuthReq;
import com.digiwin.athena.semc.entity.fineReport.FineReportAuth;
import com.digiwin.athena.semc.mapper.fineReport.FineReportAuthMapper;
import com.digiwin.athena.semc.proxy.fineReport.model.resp.FineReportTreeVO;
import com.digiwin.athena.semc.proxy.fineReport.service.FineReportProxyService;
import com.digiwin.athena.semc.proxywrapper.IamUserService;
import com.digiwin.athena.semc.proxywrapper.dto.IamUserAuthInfo;
import com.digiwin.athena.semc.service.fineReport.FineReportAuthService;
import com.digiwin.athena.semc.service.portal.TripartiteApplicationService;
import com.digiwin.athena.semc.util.Utils;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 三方应用-帆软报表权限表服务实现类
 *
 * @author sunqgz
 * @since 2025-08-12
 */
@Slf4j
@Service
public class FineReportAuthServiceImpl extends ServiceImpl<FineReportAuthMapper, FineReportAuth> implements FineReportAuthService {

    @Resource
    private FineReportAuthMapper fineReportAuthMapper;

    @Resource
    private FineReportProxyService fineReportProxyService;

    @Resource
    private IamUserService iamUserService;

    @Resource
    private TripartiteApplicationService tripartiteApplicationService;

    /**
     * 查询帆软报表列表
     *
     * @param queryFineReportListReq 请求对象
     * @return 返回
     */
    public JSONObject queryFineReportList(QueryFineReportListReq queryFineReportListReq) {
        JSONObject resp = new JSONObject();
        // 先查询当前租户的帆软应用配置，用于调帆软接口查询报表
        FineReportConfigDTO fineReportConfigDTO = tripartiteApplicationService.queryFineReportConfig();
        if (ObjectUtils.isEmpty(fineReportConfigDTO)) {
            resp.put("allAuthFlag", Constants.ALL_AUTH_FLAG_NO);
            resp.put("fineReportList", Lists.newArrayList());
            return resp;
        }

        // 若不传设备类型，则查询所有设备类型的报表
        List<FineReportTreeVO> totalFineReportList = Lists.newArrayList();
        if (ObjectUtils.isEmpty(queryFineReportListReq.getDeviceType())) {
            // 帆软目录接口，入参设备类型枚举
            List<String> deviceTypeList = Arrays.asList(Constants.FineReportDeviceEnum.PC_DEVICVE.getFlag(),
                    Constants.FineReportDeviceEnum.PAD_DEVICVE.getFlag(),
                    Constants.FineReportDeviceEnum.PFONE_DEVICE.getFlag());
            deviceTypeList.forEach(x -> queryFineReportListByDeviceType(fineReportConfigDTO, x, totalFineReportList));
        }
        // pc端
        else if (Constants.PC_FINE_REPORT.equals(queryFineReportListReq.getDeviceType())) {
            queryFineReportListByDeviceType(fineReportConfigDTO, Constants.FineReportDeviceEnum.PC_DEVICVE.getFlag(), totalFineReportList);
        }
        // 移动端
        else if (Constants.MOBILE_FINE_REPORT.equals(queryFineReportListReq.getDeviceType())) {
            List<String> deviceTypeList = Arrays.asList(Constants.FineReportDeviceEnum.PAD_DEVICVE.getFlag(),
                    Constants.FineReportDeviceEnum.PFONE_DEVICE.getFlag());
            deviceTypeList.forEach(x -> queryFineReportListByDeviceType(fineReportConfigDTO, x, totalFineReportList));
        }
        Map<String, FineReportTreeVO> fineReprotIdMap = totalFineReportList.stream().collect(Collectors.toMap(FineReportTreeVO::getId, Function.identity(), (a, b) -> a));
        List<FineReportTreeVO> fineReportTreeList = new ArrayList<>(fineReprotIdMap.values());
        if (CollectionUtils.isEmpty(fineReportTreeList)) {
            resp.put("allAuthFlag", Constants.ALL_AUTH_FLAG_NO);
            resp.put("fineReportList", fineReportTreeList);
            return resp;
        }

        // 查询当前权限对象有权限的帆软报表，标记是否有权限
        if (StringUtils.isNotBlank(queryFineReportListReq.getAuthBizId())) {
            LambdaQueryWrapper<FineReportAuth> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(FineReportAuth::getAuthBizId, queryFineReportListReq.getAuthBizId());
            List<FineReportAuth> fineReportAuthList = fineReportAuthMapper.selectList(queryWrapper);
            return buildFineReportList(fineReportTreeList, fineReportAuthList);
        }
        // 查询当前用户有权限的帆软报表，标记是否有权限
        else if (ObjectUtils.isNotEmpty(queryFineReportListReq.getDisplayPermissionFlag()) && Boolean.TRUE.equals(queryFineReportListReq.getDisplayPermissionFlag())) {
            List<FineReportAuth> fineReportAuthList = queryFineReportAuthList();
            return buildFineReportList(fineReportTreeList, fineReportAuthList);
        }

        // 不参与校验权限，则所有报表都可查看
        resp.put("allAuthFlag", Constants.ALL_AUTH_FLAG_YES);
        resp.put("fineReportList", fineReportTreeList);
        return resp;
    }

    /**
     * 查询帆软报表列表
     *
     * @param fineReportConfigDTO 帆软应用配置
     * @param deviceType          设备类型
     * @param totalFineReportList 返回列表
     */
    private void queryFineReportListByDeviceType(FineReportConfigDTO fineReportConfigDTO, String deviceType, List<FineReportTreeVO> totalFineReportList) {
        List<FineReportTreeVO> fineReportTreeList = fineReportProxyService.queryFineReportEntryTree(fineReportConfigDTO.getClientId(), fineReportConfigDTO.getSecret(), deviceType);
        fineReportTreeList = fineReportTreeList.stream().filter(y -> Constants.FINE_REPORT_TYPE.equals(y.getEntryType())).sorted(Comparator.comparing(FineReportTreeVO::getSortIndex)).collect(Collectors.toList());
        totalFineReportList.addAll(fineReportTreeList);
    }

    /**
     * 查询当前用户有权限的报表
     *
     * @return 返回
     */
    @Override
    public List<FineReportAuth> queryFineReportAuthList() {
        // 查询当前用户所属组织、角色
        IamUserAuthInfo iamUserAuthInfo = iamUserService.getUserAuthSids(Utils.getUserId(), Utils.getTenantId(), Utils.getUserToken());

        List<Long> authIdList = Lists.newArrayList();
        // 拼接组织条件
        if (ObjectUtils.isNotEmpty(iamUserAuthInfo) && CollectionUtils.isNotEmpty(iamUserAuthInfo.getOrgSids())) {
            authIdList.addAll(iamUserAuthInfo.getOrgSids());
        }
        // 拼接角色条件
        if (ObjectUtils.isNotEmpty(iamUserAuthInfo) && CollectionUtils.isNotEmpty(iamUserAuthInfo.getRoleSids())) {
            authIdList.addAll(iamUserAuthInfo.getRoleSids());
        }
        // 拼接部门条件
        if (ObjectUtils.isNotEmpty(iamUserAuthInfo) && CollectionUtils.isNotEmpty(iamUserAuthInfo.getDeptSids())) {
            authIdList.addAll(iamUserAuthInfo.getDeptSids());
        }
        // 拼接用户条件
        authIdList.add(Utils.getUserSid());

        List<FineReportAuth> fineReportAuthList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(authIdList)) {
            Collections.sort(authIdList);
            // 拼接组织条件
            QueryWrapper<FineReportAuth> condition = new QueryWrapper<>();
            condition.apply("(" + "(auth_id in(" + Joiner.on(",").join(authIdList) + "))" + ")");
            condition.select("all_auth_flag", "fine_report_id");
            fineReportAuthList = fineReportAuthMapper.selectList(condition);
        }
        return fineReportAuthList;
    }

    /**
     * 构建帆软报表返回列表
     *
     * @param fineReportTreeList 报表列表
     * @param fineReportAuthList 有权限的报表列表
     * @return 返回
     */
    private static JSONObject buildFineReportList(List<FineReportTreeVO> fineReportTreeList, List<FineReportAuth> fineReportAuthList) {
        JSONObject resp = new JSONObject();
        // 无任何报表的权限
        if (CollectionUtils.isEmpty(fineReportAuthList)) {
            fineReportTreeList.forEach(x -> x.setHavePermissionFlag(Boolean.FALSE));
            resp.put("allAuthFlag", Constants.ALL_AUTH_FLAG_NO);
            resp.put("fineReportList", fineReportTreeList);
            return resp;
        }

        // 权限对象勾选了全选报表，可以查看所有报表
        boolean flag = fineReportAuthList.stream().anyMatch(x -> Constants.ALL_AUTH_FLAG_YES.equals(x.getAllAuthFlag()));
        if (flag) {
            fineReportTreeList.forEach(x -> x.setHavePermissionFlag(Boolean.TRUE));
            resp.put("allAuthFlag", Constants.ALL_AUTH_FLAG_YES);
            resp.put("fineReportList", fineReportTreeList);
        } else { // 否则，标记当前权限对象有权限的报表
            List<String> reprotIdList = fineReportAuthList.stream().map(FineReportAuth::getFineReportId).collect(Collectors.toList());
            fineReportTreeList.forEach(x -> x.setHavePermissionFlag(reprotIdList.contains(x.getId()) ? Boolean.TRUE : Boolean.FALSE));
            resp.put("allAuthFlag", Constants.ALL_AUTH_FLAG_NO);
            resp.put("fineReportList", fineReportTreeList);
        }
        return resp;
    }

    /**
     * 保存权限对象关联的帆软报表
     *
     * @param saveFineReportAuthReq 请求对象
     */
    @Override
    @Transactional
    public void saveFineReportAuth(SaveFineReportAuthReq saveFineReportAuthReq) {
        // 先删除当前权限关系
        QueryWrapper<FineReportAuth> condition = new QueryWrapper<>();
        condition.eq("auth_biz_id", saveFineReportAuthReq.getAuthBizId());
        fineReportAuthMapper.delete(condition);

        // 新增权限关系
        if (Constants.ALL_AUTH_FLAG_YES.equals(saveFineReportAuthReq.getAllAuthFlag())) {
            FineReportAuth fineReportAuth = new FineReportAuth();
            fineReportAuth.setAuthId(saveFineReportAuthReq.getAuthId());
            fineReportAuth.setAuthBizId(saveFineReportAuthReq.getAuthBizId());
            fineReportAuth.setAuthName(saveFineReportAuthReq.getAuthName());
            fineReportAuth.setAuthType(saveFineReportAuthReq.getAuthType());
            fineReportAuth.setAllAuthFlag(Constants.ALL_AUTH_FLAG_YES);
            this.save(fineReportAuth);
        } else {
            if (CollectionUtils.isEmpty(saveFineReportAuthReq.getReportIdList())) {
                return;
            }
            List<FineReportAuth> fineReportAuthList = saveFineReportAuthReq.getReportIdList().stream().map(x -> {
                FineReportAuth fineReportAuth = new FineReportAuth();
                fineReportAuth.setFineReportId(x);
                fineReportAuth.setAuthId(saveFineReportAuthReq.getAuthId());
                fineReportAuth.setAuthBizId(saveFineReportAuthReq.getAuthBizId());
                fineReportAuth.setAuthName(saveFineReportAuthReq.getAuthName());
                fineReportAuth.setAuthType(saveFineReportAuthReq.getAuthType());
                return fineReportAuth;
            }).collect(Collectors.toList());
            // 500个一组分批存储，防止一次数据过大
            List<List<FineReportAuth>> partList = Lists.partition(fineReportAuthList, 500);
            for (List<FineReportAuth> tempList : partList) {
                this.saveBatch(tempList);
            }
        }
    }
}