package com.digiwin.athena.semc.service.fineReport.impl;


import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.ResponseBody;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.fineReport.FineReportConfigDTO;
import com.digiwin.athena.semc.dto.portal.FineReportQueryReq;
import com.digiwin.athena.semc.entity.fineReport.FineReportUserMapping;
import com.digiwin.athena.semc.entity.portal.TripartiteApplication;
import com.digiwin.athena.semc.mapper.fineReport.FineReportUserMappingMapper;
import com.digiwin.athena.semc.proxy.fineReport.model.resp.FineReportUserVO;
import com.digiwin.athena.semc.proxy.fineReport.service.FineReportProxyService;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.RoleDTO;
import com.digiwin.athena.semc.service.cache.CommonConfigService;
import com.digiwin.athena.semc.service.fineReport.FineReportManageService;
import com.digiwin.athena.semc.service.portal.TripartiteApplicationService;
import com.digiwin.athena.semc.util.Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 三方应用-帆软报表服务实现类
 *
 * @author sunqgz
 * @since 2025-08-11
 */
@Slf4j
@Service
public class FineReportManageServiceImpl extends ServiceImpl<FineReportUserMappingMapper, FineReportUserMapping> implements FineReportManageService {

    @Resource
    private FineReportUserMappingMapper fineReportUserMappingMapper;

    @Resource
    private TripartiteApplicationService tripartiteApplicationService;

    @Resource
    private CommonConfigService commonConfigService;

    @Resource
    private FineReportProxyService fineReportProxyService;

    @Resource
    private IamService iamService;

    /**
     * 保存租户的帆软参数配置
     *
     * @param fineReportConfigDTO 请求参数
     */
    public void saveFineReportConfig(FineReportConfigDTO fineReportConfigDTO) {
        LambdaUpdateWrapper<TripartiteApplication> updateWrapper = new LambdaUpdateWrapper<>();
        String value = JSON.toJSONString(fineReportConfigDTO);
        updateWrapper.set(TripartiteApplication::getApplicationConfig, value);
        updateWrapper.set(TripartiteApplication::getModifyUserId, Utils.getUserId());
        updateWrapper.set(TripartiteApplication::getModifyTime, LocalDateTime.now());
        updateWrapper.eq(TripartiteApplication::getTenantId, Utils.getTenantId());
        tripartiteApplicationService.update(updateWrapper);
        // 更新缓存
        commonConfigService.setFineReportConfigCache(fineReportConfigDTO, Utils.getTenantId());
    }

    /**
     * 保存用户映射
     *
     * @param fineReportUserMapping 用户映射
     * @return 用户映射
     */
    @Override
    @Transactional
    public FineReportUserMapping saveUserMapping(FineReportUserMapping fineReportUserMapping) {
        Long id = fineReportUserMapping.getId();
        validateUserMapping(fineReportUserMapping, id);
        FineReportUserVO fineReportUserVO = new FineReportUserVO();
        fineReportUserVO.setRealName(fineReportUserMapping.getFrUserName());
        fineReportUserVO.setPassword(fineReportUserMapping.getFrUserPwd());
        fineReportUserVO.setEmail(fineReportUserMapping.getFrUserEmail());
        fineReportUserVO.setMobile(fineReportUserMapping.getFrUserMobile());

        FineReportConfigDTO fineReportConfigDTO = tripartiteApplicationService.queryFineReportConfig();
        if (ObjectUtils.isEmpty(fineReportConfigDTO)) {
            throw new RuntimeException("无帆软报表配置");
        }

        if (Objects.isNull(id)) {
            // 新增
            fineReportUserMappingMapper.insert(fineReportUserMapping);
        } else {
            // 更新
            fineReportUserMappingMapper.updateById(fineReportUserMapping);
        }
        List<FineReportUserVO> fineReportUsers = fineReportProxyService.queryFineReportUser(fineReportConfigDTO.getClientId(), fineReportConfigDTO.getSecret(), fineReportUserMapping.getFrUserId());
        boolean match = fineReportUsers.stream().anyMatch(u -> fineReportUserMapping.getFrUserId().equals(u.getUsername()));
        if (!match) {
            // 不存在，则新增
            // 调用帆软接口
            fineReportUserVO.setUsername(fineReportUserMapping.getFrUserId());
            fineReportProxyService.addFineReportUser(fineReportConfigDTO.getClientId(), fineReportConfigDTO.getSecret(), fineReportUserVO);
        } else {
            // 存在，则更新
            // 调用帆软接口
            fineReportUserVO.setUser(fineReportUserMapping.getFrUserId());
            fineReportProxyService.updateFineReportUser(fineReportConfigDTO.getClientId(), fineReportConfigDTO.getSecret(), fineReportUserVO);
        }
        return fineReportUserMapping;
    }

    /**
     * 查询用户映射
     *
     * @param tenantId 租户ID
     * @param req      查询参数
     * @return 用户映射列表
     */
    @Override
    public ResultPageBean queryUserMappings(String tenantId, FineReportQueryReq req) {
        // 分页查询
        LambdaQueryWrapper<FineReportUserMapping> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(FineReportUserMapping::getTenantId, tenantId);
        queryWrapper.orderByDesc(FineReportUserMapping::getModifyTime);
        Page<FineReportUserMapping> pageResult = fineReportUserMappingMapper.selectPage(new Page<>(req.getPageNum(), req.getPageSize()), queryWrapper);
        return ResultPageBean.success(ResponseBody.getInstance(pageResult.getRecords(), PageInfo.getPageInfo(req.getPageNum(), req.getPageSize(), (int) pageResult.getTotal())));
    }

    /**
     * 删除用户映射
     *
     * @param id 用户映射ID
     */
    @Override
    public void deleteUserMapping(Long id) {
        FineReportUserMapping fineReportUserMapping = fineReportUserMappingMapper.selectById(id);
        if (!Objects.isNull(fineReportUserMapping)) {
            fineReportUserMappingMapper.deleteById(id);
            FineReportConfigDTO fineReportConfigDTO = tripartiteApplicationService.queryFineReportConfig();
            // 调用帆软接口，没有配置就不调用了
            if (ObjectUtils.isNotEmpty(fineReportConfigDTO)) {
                fineReportProxyService.deleteFineReportUser(fineReportConfigDTO.getClientId(), fineReportConfigDTO.getSecret(), fineReportUserMapping.getFrUserId());
            }
        }
    }

    /**
     * 验证用户映射参数
     *
     * @param fineReportUserMapping 用户映射
     */
    private void validateUserMapping(FineReportUserMapping fineReportUserMapping, Long id) {
        Long userIdCount = fineReportUserMappingMapper.selectCount(new LambdaQueryWrapper<FineReportUserMapping>().eq(FineReportUserMapping::getUserId, fineReportUserMapping.getUserId())
                .eq(FineReportUserMapping::getTenantId, fineReportUserMapping.getTenantId()));
        if (!Objects.isNull(id) && userIdCount > 1) {
            throw new RuntimeException("用户已绑定");
        } else if (Objects.isNull(id) && userIdCount > 0) {
            throw new RuntimeException("用户已绑定");
        }
        // 验证帆软用户是否存在
        Long frUserIdCount = fineReportUserMappingMapper.selectCount(new LambdaQueryWrapper<FineReportUserMapping>().eq(FineReportUserMapping::getFrUserId, fineReportUserMapping.getFrUserId())
                .eq(FineReportUserMapping::getTenantId, fineReportUserMapping.getTenantId()));
        if (!Objects.isNull(id) && frUserIdCount > 1) {
            throw new RuntimeException("帆软用户已存在");
        } else if (Objects.isNull(id) && frUserIdCount == 1) {
            throw new RuntimeException("帆软用户已存在");
        }
    }

    /**
     * 查询当前用户映射关系
     *
     * @return 返回
     */
    @Override
    public FineReportUserMapping queryCurrentUserMapping() {
        LambdaUpdateWrapper<FineReportUserMapping> wrapper = new LambdaUpdateWrapper<>();
        wrapper.eq(FineReportUserMapping::getUserId, Utils.getUserId());
        wrapper.eq(FineReportUserMapping::getTenantId, Utils.getTenantId());
        List<FineReportUserMapping> userMappingList = fineReportUserMappingMapper.selectList(wrapper);
        if (CollectionUtils.isNotEmpty(userMappingList)) {
            return userMappingList.get(0);
        }
        return null;
    }

    /**
     * 查询雅典娜用户
     */
    @Override
    public List<RoleDTO>  queryAthenaUserList() {
        List<RoleDTO> roleDTOS = iamService.queryEnabledUserList();
        if (CollectionUtils.isNotEmpty(roleDTOS)) {
            String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
            LambdaQueryWrapper<FineReportUserMapping> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.select(FineReportUserMapping::getUserId);
            queryWrapper.eq(FineReportUserMapping::getTenantId, tenantId);
            List<FineReportUserMapping> fineReportUserMappings = fineReportUserMappingMapper.selectList(queryWrapper);
            if (CollectionUtils.isNotEmpty(fineReportUserMappings)) {
                List<String> userIds = fineReportUserMappings.stream().map(FineReportUserMapping::getUserId).collect(Collectors.toList());
                roleDTOS.forEach(roleDTO -> {
                    // 已经绑定过的账号不可再次绑定
                    if (userIds.contains(roleDTO.getId())) {
                        roleDTO.setDisabledRole(true);
                    }
                });
            }
            return roleDTOS;
        }
        return Collections.emptyList();
    }
}