package com.digiwin.athena.semc.service.homepage.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.dto.homepage.AdvertisementDTO;
import com.digiwin.athena.semc.dto.homepage.ReadAdvertisementReq;
import com.digiwin.athena.semc.entity.homepage.ReadAdvertisement;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.mapper.homepage.ReadAdvertisementMapper;
import com.digiwin.athena.semc.proxy.es.service.EsService;
import com.digiwin.athena.semc.service.cache.RedisLock;
import com.digiwin.athena.semc.service.homepage.IAdvertisementService;
import com.digiwin.athena.semc.util.DateUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @description 首页tbb消息通知列表
 * @author: sungqz
 * @create: 2023-8-23
 */
@Slf4j
@Service
public class AdvertisementServiceImpl implements IAdvertisementService {

    @Resource
    private EsService esService;

    @Resource
    private EnvProperties envProperties;

    @Autowired
    ReadAdvertisementMapper readAdvertisementMapper;

    /**
     * 查询es服务获取广告列表
     *
     * @param accessToken accessToken
     * @return
     */
    @Override
    public List<AdvertisementDTO> queryAdListFromEs(String accessToken) {
        String serviceRegion = envProperties.getCountry();
        List<AdvertisementDTO> adRespList = esService.queryAdList(serviceRegion, "DataAnalysis", accessToken);
        if (CollectionUtils.isEmpty(adRespList)) {
            return Lists.newArrayList();
        }

        // 查询当前租户下当前用户的已读广告
        List<ReadAdvertisement> readAdvertisementList = queryReadList();
        List<Long> adIdList = readAdvertisementList.stream().map(ReadAdvertisement::getAdvertisementId).collect(Collectors.toList());
        // 计算剩余时间
        Date nowDate = new Date();
        List<AdvertisementDTO> sortedList = adRespList.stream().sorted(Comparator.comparing(AdvertisementDTO::getEffectiveTime).reversed().thenComparing(AdvertisementDTO::getAdTitle)).collect(Collectors.toList());
        return sortedList.stream().peek(x -> {
            // 已读或未读状态
            x.setReadStatus(adIdList.contains(Long.valueOf(x.getId())));
            try {
                String dateDifference = DateUtils.calDateDifference(nowDate, new SimpleDateFormat(DateUtils.DATE_TIME_NORMAL_FORMATTER).parse(x.getLostEffectiveTime()));
                x.setRemainingTime(dateDifference);
            } catch (ParseException e) {
                log.error("DateUtils.calDateDifference calculate date difference occur error", e);
            }
        }).collect(Collectors.toList());
    }

    /**
     * 新增广告阅读记录
     *
     * @param readAdvertisementReq 读取广告请求对象
     * @return
     */
    @Override
    public boolean insertAdReadRecord(ReadAdvertisementReq readAdvertisementReq) {
        RedisLock redisLock = new RedisLock("insertAdvertisementRecord");
        try {
            // 开启分布式锁
            redisLock.lock();
            ReadAdvertisement readAdvertisement = readAdvertisementMapper.selectOne(new QueryWrapper<ReadAdvertisement>()
                    .eq("advertisement_id", readAdvertisementReq.getAdId()).eq("create_user_id", AppAuthContextHolder.getContext().getAuthoredUser().getUserId()));
            // 如果当前租户下当前用户是首次阅读，则新增阅读记录
            if (readAdvertisement == null) {
                readAdvertisement = new ReadAdvertisement();
                readAdvertisement.setAdvertisementId(readAdvertisementReq.getAdId());
                readAdvertisement.setAdvertisementTitle(readAdvertisementReq.getAdTitle());
                readAdvertisement.setImportanceFlag(readAdvertisementReq.getImportanceFlag());
                readAdvertisement.setReadCount(1);
                readAdvertisement.setReadDate(DateUtils.getNowTime(DateUtils.DATE_NORMAL_FORMATTER));
                readAdvertisement.setReadTime(DateUtils.getNowTime(DateUtils.TIME_NORMAL_FORMATTER));
                readAdvertisement.setFirstReadPlatform(readAdvertisementReq.getReadPlatform());
                readAdvertisement.setCreateUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
                readAdvertisement.setTenantId(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
                readAdvertisementMapper.insert(readAdvertisement);
            } else { // 阅读过，则阅读次数加1
                readAdvertisement.setReadCount(readAdvertisement.getReadCount() + 1);
                readAdvertisementMapper.updateById(readAdvertisement);
            }
        } catch (Exception e) {
            log.error("AdvertisementServiceImpl | insert read record occur error. param:{}", readAdvertisementReq, e);
            return false;
        } finally {
            redisLock.unlock();
        }
        return true;
    }

    /**
     * 查询当前租户下当前用户的已读列表
     *
     * @return
     */
    @Override
    public List<ReadAdvertisement> queryReadList() {
        return readAdvertisementMapper.selectList(new QueryWrapper<ReadAdvertisement>().eq("create_user_id", AppAuthContextHolder.getContext().getAuthoredUser().getUserId()));
    }
}
