package com.digiwin.athena.semc.service.homepage.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.enums.BizObjectTypeEnum;
import com.digiwin.athena.semc.common.enums.EAIServiceNameEnum;
import com.digiwin.athena.semc.common.enums.HomeSearchTypeEnum;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.homepage.HomeSearchDTO;
import com.digiwin.athena.semc.dto.homepage.HomeSearchReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.common.BizObjAuthRel;
import com.digiwin.athena.semc.entity.portal.LabelSystem;
import com.digiwin.athena.semc.entity.portal.News;
import com.digiwin.athena.semc.entity.portal.Notice;
import com.digiwin.athena.semc.entity.sso.ThirdSsoInfo;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.mapper.common.FolderMapper;
import com.digiwin.athena.semc.mapper.portal.LabelMapper;
import com.digiwin.athena.semc.mapper.portal.LabelSystemMapper;
import com.digiwin.athena.semc.mapper.portal.NewsMapper;
import com.digiwin.athena.semc.mapper.portal.NoticeMapper;
import com.digiwin.athena.semc.mapper.sso.ThirdSsoInfoMapper;
import com.digiwin.athena.semc.proxy.athena.service.atmc.AtmcService;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.proxy.kbs.service.KbsService;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.service.common.IBizObjAuthRelService;
import com.digiwin.athena.semc.service.homepage.IHomeSearchService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.vo.portal.FineReportVo;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;


@Slf4j
@Service
public class HomeSearchServiceImpl implements IHomeSearchService {

    @Autowired
    private AppLinkService appLinkService;

    @Autowired
    private NewsMapper newsMapper;

    @Autowired
    private NoticeMapper noticeMapper;

    @Autowired
    private LabelMapper labelMapper;

    @Autowired
    private LabelSystemMapper labelSystemMapper;

    @Autowired
    private ThirdSsoInfoMapper thirdSsoInfoMapper;

    @Resource
    private ESPService espService;

    @Resource
    private AtmcService atmcService;

    @Resource
    private FolderMapper folderMapper;

    @Resource
    private IBizObjAuthRelService bizObjAuthRelService;

    @Resource
    private ICacheService cacheService;

    @Resource
    private KbsService kbsService;

    @Resource
    EnvProperties envProperties;

    /**
     * 全局搜索
     * 智能入口：友情链接应用、三方报表、三方待办、新闻公告
     * 小AI：项目、任务、报表、基础资料录入、发起项目
     *
     * @param homeSearchReq 请求实体
     * @param user          租户
     * @return
     */
    @Override
    public List<HomeSearchDTO> searchResult(HomeSearchReq homeSearchReq, AuthoredUser user) {
        List<HomeSearchDTO> searchResultList = new ArrayList<>();
        //报表list
        List<HomeSearchDTO> reportList = new ArrayList<>();
        long start = System.currentTimeMillis();
        // 点击 全部或者 0：项目卡 1：任务卡 2：发起项目 3：基础资料录入 4：报表
        //先从小AI获取搜索结果
        if (null == homeSearchReq.getType() || (null != homeSearchReq.getType() && homeSearchReq.getType().intValue() < HomeSearchTypeEnum.APP_LINK.getValue().intValue())) {
            List<HomeSearchDTO> aiResultList = selectAiResult(homeSearchReq, user);
            long endAI = System.currentTimeMillis();
            log.info("search result 小AI请求耗时:" + (endAI - start) + " ms");
            //组装顺序
            // a.项目：保持原有逻辑，项目卡
            //项目搜索结果
            List<HomeSearchDTO> itemList = aiResultList.stream().filter(item -> HomeSearchTypeEnum.ITEM_CARD.getValue().equals(item.getType())).collect(Collectors.toList());
            //名称按ASCII排序
            Collections.sort(itemList);
            searchResultList.addAll(itemList);

            //b.任务：智驱任务卡+三方待办
            List<HomeSearchDTO> taskList = aiResultList.stream().filter(item -> HomeSearchTypeEnum.TASK_CARD.getValue().equals(item.getType())).collect(Collectors.toList());
            //点击全部，或者点击任务时
            if (null == homeSearchReq.getType() || (null != homeSearchReq.getType() && homeSearchReq.getType().equals(HomeSearchTypeEnum.TASK_CARD.getValue()))) {
                //三方待办
                long startTodo = System.currentTimeMillis();
                List<HomeSearchDTO> toDoList = selectToDoEsp(homeSearchReq, user);
                long endTodo = System.currentTimeMillis();
                log.info("search result 三方待办请求耗时:" + (endTodo - startTodo) + " ms");
                taskList.addAll(toDoList);
            }
            //名称按ASCII排序
            Collections.sort(taskList);
            searchResultList.addAll(taskList);

            //c.发起项目：保持原有逻辑，发起项目的内容
            List<HomeSearchDTO> projectList = aiResultList.stream().filter(item -> HomeSearchTypeEnum.INITIATES_PROJECTS.getValue().equals(item.getType())).collect(Collectors.toList());
            //名称按ASCII排序
            Collections.sort(projectList);
            searchResultList.addAll(projectList);

            //d.基础资料录入：基础资料录入的菜单
            List<HomeSearchDTO> basicDataList = aiResultList.stream().filter(item -> HomeSearchTypeEnum.BASIC_DATA.getValue().equals(item.getType())).collect(Collectors.toList());
            //名称按ASCII排序
            Collections.sort(basicDataList);
            searchResultList.addAll(basicDataList);

            long startReport = System.currentTimeMillis();
            // f.报表：包含报表和三方报表
            reportList = aiResultList.stream().filter(item -> HomeSearchTypeEnum.REPORT.getValue().equals(item.getType())).collect(Collectors.toList());
            //点击全部，或者点击报表时
            if (null == homeSearchReq.getType() || (null != homeSearchReq.getType() && homeSearchReq.getType().equals(HomeSearchTypeEnum.REPORT.getValue()))) {
                //三方报表
                List<HomeSearchDTO> threeReportList = selectReport(homeSearchReq, user);
                reportList.addAll(threeReportList);
                //查询帆软报表
                List<HomeSearchDTO> fineReportList =  selectFineReport( homeSearchReq,  user);
                reportList.addAll(fineReportList);
            }
            //名称按ASCII排序
            Collections.sort(reportList);
            long endReport = System.currentTimeMillis();
            log.info("search result  报表请求耗时:" + (endReport - startReport) + " ms");
        }
        //点击全部，或者点击友情链接时
        if (null == homeSearchReq.getType() || (null != homeSearchReq.getType() && homeSearchReq.getType().equals(HomeSearchTypeEnum.APP_LINK.getValue()))) {
            long startAppLink = System.currentTimeMillis();
            //e.友情链接的应用：友情链接的应用
            List<HomeSearchDTO> appLinkList = selectAppLinkList(homeSearchReq, user);
            searchResultList.addAll(appLinkList);
            long endAppLink = System.currentTimeMillis();
            log.info("search result  友情链接请求耗时:" + (endAppLink - startAppLink) + " ms");
        }
        //将报表数据加入集合
        if (!CollectionUtils.isEmpty(reportList)) {
            searchResultList.addAll(reportList);
        }
        //点击全部，或者点击其他时
        if (null == homeSearchReq.getType() || (null != homeSearchReq.getType() && homeSearchReq.getType() == 6)) {
            // g.其他：新闻公告
            List<HomeSearchDTO> newsNoticeList = selectNewsNotice(homeSearchReq, user);
            searchResultList.addAll(newsNoticeList);
        }
        long end = System.currentTimeMillis();
        log.info("search result  请求总耗时:" + (end - start) + " ms");
        return searchResultList;
    }

    @Override
    public net.sf.json.JSONObject getKbsFileDetail(String fileId) {
        return kbsService.getFileDetail(fileId);
    }

    /**
     * 根据应用获取appid和回调地址
     *
     * @param appName
     * @param tenantId
     * @param ssoInfoMap
     * @return
     */
    public HomeSearchDTO getSsoInfo(String appName, String tenantId, Map<String, HomeSearchDTO> ssoInfoMap) {
        String mapKey = tenantId + appName;
        //先从map中获取，map中没有从数据库中查询
        HomeSearchDTO sslInfoDTOMap = ssoInfoMap.get(mapKey);
        if (null != sslInfoDTOMap) {
            return sslInfoDTOMap;
        }
        HomeSearchDTO sslInfoDTO = new HomeSearchDTO();
        //取sso配置
        List<ThirdSsoInfo> thirdSsoInfoList = thirdSsoInfoMapper.querySsoInfoBy(appName);
        if (!CollectionUtils.isEmpty(thirdSsoInfoList)) {
            sslInfoDTO.setAppId(thirdSsoInfoList.get(0).getAppId());
            sslInfoDTO.setCallBackUrl(thirdSsoInfoList.get(0).getCallBackUrl());
            sslInfoDTO.setSystemType(thirdSsoInfoList.get(0).getExtend());
            sslInfoDTO.setUserBindFlag(thirdSsoInfoList.get(0).getUserBindFlag());
            sslInfoDTO.setAppToken(thirdSsoInfoList.get(0).getAppToken());
            ssoInfoMap.put(mapKey, sslInfoDTO);
        }
        return sslInfoDTO;
    }

    /**
     * 小AI：项目、任务、报表、基础资料录入、发起项目
     *
     * @param homeSearchReq
     * @param user
     * @return
     */
    public List<HomeSearchDTO> selectAiResult(HomeSearchReq homeSearchReq, AuthoredUser user) {
        List<HomeSearchDTO>  aiList = atmcService.searchResult(homeSearchReq.getName(), user.getToken(), homeSearchReq.getType());
        for (HomeSearchDTO ai : aiList) {
            ai.setCopyCode(ai.getCode());
        }
        return aiList;
    }

    /**
     * 从ESP查询三方报表
     *
     * @param homeSearchReq
     * @param user
     * @return
     */
    public List<HomeSearchDTO> selectToDoEsp(HomeSearchReq homeSearchReq, AuthoredUser user) {
        List<HomeSearchDTO> todoEspList = new ArrayList<>();
        try {
            Long labelId = labelMapper.getByLabelKey(Constants.LabelEnum.THIRD_TO_DO.getKey());
            if (null == labelId) {
                return todoEspList;
            }
            Map<String, HomeSearchDTO> ssoInfoMap = new HashMap<>();
            List<LabelSystem> labelSystemList = labelSystemMapper.selectByLabelId(labelId, "");
			Map<String, String> extHeader = new HashMap<>();
            extHeader.put("digi-userToken", user.getToken());
            for (LabelSystem labelSystem : labelSystemList) {

                //取sso配置
                HomeSearchDTO searcshTodoMap = getSsoInfo(labelSystem.getSystemName(), user.getTenantId(), ssoInfoMap);
                searcshTodoMap.setAppToken(StringUtils.isBlank(searcshTodoMap.getAppToken()) ? envProperties.getAppToken() : searcshTodoMap.getAppToken());
                try {
                    extHeader.put("digi-appToken", searcshTodoMap.getAppToken());
                    // 通过 EAI 调用三方待办获取待办数据
                    Map<String, Object> result = espService.queryByEsp(labelSystem.getMiddleSystemName(), labelSystem.getMiddleSystemUid(),
                            EAIServiceNameEnum.TODOLIST.getServiceName(), extHeader, null, homeSearchReq.getName(),
                            PageInfo.getPageInfo(Constants.pageNum, Constants.pageSize));
                    if (Objects.nonNull(result.get("to_do_list"))) {
                        JSONArray array = JSONArray.parseArray(JSONObject.toJSONString(result.get("to_do_list")));
                        for (int i = 0; i < array.size(); i++) {
                            JSONObject jsonObject = array.getJSONObject(i);
                            String toDoDataStr=jsonObject.getString("to_do_data");
                            JSONObject toDoData = JSONObject.parseObject(toDoDataStr);
                            String toDoUrl = jsonObject.getString("to_do_url");
                            toDoUrl = toDoUrl + (toDoUrl.contains("?") ? "&" : "?") + "appToken=" + searcshTodoMap.getAppToken() + "&tenantId=" + user.getTenantId() + "&bindMode=" + searcshTodoMap.getUserBindFlag();
                            HomeSearchDTO todoDTO = new HomeSearchDTO();
                            String name="";
                            String title = toDoData.getString("title");
                            if(StringUtils.isNotEmpty(title)){
                                name=title;
                            }
                            String subject = toDoData.getString("Subject");
                            if(StringUtils.isEmpty(name)&& StringUtils.isNotEmpty(subject)){
                                name=subject;
                            }
                            String taskName = toDoData.getString("task_name");
                            if(StringUtils.isEmpty(name)&&StringUtils.isNotEmpty(taskName)){
                                name=taskName;
                            }
                            todoDTO.setName(name);
                            todoDTO.setProjectDefCode(toDoUrl);
                            todoDTO.setType(HomeSearchTypeEnum.THREE_TODO.getValue());
                            todoDTO.setAppId(searcshTodoMap.getAppId());
                            todoDTO.setSystemType(searcshTodoMap.getSystemType());
                            todoDTO.setSystemId(labelSystem.getId());
                            todoDTO.setCallBackUrl(searcshTodoMap.getCallBackUrl());
                            //搜索匹配
                            if(StringUtils.isNotEmpty(todoDTO.getName())&& toDoDataStr.contains(homeSearchReq.getName())){
                                todoEspList.add(todoDTO);
                            }

                        }
                    }
                } catch (Exception e) {
                    log.error("search result 查询三方待办ESP error", e);
                }
            }
        } catch (Exception e) {
            log.error("search result 查询第三方报表 exception", e);
            return todoEspList;
        }
        return todoEspList;
    }

    /**
     * 知识中台文档搜索
     *
     * @param homeSearchReq
     * @return
     */
    public List<HomeSearchDTO> selectKbsFile(HomeSearchReq homeSearchReq) {
        List<HomeSearchDTO> fileList = new ArrayList<>();
        try {
            net.sf.json.JSONArray kbsFileJsonList= kbsService.selectKnowledgeInfo(homeSearchReq.getName());
            List<String> fileIds=new ArrayList<>();
            for (Object kbsFileJson : kbsFileJsonList) {
                net.sf.json.JSONObject jsonObjectKnowledgeInfo = net.sf.json.JSONObject.fromObject(kbsFileJson);
                String knowledgeTitle = (String) jsonObjectKnowledgeInfo.get("knowledge_title");
                String knowledgeId = (String) jsonObjectKnowledgeInfo.get("knowledge_id");
                HomeSearchDTO fileDTO = new HomeSearchDTO();
                if(fileIds.contains(knowledgeId)){
                    continue;
                }
                fileDTO.setName(knowledgeTitle);
                fileDTO.setCode(knowledgeId);
                fileDTO.setId(knowledgeId);
                fileDTO.setType(HomeSearchTypeEnum.KBS_FILE.getValue());
                fileList.add(fileDTO);
                fileIds.add(knowledgeId);
            }
        } catch (Exception e) {
            log.error("search result 知识中台文档搜索 exception", e);
            return fileList;
        }
        return fileList;
    }
    /**
     * 新闻公告
     *
     * @param homeSearchReq
     * @param user
     * @return
     */
    public List<HomeSearchDTO> selectNewsNotice(HomeSearchReq homeSearchReq, AuthoredUser user) {
        List<HomeSearchDTO> newsNoticeList = new ArrayList<>();
        try {
            Map<String, HomeSearchDTO> ssoInfoMap = new HashMap<>();
            //查询用户新闻信息
            News news = new News();
            news.setNewsUser(user.getUserId());
            news.setTenantId(user.getTenantId());
            news.setNewsTitle(homeSearchReq.getName());
            List<News> newsList = newsMapper.selectByUser(news);
            for (News newsDb : newsList) {
                HomeSearchDTO homeSearchDTO = new HomeSearchDTO();
                homeSearchDTO.setName(newsDb.getNewsTitle());
                homeSearchDTO.setCode(newsDb.getId().toString());
                homeSearchDTO.setId(newsDb.getId().toString());
                homeSearchDTO.setType(HomeSearchTypeEnum.NEWS.getValue());
                homeSearchDTO.setProjectDefCode(newsDb.getNewsUrl());
                //根据应用获取appid和回调地址
                HomeSearchDTO searcshTodoMap = getSsoInfo(newsDb.getAppName(), user.getTenantId(), ssoInfoMap);
                homeSearchDTO.setAppId(searcshTodoMap.getAppId());
                homeSearchDTO.setCallBackUrl(searcshTodoMap.getCallBackUrl());
                homeSearchDTO.setSystemType(searcshTodoMap.getSystemType());
                homeSearchDTO.setStartTime(DateUtils.DateToString(newsDb.getPublishTime()));
                newsNoticeList.add(homeSearchDTO);
            }
            //查询用户新闻信息
            Notice noticeQry = new Notice();
            noticeQry.setNoticeUser(user.getUserId());
            noticeQry.setTenantId(user.getTenantId());
            noticeQry.setNoticeTitle(homeSearchReq.getName());
            List<Notice> noticeList = noticeMapper.selectByUser(noticeQry);
            for (Notice notice : noticeList) {
                HomeSearchDTO homeSearchDTO = new HomeSearchDTO();
                homeSearchDTO.setName(notice.getNoticeTitle());
                homeSearchDTO.setCode(notice.getId().toString());
                homeSearchDTO.setId(notice.getId().toString());
                homeSearchDTO.setType(HomeSearchTypeEnum.NOTICE.getValue());
                homeSearchDTO.setProjectDefCode(notice.getNoticeUrl());
                //根据应用获取appid和回调地址
                HomeSearchDTO searcshTodoMap = getSsoInfo(notice.getAppName(), user.getTenantId(), ssoInfoMap);
                homeSearchDTO.setAppId(searcshTodoMap.getAppId());
                homeSearchDTO.setCallBackUrl(searcshTodoMap.getCallBackUrl());
                homeSearchDTO.setSystemType(searcshTodoMap.getSystemType());
                homeSearchDTO.setStartTime(DateUtils.DateToString(notice.getPublishTime()));
                newsNoticeList.add(homeSearchDTO);
            }
        } catch (Exception e) {
            log.error("search result selectNewsNotice exception", e);
            return newsNoticeList;
        }
        //名称按ASCII排序
        Collections.sort(newsNoticeList);
        return newsNoticeList;
    }
    /**
     * 帆软报表
     *
     * @param homeSearchReq
     * @param user
     * @return
     */
    public List<HomeSearchDTO> selectFineReport(HomeSearchReq homeSearchReq, AuthoredUser user) {
        List<HomeSearchDTO> reportList = new ArrayList<>();
        try {
            //判断开关是否开启
            Object result = cacheService.get("FineReport-Availability:" + AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            Boolean isOpen=Objects.isNull(result) ? true : Boolean.parseBoolean(result.toString());
            //开关已经关闭
            if(!isOpen){
                return reportList;
            }
            // 查询当前用户 + 当前角色 + 当前组织的所有报表
            List<BizObjAuthRel> bizObjAuthRelList = bizObjAuthRelService.qryBizObjAuthRel(BizObjectTypeEnum.FINE_REPORT.getValue());
            if (CollectionUtils.isEmpty(bizObjAuthRelList)) {
                return reportList;
            }
            // 查询所有报表信息
            List<String> reportIdList = bizObjAuthRelList.stream().map(BizObjAuthRel::getBizObjId).collect(Collectors.toList());
            List<FineReportVo> fineReportVoList=folderMapper.fineReportBy(homeSearchReq.getName(),user.getTenantId(),reportIdList);
            for (FineReportVo fineReportVo : fineReportVoList) {
                HomeSearchDTO reportDTO = new HomeSearchDTO();
                reportDTO.setId(fineReportVo.getId()+"");
                reportDTO.setName(fineReportVo.getName());
                reportDTO.setProjectDefCode(fineReportVo.getUrl());
                reportDTO.setType(HomeSearchTypeEnum.FINE_REPORT.getValue());
                reportList.add(reportDTO);
            }
        } catch (Exception e) {
            log.error("search result 查询帆软报表 exception", e);
            return reportList;
        }
        return reportList;
    }
    /**
     * 三方报表
     *
     * @param homeSearchReq
     * @param user
     * @return
     */
    public List<HomeSearchDTO> selectReport(HomeSearchReq homeSearchReq, AuthoredUser user) {
        List<HomeSearchDTO> reportList = new ArrayList<>();
        try {
            Long labelId = labelMapper.getByLabelKey(Constants.LabelEnum.REPORT_FORMS.getKey());
            if (null == labelId) {
                return reportList;
            }
            Map<String, HomeSearchDTO> ssoInfoMap = new HashMap<>();
            List<LabelSystem> labelSystemList = labelSystemMapper.selectByLabelId(labelId, "");
			Map<String, String> extHeader = new HashMap<>();
            extHeader.put("digi-userToken", user.getToken());
            for (LabelSystem labelSystem : labelSystemList) {

                //取sso配置
                HomeSearchDTO searcshTodoMap = getSsoInfo(labelSystem.getSystemName(), user.getTenantId(), ssoInfoMap);
                searcshTodoMap.setAppToken(StringUtils.isBlank(searcshTodoMap.getAppToken()) ? envProperties.getAppToken() : searcshTodoMap.getAppToken());
                try {
                    // 通过 EAI 调用三方接口获取报表数据
					extHeader.put("digi-appToken", searcshTodoMap.getAppToken());
                    Map<String, Object> result = espService.queryByEsp(labelSystem.getMiddleSystemName(), labelSystem.getMiddleSystemUid(),
                            EAIServiceNameEnum.REPORT.getServiceName(), extHeader, null, homeSearchReq.getName(),
                            PageInfo.getPageInfo(Constants.pageNum, Constants.pageSize));
                    if (Objects.nonNull(result.get("report_info_list"))) {
                        JSONArray array = JSONArray.parseArray(JSONObject.toJSONString(result.get("report_info_list")));
                        for (int i = 0; i < array.size(); i++) {
                            JSONObject jsonObject = array.getJSONObject(i);
                            String reportName = jsonObject.getString("report_name");
                            String reportUrl = jsonObject.getString("report_url");
                            reportUrl = reportUrl + (reportUrl.contains("?") ? "&" : "?") + "appToken=" + searcshTodoMap.getAppToken() + "&tenantId=" + user.getTenantId() + "&bindMode=" + searcshTodoMap.getUserBindFlag();
                            String createTime = jsonObject.getString("create_time");
                            String belongModule = jsonObject.getString("belong_module");
                            HomeSearchDTO reportDTO = new HomeSearchDTO();
                            reportDTO.setName(reportName);
                            reportDTO.setCode(belongModule);
                            reportDTO.setStartTime(createTime);
                            reportDTO.setProjectDefCode(reportUrl);
                            reportDTO.setType(HomeSearchTypeEnum.THREE_REPORT.getValue());
                            reportDTO.setAppId(searcshTodoMap.getAppId());
                            reportDTO.setSystemType(searcshTodoMap.getSystemType());
                            reportDTO.setSystemId(labelSystem.getId());
                            reportDTO.setCallBackUrl(searcshTodoMap.getCallBackUrl());
                            //搜索匹配
                            if(StringUtils.isNotEmpty(reportDTO.getName())&& reportName.contains(homeSearchReq.getName())){
                                reportList.add(reportDTO);
                            }
                        }
                    }
                } catch (Exception e) {
                    log.error("search result 查询第三方报表 error", e);
                }
            }
        } catch (Exception e) {
            log.error("search result 查询第三方报表 exception", e);
            return reportList;
        }
        return reportList;
    }

    /**
     * 查询友情链接
     *
     * @param homeSearchReq
     * @param user
     * @return
     */
    public List<HomeSearchDTO> selectAppLinkList(HomeSearchReq homeSearchReq, AuthoredUser user) {
        List<HomeSearchDTO> appLinkList = new ArrayList<>();
        try {
            List<AppLinkDTO> appLinkDTOList = appLinkService.queryDisplayList(user);
            for (AppLinkDTO link : appLinkDTOList) {
                HomeSearchDTO homeSearchDTO = new HomeSearchDTO();
                homeSearchDTO.setName(link.getName());
                homeSearchDTO.setCode(link.getId());
                homeSearchDTO.setId(link.getId());
                homeSearchDTO.setAppToken(link.getAppToken());
                homeSearchDTO.setProjectDefCode(link.getCloudwebsite());
                homeSearchDTO.setIsNeedSsoUrl(link.getIsNeedSsoUrl());
                homeSearchDTO.setTaskDefCode(link.getAppToken());
                homeSearchDTO.setType(HomeSearchTypeEnum.APP_LINK.getValue());
                homeSearchDTO.setSystemType(link.getSystemType()+"");
                homeSearchDTO.setApplicationAppId(link.getApplicationAppId());
                homeSearchDTO.setProtocolType(link.getProtocolType());
                homeSearchDTO.setCallBackUrl(link.getCallBackUrl());
                //搜索关键词在友情链接中
                if (StringUtils.isNotEmpty(homeSearchDTO.getName())&& link.getName().contains(homeSearchReq.getName())) {
                    appLinkList.add(homeSearchDTO);
                }
            }
        } catch (Exception e) {
            log.error("search result 友情链接 exception", e);
            return appLinkList;
        }
        //名称按ASCII排序
        Collections.sort(appLinkList);
        return appLinkList;
    }

}
