package com.digiwin.athena.semc.service.menu.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.ResponseBody;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.menu.DeleteMenuTemplateReq;
import com.digiwin.athena.semc.dto.menu.QueryMenuTemplateReq;
import com.digiwin.athena.semc.dto.menu.UpdateMenuTemplateReq;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuTemplate;
import com.digiwin.athena.semc.entity.portal.PortalInfo;
import com.digiwin.athena.semc.mapper.menu.CustomizedMenuTemplateMapper;
import com.digiwin.athena.semc.mapper.portal.PortalInfoMapper;
import com.digiwin.athena.semc.proxy.km.service.KmService;
import com.digiwin.athena.semc.service.menu.CustomizedMenuTemplateService;
import com.digiwin.athena.semc.service.portal.PortalInfoService;
import com.digiwin.athena.semc.service.tenant.TenantAppService;
import com.digiwin.athena.semc.service.workbench.util.WorkbenchUtils;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 自定义菜单模板表(TCustomizedMenuTemplate)表服务实现类
 *
 * @author sunqgz
 * @since 2023-11-08
 */
@Slf4j
@Service
public class CustomizedMenuTemplateServiceImpl extends ServiceImpl<CustomizedMenuTemplateMapper, CustomizedMenuTemplate> implements CustomizedMenuTemplateService {

    @Resource
    private CustomizedMenuTemplateMapper customizedMenuTemplateMapper;

    @Resource
    private PortalInfoService portalInfoService;

    @Resource
    private PortalInfoMapper portalInfoMapper;

    @Resource
    private TenantAppService tenantAppService;

    @Autowired
    private KmService kmService;

    /**
     * 自定义菜单列表配置查询菜单模板列表
     *
     * @return 返回对象
     */
    @Override
    public PageInfoResp<CustomizedMenuTemplate> queryMenuTemplateList(QueryMenuTemplateReq queryMenuTemplateReq) {
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        boolean isOnlyAsa = tenantAppService.isOnlyAsaSalesPlan(tenantId);

        // 列表
        PageInfoResp<CustomizedMenuTemplate> pageInfoResp = new PageInfoResp<>();
        Integer startNum = (queryMenuTemplateReq.getPageNum() - 1) * queryMenuTemplateReq.getPageSize();
        List<CustomizedMenuTemplate> templateList = new ArrayList<>();

        // 娜娜销售包下，不展示通用菜单模板
        if (isOnlyAsa) {
            templateList = customizedMenuTemplateMapper.queryTenantTempPageList(queryMenuTemplateReq.getTemplateName(), queryMenuTemplateReq.getTemplateStatus(),
                    startNum, queryMenuTemplateReq.getPageSize(), tenantId);
        } else {
            templateList = customizedMenuTemplateMapper.queryTempPageList(queryMenuTemplateReq.getTemplateName(), queryMenuTemplateReq.getTemplateStatus(),
                    startNum, queryMenuTemplateReq.getPageSize(), tenantId);
        }

        if (CollectionUtils.isEmpty(templateList)) {
            return pageInfoResp;
        }
        // 总数
        Integer totalCount = 0;
        if (isOnlyAsa) {
            totalCount = customizedMenuTemplateMapper.queryTenantTempCount(queryMenuTemplateReq.getTemplateName(), queryMenuTemplateReq.getTemplateStatus(), tenantId);
        } else {
            totalCount = customizedMenuTemplateMapper.queryTempCount(queryMenuTemplateReq.getTemplateName(), queryMenuTemplateReq.getTemplateStatus(), tenantId);
        }

        int totalPages = totalCount % queryMenuTemplateReq.getPageSize() == 0 ? totalCount / queryMenuTemplateReq.getPageSize()
                : totalCount / queryMenuTemplateReq.getPageSize() + 1;
        pageInfoResp.setTotalPages(totalPages);
        pageInfoResp.setPageNo(queryMenuTemplateReq.getPageNum());
        pageInfoResp.setPageSize(queryMenuTemplateReq.getPageSize());
        pageInfoResp.setTotalRecords(totalCount);

        // 娜娜销售包没有门户概念
        if (!isOnlyAsa) {
            // 查询关联的门户
            QueryWrapper<PortalInfo> portalInfoWrapper = new QueryWrapper<>();
            portalInfoWrapper.select("id", "menu_template_id", "name");
            List<PortalInfo> portalInfoList = portalInfoMapper.selectList(portalInfoWrapper);
            if (CollectionUtils.isEmpty(portalInfoList)) {
                pageInfoResp.setList(templateList);
                return pageInfoResp;
            }

            Map<Long, List<PortalInfo>> menuTempIdMap = portalInfoList.stream().collect(Collectors.groupingBy(PortalInfo::getMenuTemplateId));
            templateList.forEach(x -> {
                List<PortalInfo> portalList = menuTempIdMap.get(x.getId());
                if (CollectionUtils.isNotEmpty(portalList)) {
                    List<String> nameList = portalList.stream().map(PortalInfo::getName).collect(Collectors.toList());
                    x.setPortalList(nameList);
                }
            });
        }

        pageInfoResp.setList(templateList);
        return pageInfoResp;
    }

    @Override
    public List<CustomizedMenuTemplate> queryEnableMenuTemplateList() {
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        boolean isOnlyAsa = tenantAppService.isOnlyAsaSalesPlan(tenantId);

        List<CustomizedMenuTemplate> templateList = new ArrayList<>();

        // 娜娜销售包下，不展示通用菜单模板
        if (isOnlyAsa) {
            templateList = customizedMenuTemplateMapper.queryTenantEnableTempList(null, tenantId);
        } else {
            templateList = customizedMenuTemplateMapper.queryEnableTempList(null, tenantId);
        }

        return templateList;
    }

    /**
     * 更新菜单模板状态
     *
     * @param updateMenuTemplateReq 入参
     * @return 返回对象
     */
    @Override
    public Boolean updateStatus(UpdateMenuTemplateReq updateMenuTemplateReq) {
        // 单个更新
        if (updateMenuTemplateReq.getIdList().size() == Constants.NUMBER_ONE) {
            CustomizedMenuTemplate menuTemplate = new CustomizedMenuTemplate();
            menuTemplate.setId(updateMenuTemplateReq.getIdList().get(0));
            menuTemplate.setTemplateStatus(updateMenuTemplateReq.getTemplateStatus());
            menuTemplate.setModifyUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
            customizedMenuTemplateMapper.updateById(menuTemplate);
        } else {
            // 过滤掉关联了门户的菜单模板
            List<PortalInfo> portalInfoList = portalInfoService.queryPortalByMenuTempId(updateMenuTemplateReq.getIdList());
            List<Long> idList = portalInfoList.stream().map(PortalInfo::getMenuTemplateId).distinct().collect(Collectors.toList());
            updateMenuTemplateReq.getIdList().removeAll(idList);

            if (CollectionUtils.isNotEmpty(updateMenuTemplateReq.getIdList())) {
                UpdateWrapper<CustomizedMenuTemplate> updateWrapper = new UpdateWrapper<>();
                updateWrapper.set("template_status", updateMenuTemplateReq.getTemplateStatus())
                        .set("modify_user_id", AppAuthContextHolder.getContext().getAuthoredUser().getUserId())
                        .set("modify_user_name", AppAuthContextHolder.getContext().getAuthoredUser().getUserName())
                        .set("modify_time", LocalDateTime.now());
                updateWrapper.in("id", updateMenuTemplateReq.getIdList());
                this.update(updateWrapper);
            }
        }
        return true;
    }

    /**
     * 通过模板id主键删除数据
     *
     * @param deleteMenuTemplateReq 主键
     * @return 是否成功
     */
    @Override
    public Boolean deleteTemplateById(DeleteMenuTemplateReq deleteMenuTemplateReq) {
        if (deleteMenuTemplateReq.getIdList().size() == Constants.NUMBER_ONE) {
            customizedMenuTemplateMapper.delete(new QueryWrapper<CustomizedMenuTemplate>().eq("id", deleteMenuTemplateReq.getIdList().get(0)));
        } else {
            // 过滤掉关联了门户的菜单模板
            List<PortalInfo> portalInfoList = portalInfoService.queryPortalByMenuTempId(deleteMenuTemplateReq.getIdList());
            List<Long> idList = portalInfoList.stream().map(PortalInfo::getMenuTemplateId).distinct().collect(Collectors.toList());
            deleteMenuTemplateReq.getIdList().removeAll(idList);
            if (CollectionUtils.isNotEmpty(deleteMenuTemplateReq.getIdList())) {
                customizedMenuTemplateMapper.delete(new QueryWrapper<CustomizedMenuTemplate>().in("id", deleteMenuTemplateReq.getIdList()));
            }
        }
        return true;
    }

    /**
     * 门户布局查询菜单模板列表
     *
     * @param queryMenuTemplateReq 入参
     * @param portalMenuTempId     门户关联的菜单模板id
     * @return 返回
     */
    @Override
    public ResultPageBean queryPortalTempList(QueryMenuTemplateReq queryMenuTemplateReq, Long portalMenuTempId) {
        // 查询当前租户下所有已发布的菜单模板
        List<CustomizedMenuTemplate> templateList = customizedMenuTemplateMapper.queryEnableTempList(queryMenuTemplateReq.getTemplateName(), AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());

        // 查询当前门户关联的复制出来的isv菜单模板
        CustomizedMenuTemplate isvMenuTemplate = null;
        if (queryMenuTemplateReq.getPortalId() != null) {
            LambdaQueryWrapper<CustomizedMenuTemplate> wrapper = new LambdaQueryWrapper<>();
            wrapper.eq(CustomizedMenuTemplate::getPortalId, queryMenuTemplateReq.getPortalId());
            isvMenuTemplate = InterceptorIgnoreUtil.handler(() -> customizedMenuTemplateMapper.selectOne(wrapper));
            if (ObjectUtils.isNotEmpty(isvMenuTemplate)) {
                templateList.add(isvMenuTemplate);
            }
        }
        if (CollectionUtils.isEmpty(templateList)) {
            return ResultPageBean.success(ResponseBody.getInstance(templateList, PageInfo.getPageInfo(queryMenuTemplateReq.getPageNum(), queryMenuTemplateReq.getPageSize(), 0)));
        }

        // 构建当前门户关联的菜单
        Map<Long, CustomizedMenuTemplate> templateIdMap = templateList.stream().collect(Collectors.toMap(CustomizedMenuTemplate::getId, Function.identity()));
        if (ObjectUtils.isNotEmpty(portalMenuTempId) && ObjectUtils.isNotEmpty(templateIdMap.get(portalMenuTempId))) {
            templateIdMap.get(portalMenuTempId).setRelatedFlag(Boolean.TRUE);

            // 排序
            List<CustomizedMenuTemplate> relatedTemp = templateList.stream().filter(x -> x.getRelatedFlag().equals(Boolean.TRUE)).collect(Collectors.toList());
            templateList.removeIf(x -> x.getRelatedFlag().equals(Boolean.TRUE));
            CustomizedMenuTemplate current = relatedTemp.get(0);
            templateList.add(0, current);
            if (isvMenuTemplate != null) {
                if (portalMenuTempId.equals(isvMenuTemplate.getId())) {
                    // 选中菜单 = 选择ISV模板
                    // 第一条数据为用户所选的 xx门户-菜单
                    // 第二条数据为 通用菜单
                    // 第三条数据为 其他。。。菜单
                    List<CustomizedMenuTemplate> generalTemp = templateList.stream().filter(x -> x.getGeneralFlag().equals(Constants.GENERAL_MENU_FLAG_YES)).collect(Collectors.toList());
                    if (CollectionUtils.isNotEmpty(generalTemp)) {
                        templateList.remove(generalTemp.get(0));
                        templateList.add(1, generalTemp.get(0));
                    }
                } else {
                    // 选中菜单 != 选择ISV模板
                    // 第一条为选中的菜单
                    //第二条为 xx门户 -菜单
                    templateList.remove(isvMenuTemplate);
                    templateList.add(1, isvMenuTemplate);
                    //第三条为 通用菜单
                    List<CustomizedMenuTemplate> generalTemp = templateList.stream().filter(x -> x.getGeneralFlag().equals(Constants.GENERAL_MENU_FLAG_YES)).collect(Collectors.toList());
                    if (CollectionUtils.isNotEmpty(generalTemp)) {
                        if (!current.getId().equals(generalTemp.get(0).getId())) {
                            // 选中菜单 != 通用菜单
                            templateList.remove(generalTemp.get(0));
                            templateList.add(2, generalTemp.get(0));
                        }
                    }
                    //第四条为 其他。。。菜单
                }
            } else {
                // 第一条为选中的菜单
                // 第二条数据为 通用菜单
                // 第三条数据为 其他。。。菜单
                List<CustomizedMenuTemplate> generalTemp = templateList.stream().filter(x -> x.getGeneralFlag().equals(Constants.GENERAL_MENU_FLAG_YES)).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(generalTemp)) {
                    if (!current.getId().equals(generalTemp.get(0).getId())) {
                        // 选中菜单 != 通用菜单
                        templateList.remove(generalTemp.get(0));
                        templateList.add(1, generalTemp.get(0));
                    }
                }
            }
        } else {
            List<CustomizedMenuTemplate> generalTemp = templateList.stream().filter(x -> x.getGeneralFlag().equals(Constants.GENERAL_MENU_FLAG_YES)).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(generalTemp)) {
                generalTemp.get(0).setRelatedFlag(Boolean.TRUE);
                templateList.removeIf(x -> x.getRelatedFlag().equals(Boolean.TRUE));
                templateList.add(0, generalTemp.get(0));
            }
        }
        ResultPageBean resultPageBean = ResultPageBean.success(ResponseBody.getInstance(templateList.stream()
                .skip((queryMenuTemplateReq.getPageNum() - 1) * queryMenuTemplateReq.getPageSize().longValue())
                .limit(queryMenuTemplateReq.getPageSize()).collect(Collectors.toList()), PageInfo.getPageInfo(
                queryMenuTemplateReq.getPageNum(), queryMenuTemplateReq.getPageSize(), templateList.size())));
        return resultPageBean;
    }

    /**
     * 检验模板名称是否存在
     *
     * @param templateId   模板id
     * @param templateName 模板名称
     * @return 返回对象
     */
    @Override
    public Boolean validExist(Long templateId, String templateName) {
        List<CustomizedMenuTemplate> templateList = customizedMenuTemplateMapper.validExist(templateName, templateId, AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        return CollectionUtils.isNotEmpty(templateList);
    }

    /**
     * 通过模板id查询模板
     *
     * @param templateId 模板id
     * @return 实例对象
     */
    @Override
    public CustomizedMenuTemplate queryTemplateById(Long templateId) {
        QueryWrapper<CustomizedMenuTemplate> wrapper = new QueryWrapper<>();
        wrapper.eq("id", templateId);
        wrapper.in("tenant_id", WorkbenchUtils.getTenantIds());
        return InterceptorIgnoreUtil.handler(() -> customizedMenuTemplateMapper.selectOne(wrapper));
    }

    @Override
    public CustomizedMenuTemplate queryTemplateByDesignerId(Long menuDesignerId) {
        // 查询租户版本
        String tenantVersion = kmService.queryTenantVersionWithDefault();
        LambdaQueryWrapper<CustomizedMenuTemplate> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(CustomizedMenuTemplate::getMenuDesignerId, menuDesignerId);
        wrapper.eq(CustomizedMenuTemplate::getVersion, tenantVersion);
        return InterceptorIgnoreUtil.handler(() -> customizedMenuTemplateMapper.selectOne(wrapper));
    }

    @Override
    public List<CustomizedMenuTemplate> queryTemplateByIds(List<Long> templateIds) {
        return InterceptorIgnoreUtil.handler(() -> customizedMenuTemplateMapper.selectBatchIds(templateIds));
    }

    /**
     * 更新其余模板状态为未发布
     *
     * @param updateMenuTemplateReq 入参
     * @return 返回对象
     */
    @Override
    public Boolean unusableOtherTemplate(UpdateMenuTemplateReq updateMenuTemplateReq, String tenantId) {
        customizedMenuTemplateMapper.updateOtherTemplateStatus(updateMenuTemplateReq.getIdList().get(0), Constants.VALID_STATUS_UNUSABLE, tenantId);
        return true;
    }
}