package com.digiwin.athena.semc.service.menu.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.menu.SaveMenuReq;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuTemplate;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuWork;
import com.digiwin.athena.semc.entity.portal.LabelSystemAuth;
import com.digiwin.athena.semc.entity.portal.LabelSystemCustom;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.portal.LabelSystemPre;
import com.digiwin.athena.semc.mapper.menu.CustomizedMenuTemplateMapper;
import com.digiwin.athena.semc.mapper.menu.CustomizedMenuWorkMapper;
import com.digiwin.athena.semc.mapper.portal.LabelSystemCustomMapper;
import com.digiwin.athena.semc.mapper.portal.LabelSystemDataMapper;
import com.digiwin.athena.semc.mapper.portal.LabelSystemPreMapper;
import com.digiwin.athena.semc.proxy.athena.service.audc.AudcService;
import com.digiwin.athena.semc.proxy.athena.service.audc.model.AuthorityMenuDTO;
import com.digiwin.athena.semc.proxy.athena.service.audc.model.SystemMenuDTO;
import com.digiwin.athena.semc.proxy.km.service.KmService;
import com.digiwin.athena.semc.proxy.km.service.model.AppJobDTO;
import com.digiwin.athena.semc.proxy.trans.service.TranslateService;
import com.digiwin.athena.semc.service.menu.CustomizedMenuWorkService;
import com.digiwin.athena.semc.service.mobile.MobileUserAuthService;
import com.digiwin.athena.semc.service.portal.LabelSystemAuthService;
import com.digiwin.athena.semc.service.workbench.constant.PublishSourceEnum;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.common.UserAuthSystemLabelVo;
import com.digiwin.athena.semc.vo.menu.CustomizedMenuWorkResp;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.digiwin.athena.semc.common.Constants.SYSTEM_MENU_FLAG_YES;

/**
 * 自定义菜单作业表(TCustomizedMenuWork)表服务实现类
 *
 * @author sunqgz
 * @since 2023-11-10
 */
@Slf4j
@Service
public class CustomizedMenuWorkServiceImpl extends ServiceImpl<CustomizedMenuWorkMapper, CustomizedMenuWork> implements CustomizedMenuWorkService {

    @Resource
    private CustomizedMenuWorkMapper customizedMenuWorkMapper;

    @Resource
    private CustomizedMenuTemplateMapper customizedMenuTemplateMapper;

    @Resource
    private LabelSystemDataMapper labelSystemDataMapper;

    @Resource
    private LabelSystemPreMapper labelSystemPreMapper;

    @Resource
    private LabelSystemCustomMapper labelSystemCustomMapper;

    @Resource
    private KmService kmService;

    @Resource
    private AudcService audcService;

    @Resource
    private TranslateService translateService;

    @Resource
    private MobileUserAuthService mobileUserAuthService;

    @Autowired
    private LabelSystemAuthService labelSystemAuthService;

    @Resource
    private Converter converter;

    /**
     * 小AI权限接口中针对系统菜单权限的判断类型
     */
    public static final String ATMC_SYS_MENU_FLAG = "sysMenu";

    /**
     * 作业类型的映射关系(km作业类型映射小AI作业类型)
     */
    private static final Map<String, String> jobTypeMap;

    static {
        jobTypeMap = new HashMap<>();
        jobTypeMap.put("baseDataEntry", "baseDataEntry");
        jobTypeMap.put("statement", "report");
        jobTypeMap.put("manualStartProject", "startProject");
    }


    /**
     * 查询当前模板的菜单树
     *
     * @param templateId 模板id
     * @return 返回对象
     */
    @Override
    public List<CustomizedMenuWorkResp> queryManageMenu(Long templateId) {
        List<CustomizedMenuWorkResp> menuRespList = Lists.newArrayList();
        // 查询菜单列表
        List<CustomizedMenuWork> menuList = InterceptorIgnoreUtil.handler(() -> customizedMenuWorkMapper.selectList(getMenuWorkWrapper(templateId)));
        if (CollectionUtils.isEmpty(menuList)) {
            return menuRespList;
        }
        // 根据id转换为树结构
        for (CustomizedMenuWork menuWork : menuList) {
            // 找到根节点
            if (ObjectUtils.isEmpty(menuWork.getParentId())) {
                CustomizedMenuWorkResp resp = converter.convert(menuWork, CustomizedMenuWorkResp.class);
                menuRespList.add(findChildById(resp, menuList));
            }
        }
        return menuRespList;
    }

    /**
     * 获取组件类型的菜单名称
     *
     * @param menuList 菜单列表
     */
    private void buildLabelMenuName(List<CustomizedMenuWork> menuList) {
        // 预设组件
        List<Long> preLabelIdList = menuList.stream().filter(x -> Constants.CustomizedMenuTypeEnum.LABEL.getType().equals(x.getMenuType())
                && Constants.LabelTypeEnum.SYSTEM_PRE.getVal().equals(x.getMenuLabelType())).map(CustomizedMenuWork::getMenuLabelId).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(preLabelIdList)) {
            List<LabelSystemPre> preLabelList = labelSystemPreMapper.selectBatchIds(preLabelIdList);
            Map<Long, LabelSystemPre> preIdMap = preLabelList.stream().collect(Collectors.toMap(LabelSystemPre::getId, Function.identity(), (a, b) -> a));
            menuList.forEach(x -> {
                if (Constants.CustomizedMenuTypeEnum.LABEL.getType().equals(x.getMenuType())
                        && Constants.LabelTypeEnum.SYSTEM_PRE.getVal().equals(x.getMenuLabelType())) {
                    LabelSystemPre systemPre = preIdMap.get(x.getId());
                    x.setMenuName(Constants.ZH_TW_LOCALE.equals(LocaleContextHolder.getLocale().toString()) && ObjectUtils.isNotEmpty(systemPre) ? systemPre.getNameTw() : systemPre.getName());
                }
            });
        }

        // 自定义组件
        List<Long> customLabelIdList = menuList.stream().filter(x -> Constants.CustomizedMenuTypeEnum.LABEL.getType().equals(x.getMenuType())
                && Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal().equals(x.getMenuLabelType())).map(CustomizedMenuWork::getMenuLabelId).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(customLabelIdList)) {
            List<LabelSystemCustom> customLabelList = labelSystemCustomMapper.selectBatchIds(customLabelIdList);
            Map<Long, LabelSystemCustom> customIdMap = customLabelList.stream().collect(Collectors.toMap(LabelSystemCustom::getId, Function.identity(), (a, b) -> a));
            menuList.forEach(x -> {
                if (Constants.CustomizedMenuTypeEnum.LABEL.getType().equals(x.getMenuType())
                        && Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal().equals(x.getMenuLabelType())) {
                    LabelSystemCustom customLabel = customIdMap.get(x.getId());
                    x.setMenuName(ObjectUtils.isNotEmpty(customLabel) ? customLabel.getNameZh() : x.getMenuName());
                }
            });
        }
    }

    /**
     * 根据id构建菜单树
     *
     * @param resp         菜单对象
     * @param menuWorkList 菜单列表
     * @return 返回对象
     */
    private CustomizedMenuWorkResp findChildById(CustomizedMenuWorkResp resp, List<CustomizedMenuWork> menuWorkList) {
        // 子节点集合
        List<CustomizedMenuWorkResp> childList = Lists.newArrayList();
        for (CustomizedMenuWork node : menuWorkList) {
            if (Objects.equals(node.getParentId(), resp.getId())) {
                CustomizedMenuWorkResp childResp = converter.convert(node, CustomizedMenuWorkResp.class);
                // 递归调用
                childList.add(findChildById(childResp, menuWorkList));
            }
        }
        resp.setChildren(childList);
        return resp;
    }

    /**
     * 保存菜单
     *
     * @param saveMenuReq 入参
     * @param menuList    菜单列表
     * @return 返回对象
     */
    @Override
    @Transactional
    public Boolean saveMenu(SaveMenuReq saveMenuReq, List<CustomizedMenuWork> menuList) {
        // 保存菜单模板
        if (ObjectUtils.isEmpty(saveMenuReq.getTemplateId())) {
            CustomizedMenuTemplate customizedMenuTemplate = new CustomizedMenuTemplate();
            buildTemplate(saveMenuReq, customizedMenuTemplate);
            customizedMenuTemplate.setCreateUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
            customizedMenuTemplateMapper.insert(customizedMenuTemplate);
            saveMenuReq.setTemplateId(customizedMenuTemplate.getId());
        } else {
            CustomizedMenuTemplate customizedMenuTemplate = new CustomizedMenuTemplate();
            customizedMenuTemplate.setId(saveMenuReq.getTemplateId());
            buildTemplate(saveMenuReq, customizedMenuTemplate);
            customizedMenuTemplateMapper.updateById(customizedMenuTemplate);
        }

        // 删除原有的菜单
        QueryWrapper<CustomizedMenuWork> wrapper = new QueryWrapper<>();
        wrapper.eq("template_id", saveMenuReq.getTemplateId());
        customizedMenuWorkMapper.delete(wrapper);

        // 保存菜单
        if (CollectionUtils.isNotEmpty(menuList)) {
            for (int i = 0; i < menuList.size(); i++) {
                menuList.get(i).setTemplateId(saveMenuReq.getTemplateId());
                menuList.get(i).setMenuOrder(i);
            }
            return saveBatch(menuList);
        }
        return true;
    }

    /**
     * 构建模板对象
     *
     * @param saveMenuReq            入参
     * @param customizedMenuTemplate 模板
     */
    private void buildTemplate(SaveMenuReq saveMenuReq, CustomizedMenuTemplate customizedMenuTemplate) {
        customizedMenuTemplate.setTemplateName(saveMenuReq.getTemplateName());
        customizedMenuTemplate.setTemplateNameTw(saveMenuReq.getTemplateNameTw());
        customizedMenuTemplate.setTemplateDesc(saveMenuReq.getTemplateDesc());
        customizedMenuTemplate.setTemplateStatus(saveMenuReq.getTemplateStatus());
        customizedMenuTemplate.setSystemMenuFlag(saveMenuReq.getSystemMenuFlag());
        customizedMenuTemplate.setModifyUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
    }

    /**
     * 查询前台用户展示的菜单树
     *
     * @param templateId     模板id
     * @param systemMenuFlag 是否展示系统菜单 0:否 1:是
     * @return 返回对象
     */
    @Override
    public JSONObject queryDisplayMenu(Long templateId, Integer systemMenuFlag) {
        JSONObject menuResp = new JSONObject(true);
        // 查询当前模板菜单列表
        List<CustomizedMenuWork> menuList = InterceptorIgnoreUtil.handler(() -> customizedMenuWorkMapper.selectList(getMenuWorkWrapper(templateId)));
        for (int i = 0; i < menuList.size(); i++) {
            menuList.get(i).setMenuOrder(i);
        }
        // 查询小AI接口获取当前用户的作业权限
        List<AuthorityMenuDTO> authorityMenuList = queryAuthorityJobList(menuList);
        // 构建自定义菜单
        // 用于过滤自定义菜单作业（来自km作业）的类型
        Map<String, List<AuthorityMenuDTO>> kmTypeMap = authorityMenuList.stream().filter(x -> !ATMC_SYS_MENU_FLAG.equals(x.getType())).collect(Collectors.groupingBy(AuthorityMenuDTO::getType));
        List<CustomizedMenuWorkResp> menuRespList = buildCustomMenu(menuList, kmTypeMap);
        menuResp.put("customList", menuRespList);


        // 构建系统菜单
        if (SYSTEM_MENU_FLAG_YES.equals(systemMenuFlag)) {
            // 用于过滤小AI系统菜单的code
            List<String> systemCodeList = authorityMenuList.stream().filter(x -> ATMC_SYS_MENU_FLAG.equals(x.getType())).map(AuthorityMenuDTO::getCode).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(systemCodeList)) {
                List<SystemMenuDTO> systemMenuDTOList = buildSystemMenu(systemCodeList);
                menuResp.put("systemList", systemMenuDTOList);
            } else {
                menuResp.put("systemList", Lists.newArrayList());
            }
        } else {
            menuResp.put("systemList", Lists.newArrayList());
        }
        return menuResp;
    }

    /**
     * 查询小AI作业权限
     *
     * @param menuList 菜单列表
     * @return 返回
     */
    private List<AuthorityMenuDTO> queryAuthorityJobList(List<CustomizedMenuWork> menuList) {
        // 过滤出所有作业列表，用于构建查询作业权限接口的入参
        List<CustomizedMenuWork> workList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(menuList)) {
            workList = menuList.stream().filter(x -> Constants.CustomizedMenuTypeEnum.WORK.getType().equals(x.getMenuType())).collect(Collectors.toList());
        }
        JSONArray reqBody = new JSONArray();
        if (CollectionUtils.isNotEmpty(workList)) {
            Map<String, List<CustomizedMenuWork>> jobMap = workList.stream().filter(x -> !ObjectUtils.isEmpty(x.getAppCode())).collect(Collectors.groupingBy(CustomizedMenuWork::getAppCode));
            jobMap.forEach((x, y) -> {
                JSONObject item = new JSONObject();
                reqBody.add(item);
                JSONArray configArray = new JSONArray();
                item.put("appCode", x);
                item.put("jobCodeList", configArray);
                List<String> existList = Lists.newArrayList();
                for (CustomizedMenuWork o : y) {
                    JSONObject jsonObject = JSONObject.parseObject(o.getWorkConfig());
                    if (jsonObject.containsKey("code") && ObjectUtils.isNotEmpty(jsonObject.get("code"))
                            && jsonObject.containsKey("name") && ObjectUtils.isNotEmpty(jsonObject.get("name"))
                            && jsonObject.containsKey("type") && ObjectUtils.isNotEmpty(jsonObject.get("type"))) {
                        Object code = jsonObject.get("code");
                        Object name = jsonObject.get("name");
                        Object type = jsonObject.get("type");

                        // 添加作业参数
                        StringBuilder key = new StringBuilder().append(code).append(name).append(type);
                        if (!existList.contains(key.toString())) {
                            JSONObject configReq = new JSONObject();
                            configReq.put("code", code);
                            configReq.put("name", name);
                            String parseType = jobTypeMap.get(type.toString());
                            configReq.put("type", StringUtils.isBlank(parseType) ? type.toString() : parseType);
                            configArray.add(configReq);
                            existList.add(key.toString());
                        }
                    }
                }
            });
        }
        if (CollectionUtils.isEmpty(reqBody)) {
            JSONObject item = new JSONObject();
            item.put("appCode", "Athena");
            reqBody.add(item);
        }
        // 查询小AI权限接口
        return audcService.queryAuthorityMenu(reqBody);
    }

    /**
     * 构建查询菜单列表条件
     *
     * @param templateId 菜单模板id
     * @return 返回
     */
    public QueryWrapper<CustomizedMenuWork> getMenuWorkWrapper(Long templateId) {
        QueryWrapper<CustomizedMenuWork> condition = new QueryWrapper<>();
        condition.eq("template_id", templateId);
        condition.orderByAsc("menu_order");
        condition.orderByAsc("id");
//        condition.in("tenant_id", WorkbenchUtils.getTenantIds());
        return condition;
    }

    /**
     * 构建用户的自定义菜单
     *
     * @param menuList 菜单列表
     * @param typeMap  权限列表
     * @return 返回
     */
    private List<CustomizedMenuWorkResp> buildCustomMenu(List<CustomizedMenuWork> menuList, Map<String, List<AuthorityMenuDTO>> typeMap) {
        List<CustomizedMenuWorkResp> menuRespList = Lists.newArrayList();
        // 查询菜单列表
        if (CollectionUtils.isEmpty(menuList)) {
            return menuRespList;
        }
        // 不展示无作业的菜单
        List<CustomizedMenuWork> workLabelList = menuList.stream().filter(x -> Arrays.asList(Constants.CustomizedMenuTypeEnum.WORK.getType(), Constants.CustomizedMenuTypeEnum.LABEL.getType()).contains(x.getMenuType())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(workLabelList)) {
            return menuRespList;
        }

        // 过滤出当前用户有权限的作业
        List<CustomizedMenuWork> workList = filterPermissionWorkList(typeMap, menuList);
        // 过滤出当前用户有权限的自定义组件和预设组件
        List<CustomizedMenuWork> permissionLabelList = filterPermissionLabelList(menuList);
        workList.addAll(permissionLabelList);
        if (CollectionUtils.isEmpty(workList)) {
            return menuRespList;
        }

        // 过滤出包含作业的菜单
        List<CustomizedMenuWork> finalMenuList = Lists.newArrayList();
        List<Long> menuIdList = Lists.newArrayList();
        for (CustomizedMenuWork menu : menuList) {
            boolean existWork = workList.stream().anyMatch(x -> ObjectUtils.isNotEmpty(x.getParentId()) && x.getParentId().equals(menu.getId()));
            if (existWork && !menuIdList.contains(menu.getId())) {
                finalMenuList.add(menu);
                menuIdList.add(menu.getId());
            }
        }
        // 补充菜单，即一级菜单没有作业，但是二级子菜单有作业，例：A菜单下只有一个B菜单，B菜单下存在作业，则把A菜单添加进来
        List<CustomizedMenuWork> secondMenuList = finalMenuList.stream().filter(x -> Constants.CustomizedMenuLevelEnum.SECOND.getLevel().equals(x.getLevel())).collect(Collectors.toList());
        for (CustomizedMenuWork menu : menuList) {
            boolean matchFlag = secondMenuList.stream().anyMatch(x -> x.getParentId().equals(menu.getId()));
            if (matchFlag && !menuIdList.contains(menu.getId())) {
                finalMenuList.add(menu);
                menuIdList.add(menu.getId());
            }
        }
        finalMenuList.addAll(workList);
        //根据id升序
        finalMenuList = finalMenuList.stream().sorted(Comparator.comparing(CustomizedMenuWork::getMenuOrder)).collect(Collectors.toList());
        // 根据id转换为树结构
        for (CustomizedMenuWork menuWork : finalMenuList) {
            // 找到根节点
            if (ObjectUtils.isEmpty(menuWork.getParentId())) {
                CustomizedMenuWorkResp resp = converter.convert(menuWork, CustomizedMenuWorkResp.class);
                menuRespList.add(findChildById(resp, finalMenuList));
            }
        }
        return menuRespList;
    }

    /**
     * 过滤出有权限的作业
     *
     * @param typeMap  有权限的作业
     * @param menuList 菜单
     * @return 返回
     */
    private List<CustomizedMenuWork> filterPermissionWorkList(Map<String, List<AuthorityMenuDTO>> typeMap, List<CustomizedMenuWork> menuList) {
        if (MapUtils.isEmpty(typeMap)) {
            return Lists.newArrayList();
        }
        List<CustomizedMenuWork> workList = menuList.stream().filter(x -> Constants.CustomizedMenuTypeEnum.WORK.getType().equals(x.getMenuType())).collect(Collectors.toList());
        workList = workList.stream().filter(x -> {
            AppJobDTO.Job job = JsonUtils.jsonToObject(JsonUtils.objectToString(x.getWorkConfig()), AppJobDTO.Job.class);
            String jobType = jobTypeMap.get(job.getType());
            if (MapUtils.isNotEmpty(typeMap) && CollectionUtils.isNotEmpty(typeMap.get(jobType))) {
                List<String> authorityList = typeMap.get(jobType).stream().map(AuthorityMenuDTO::getCode).collect(Collectors.toList());
                return authorityList.contains(job.getCode());
            }
            return false;
        }).collect(Collectors.toList());
        return workList;
    }

    /**
     * 过滤出有权限的自定义组件和预设组件
     *
     * @param menuList 菜单列表
     * @return 返回
     */
    private List<CustomizedMenuWork> filterPermissionLabelList(List<CustomizedMenuWork> menuList) {
        List<CustomizedMenuWork> permissionLabelList = Lists.newArrayList();
        // 菜单中的自定义组件
        List<CustomizedMenuWork> customLabelList = menuList.stream().filter(x -> Constants.CustomizedMenuTypeEnum.LABEL.getType().equals(x.getMenuType())
                && Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal().equals(x.getMenuLabelType())).collect(Collectors.toList());
        // 菜单中的预设组件
        List<CustomizedMenuWork> preLabelList = menuList.stream().filter(x -> Constants.CustomizedMenuTypeEnum.LABEL.getType().equals(x.getMenuType())
                && Constants.LabelTypeEnum.SYSTEM_PRE.getVal().equals(x.getMenuLabelType())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(customLabelList) && CollectionUtils.isEmpty(preLabelList)) {
            return permissionLabelList;
        }

        // 查询预设组件信息
        Map<Integer, LabelSystemPre> preDataTypeMap = Maps.newHashMap();
        Map<Long, LabelSystemPre> preIdMap = Maps.newHashMap();
        boolean flag = Boolean.TRUE;
        if (CollectionUtils.isNotEmpty(preLabelList)) {
            // 获取预设部件
            LambdaQueryWrapper<LabelSystemPre> wrapper = new LambdaQueryWrapper<>();
            wrapper.eq(LabelSystemPre::getTenantId, Utils.getTenantId());
            wrapper.eq(LabelSystemPre::getClientType, Constants.ClientTypeTypeEnum.PC.getFlag());
            List<LabelSystemPre> systemPres = InterceptorIgnoreUtil.handler(() -> labelSystemPreMapper.selectList(wrapper));
            preDataTypeMap = systemPres.stream()
                    .collect(Collectors.toMap(LabelSystemPre::getDataType, Function.identity(), (a, b) -> a));
            preIdMap = systemPres.stream()
                    .collect(Collectors.toMap(LabelSystemPre::getId, Function.identity(), (a, b) -> a));

            // 待办、常用、收藏、数智视图类型的预设组件默认有权限，不是这三种类型的组件需要判断组件权限是否配置
            flag = systemPres.stream().anyMatch(x -> !Arrays.asList(
                    Constants.DataTypeEnum.TYPE_TODO.getVal(),
                    Constants.DataTypeEnum.TYPE_USE.getVal(),
                    Constants.DataTypeEnum.TYPE_COLLECT.getVal(),
                    Constants.DataTypeEnum.TYPE_INFO.getVal()).contains(x.getDataType()));
        }

        Map<String, LabelSystemAuth> labelSystemAuthMap = Maps.newHashMap();
        // 查询当前用户有权限的组件
        if (CollectionUtils.isNotEmpty(customLabelList) || flag) {
            QueryWrapper<LabelSystemAuth> condition = new QueryWrapper<>();
            List<LabelSystemAuth> bizObjAuthRelList = labelSystemAuthService.getBaseMapper().selectList(labelSystemAuthService.getUserAuthCon(condition));
            Map<String, LabelSystemAuth> customMap = bizObjAuthRelList.stream().collect(Collectors.toMap(value -> value.getLabelId() + "-" + value.getLabelType(), value -> value, (key1, key2) -> key1));
            labelSystemAuthMap.putAll(customMap);
            Map<Long, LabelSystemPre> finalPreIdMap = preIdMap;
            bizObjAuthRelList.stream().filter(x ->
                    Constants.LabelTypeEnum.SYSTEM_PRE.getVal().equals(x.getLabelType())).forEach(x -> {
                if (finalPreIdMap.containsKey(x.getLabelId())) {
                    LabelSystemPre pre = finalPreIdMap.get(x.getLabelId());
                    labelSystemAuthMap.put(pre.getDataType() + "", x);
                }
            });
        }

        // 过滤出有权限的自定义组件
        if (CollectionUtils.isNotEmpty(customLabelList)) {
            // 根据组件id查询数据源
            Set<Long> customLabelIdSet = customLabelList.stream().map(CustomizedMenuWork::getMenuLabelId).collect(Collectors.toSet());
            List<LabelSystemData> systemDataList = labelSystemDataMapper.queryBySystemIdList(customLabelIdSet);

            // 判断【鼎捷雅典娜APP】应用的作业权限
            UserAuthSystemLabelVo authSystemLabelVo = mobileUserAuthService.buildSystemIdApp(systemDataList);
            // 融合权限
            labelSystemAuthService.addAuthLabelMap(labelSystemAuthMap, authSystemLabelVo);

            // 过滤
            Set<String> labelKeySet = labelSystemAuthMap.keySet();
            customLabelList = customLabelList.stream().filter(x -> labelKeySet.contains(x.getMenuLabelId() + "-" + x.getMenuLabelType())).collect(Collectors.toList());
            permissionLabelList.addAll(customLabelList);
        }

        // 过滤出有权限的的预设组件
        if (CollectionUtils.isNotEmpty(preLabelList)) {
            for (CustomizedMenuWork menuWork : preLabelList) {
                LabelSystemPre preLabel = null;
                if (menuWork.getDataType() == null) {
                    // 租户菜单
                    preLabel = preIdMap.get(menuWork.getMenuLabelId());
                } else {
                    // isv菜单或复制的ISV菜单
                    preLabel = preDataTypeMap.get(menuWork.getDataType());
                }
                if (ObjectUtils.isEmpty(preLabel)) {
                    continue;
                }
                // 待办、常用、收藏、数智视图类型的预设组件默认有权限，不是这三种类型的组件则返回有权限的
                if (Arrays.asList(
                        Constants.DataTypeEnum.TYPE_TODO.getVal(),
                        Constants.DataTypeEnum.TYPE_USE.getVal(),
                        Constants.DataTypeEnum.TYPE_COLLECT.getVal(),
                        Constants.DataTypeEnum.TYPE_INFO.getVal()).contains(preLabel.getDataType())
                        || ObjectUtils.isNotEmpty(labelSystemAuthMap.get(menuWork.getMenuLabelId() + "-" + menuWork.getMenuLabelType()))
                        || ObjectUtils.isNotEmpty(labelSystemAuthMap.get(menuWork.getDataType() + ""))) {
                    menuWork.setPreLabelDataType(preLabel.getDataType());
                    permissionLabelList.add(menuWork);
                }
            }
        }
        return permissionLabelList;
    }

    /**
     * 构建用户的系统菜单
     *
     * @param systemCodeList 权限列表
     * @return 返回
     */
    private List<SystemMenuDTO> buildSystemMenu(List<String> systemCodeList) {
        // 查询小AI系统菜单
        List<SystemMenuDTO> systemMenuDTOList = audcService.querySystemMenu();
        if (CollectionUtils.isEmpty(systemMenuDTOList)) {
            return systemMenuDTOList;
        }

        List<SystemMenuDTO> systemRespList = Lists.newArrayList();
        for (SystemMenuDTO systemMenuDTO : systemMenuDTOList) {
            // 文件夹
            if (systemMenuDTO.getIsDirectory()) {
                List<SystemMenuDTO> subMenuList = systemMenuDTO.getSubMenus();
                if (CollectionUtils.isEmpty(subMenuList)) {
                    continue;
                }

                // 展示有权限的系统菜单
                List<SystemMenuDTO> subSystemRespList = Lists.newArrayList();
                for (SystemMenuDTO subMenuDTO : subMenuList) {
                    if (systemCodeList.contains(subMenuDTO.getCode())) {
                        subSystemRespList.add(subMenuDTO);
                    }
                }
                if (CollectionUtils.isNotEmpty(subSystemRespList) && subSystemRespList.size() > 0) {
                    systemMenuDTO.setSubMenus(subSystemRespList);
                    systemRespList.add(systemMenuDTO);
                }

            } else {
                if (systemCodeList.contains(systemMenuDTO.getCode())) {
                    systemRespList.add(systemMenuDTO);
                }
            }
        }
        return systemRespList;
    }

    /**
     * 查询应用作业池列表
     *
     * @return 返回对象
     */
    @Override
    public List<AppJobDTO> queryAppJobList() {
        String locale = LocaleContextHolder.getLocale().toString();
        List<AppJobDTO> appJobDTOS = kmService.queryAppJobListBy();
        //若km的数据没有这些字段则去除这些数据
        appJobDTOS.forEach(item -> {
            List<AppJobDTO.Job> datas = item.getData();
            String s = translateService.translateTextCache(item.getAppName(), StringUtils.EMPTY);
            item.setAppName(s);
            Iterator<AppJobDTO.Job> iterator = datas.iterator();
            while (iterator.hasNext()) {
                AppJobDTO.Job job = iterator.next();
                job.setAppCode(item.getAppCode());
                if (StringUtils.isBlank(job.getName()) || StringUtils.isBlank(job.getCode()) || StringUtils.isBlank(job.getType())) {
                    iterator.remove();
                }
            }
        });
        return appJobDTOS;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean initData() {
        // 1) 获取所有 publish_source=TENANT 的菜单模板，收集模板ID与其租户ID
        List<CustomizedMenuTemplate> tenantTemplates = InterceptorIgnoreUtil.handler(() ->
                customizedMenuTemplateMapper.selectList(new QueryWrapper<CustomizedMenuTemplate>()
                        .eq("publish_source", PublishSourceEnum.TENANT.name())
                        .select("id", "tenant_id"))
        );
        if (CollectionUtils.isEmpty(tenantTemplates)) {
            log.info("initData: no TENANT templates found, skip.");
            return true;
        }

        // 模板ID -> 租户ID
        Map<Long, String> templateTenantMap = tenantTemplates.stream()
                .filter(t -> t.getId() != null && t.getTenantId() != null)
                .collect(Collectors.toMap(CustomizedMenuTemplate::getId, CustomizedMenuTemplate::getTenantId, (a, b) -> a));
        if (templateTenantMap.isEmpty()) {
            log.info("initData: no template-tenant mapping found, skip.");
            return true;
        }

        // 2) 根据 tenantIds 查询每个租户的 ‘待办(data_type=0)’ 预设组件ID
        Map<String, Long> tenantTodoPreIdMap = InterceptorIgnoreUtil.handler(() ->
                        labelSystemPreMapper.selectList(new QueryWrapper<LabelSystemPre>()
                                .in("tenant_id", templateTenantMap.values())
                                .eq("data_type", Constants.DataTypeEnum.TYPE_TODO.getVal())
                                .eq("client_type", Constants.ClientTypeTypeEnum.PC.getFlag())
                                .select("id", "tenant_id", "data_type", "client_type"))
                ).stream()
                .filter(p -> p.getId() != null && p.getTenantId() != null)
                .collect(Collectors.toMap(LabelSystemPre::getTenantId, LabelSystemPre::getId, (a, b) -> a));

        if (tenantTodoPreIdMap.isEmpty()) {
            log.info("initData: no LabelSystemPre(data_type=TODO) found for target tenants, skip.");
            return true;
        }

        // 3) 查询 t_customized_menu 所有已存在的主键ID（用于生成 currentId） ，使用 HashSet 以便 O(1) 判断是否存在
        Set<Long> existIds = InterceptorIgnoreUtil.handler(() ->
                        customizedMenuWorkMapper.selectList(
                                new QueryWrapper<CustomizedMenuWork>().select("id"))
                ).stream()
                .map(CustomizedMenuWork::getId)
                .filter(Objects::nonNull)
                .collect(Collectors.toCollection(HashSet::new));

        // 4) 分批处理模板ID（每200条一组）
        List<Long> allTemplateIds = new ArrayList<>(templateTenantMap.keySet());
        final int batchSize = 200;
        for (int from = 0; from < allTemplateIds.size(); from += batchSize) {
            int to = Math.min(from + batchSize, allTemplateIds.size());
            List<Long> batchTemplateIds = allTemplateIds.subList(from, to);

            // 5) 批量查询 t_customized_menu 数据，按templateId分组
            Map<Long, List<CustomizedMenuWork>> menuByTemplateId = InterceptorIgnoreUtil.handler(() ->
                            customizedMenuWorkMapper.selectList(new QueryWrapper<CustomizedMenuWork>()
                                    .in("template_id", batchTemplateIds))
                    ).stream()
                    .collect(Collectors.groupingBy(CustomizedMenuWork::getTemplateId));

            // 当前候选ID（全局递增，不每个模板重置，以减少重复扫描开销）
            long currentIdCandidate = 1L;

            // 6) 逐模板处理
            for (Long templateId : batchTemplateIds) {
                String tenantId = templateTenantMap.get(templateId);
                Long todoPreId = tenantTodoPreIdMap.get(tenantId);
                if (todoPreId == null) {
                    // 当前租户无“待办”预设组件，跳过
                    continue;
                }

                List<CustomizedMenuWork> oneTemplateMenus = menuByTemplateId.getOrDefault(templateId, Collections.emptyList());

                // 是否已存在“待办”组件（优先用 data_type 判断；若无则用 menu_label_id 对应预设组件ID 判断）
                boolean existTodo = oneTemplateMenus.stream().anyMatch(x ->
                        Constants.LabelTypeEnum.SYSTEM_PRE.getVal().equals(x.getMenuLabelType()) && (
                                (x.getDataType() != null && Objects.equals(x.getDataType(), Constants.DataTypeEnum.TYPE_TODO.getVal()))
                                        || Objects.equals(x.getMenuLabelId(), todoPreId)
                        )
                );
                if (existTodo) {
                    // 已存在“待办”组件，跳过
                    continue;
                }

                // 生成一个全局未占用的 currentId（从1开始递增，跨模板累进）
                while (existIds.contains(currentIdCandidate)) {
                    currentIdCandidate++;
                }
                long currentId = currentIdCandidate;

                // 待插入的“待办”组件
                CustomizedMenuWork todoMenu = new CustomizedMenuWork();
                todoMenu.setId(currentId);
                todoMenu.setTemplateId(templateId);
                todoMenu.setMenuIcon("23");
                todoMenu.setMenuName("待办");
                todoMenu.setOriginalName("待办");
                todoMenu.setMenuNameTw("待辦");
                todoMenu.setMenuType(Constants.CustomizedMenuTypeEnum.LABEL.getType());
                todoMenu.setLabelTypeSource(Constants.LabelTypeSourceEnum.TENANT.getFlag());
                todoMenu.setMenuLabelId(todoPreId);
                todoMenu.setMenuLabelType(Constants.LabelTypeEnum.SYSTEM_PRE.getVal());
                // 作为一级，无父级
                todoMenu.setLevel(Constants.CustomizedMenuLevelEnum.FIRST.getLevel());
                todoMenu.setMenuOrder(0);
                todoMenu.setTenantId(tenantId);
                // isv菜单或复制的ISV菜单 才有值
                todoMenu.setDataType(null);

                // 计算该模板下最小的菜单ID
                long minIdOfTemplate = oneTemplateMenus.stream()
                        .map(CustomizedMenuWork::getId)
                        .filter(Objects::nonNull)
                        .min(Long::compareTo)
                        .orElse(Long.MAX_VALUE);
                // 7.1 / 7.2 分支逻辑
                if (currentId < minIdOfTemplate) {
                    // 7.1 直接新增
                    InterceptorIgnoreUtil.handler(() -> this.save(todoMenu));
                    // 当前ID存入
                    existIds.add(currentId);
                    currentIdCandidate++;
                } else {
                    // 7.2 新增后，移除原有该模板菜单并重新插入（按原数据恢复）
                    todoMenu.setId(null);
                    InterceptorIgnoreUtil.handler(() -> this.save(todoMenu));

                    // 删除原有的（不包含刚新增的）
                    List<Long> oldIds = oneTemplateMenus.stream()
                            .map(CustomizedMenuWork::getId)
                            .filter(Objects::nonNull)
                            .collect(Collectors.toList());
                    if (!oldIds.isEmpty()) {
                        InterceptorIgnoreUtil.handler(() -> this.remove(new QueryWrapper<CustomizedMenuWork>()
                                .eq("template_id", templateId)
                                .in("id", oldIds)));
                        // 重新插入原数据
                        oneTemplateMenus.forEach(x -> x.setId(null));
                        InterceptorIgnoreUtil.handler(() -> this.saveBatch(oneTemplateMenus));
                    }
                    // 当前ID存入
                    oldIds.forEach(existIds::remove);
                    existIds.add(todoMenu.getId());
                    existIds.addAll(oneTemplateMenus.stream().map(CustomizedMenuWork::getId).filter(Objects::nonNull).collect(Collectors.toList()));
                }
            }
        }

        log.info("initData: completed initialization for TENANT templates.");
        return true;
    }

}