package com.digiwin.athena.semc.service.menu.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.ObjectUtils;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.menu.manage.SaveMenuAuthReq;
import com.digiwin.athena.semc.entity.menu.manage.ManageMenuAuth;
import com.digiwin.athena.semc.mapper.menu.manage.ManageMenuAuthMapper;
import com.digiwin.athena.semc.proxywrapper.IamUserService;
import com.digiwin.athena.semc.proxywrapper.dto.IamUserAuthInfo;
import com.digiwin.athena.semc.service.menu.ManageMenuAuthService;
import com.digiwin.athena.semc.util.Utils;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 管理后台菜单权限配置 服务接口实现类
 *
 * @author sunqgz
 * @since 2024-08-26
 */
@Slf4j
@Service
public class ManageMenuAuthServiceImpl extends ServiceImpl<ManageMenuAuthMapper, ManageMenuAuth> implements ManageMenuAuthService {

    @Resource
    private ManageMenuAuthMapper manageMenuAuthMapper;

    @Resource
    private IamUserService iamUserService;

    /**
     * 管理后台-查询权限对象关联的菜单
     *
     * @param authId 权限id
     * @return 返回
     */
    @Override
    public List<ManageMenuAuth> queryAuthRelMenu(String authId) {
        QueryWrapper<ManageMenuAuth> condition = new QueryWrapper<ManageMenuAuth>().eq("auth_id", authId);
        return manageMenuAuthMapper.selectList(condition);
    }

    /**
     * 管理后台-保存权限对象关联的菜单
     *
     * @param saveMenuAuthReq 请求入参
     */
    @Override
    @Transactional
    public void saveAuthRelMenu(SaveMenuAuthReq saveMenuAuthReq) {
        // 删除当前权限关系
        QueryWrapper<ManageMenuAuth> condition = new QueryWrapper<>();
        condition.eq("auth_id", saveMenuAuthReq.getAuthId());
        manageMenuAuthMapper.delete(condition);

        if (Constants.ALL_AUTH_FLAG_YES.equals(saveMenuAuthReq.getAllMenuAuthFlag())) {
            ManageMenuAuth manageMenuAuth = new ManageMenuAuth();
            manageMenuAuth.setAuthId(saveMenuAuthReq.getAuthId());
            manageMenuAuth.setAuthName(saveMenuAuthReq.getAuthName());
            manageMenuAuth.setAuthType(saveMenuAuthReq.getAuthType());
            manageMenuAuth.setAllMenuAuthFlag(Constants.ALL_AUTH_FLAG_YES);
            this.save(manageMenuAuth);
        } else {
            if (CollectionUtils.isEmpty(saveMenuAuthReq.getMenuKeyList())) {
                return;
            }
            // 新增权限关系
            List<ManageMenuAuth> menuAuthList = saveMenuAuthReq.getMenuKeyList().stream().map(x -> {
                ManageMenuAuth manageMenuAuth = new ManageMenuAuth();
                manageMenuAuth.setMenuKey(x);
                manageMenuAuth.setAuthId(saveMenuAuthReq.getAuthId());
                manageMenuAuth.setAuthName(saveMenuAuthReq.getAuthName());
                manageMenuAuth.setAuthType(saveMenuAuthReq.getAuthType());
                return manageMenuAuth;
            }).collect(Collectors.toList());
            // 500个一组分批存储，防止一次数据过大
            List<List<ManageMenuAuth>> partList = Lists.partition(menuAuthList, 500);
            for (List<ManageMenuAuth> tempList : partList) {
                this.saveBatch(tempList);
            }
        }
    }

    /**
     * 查询当前登录用户关联的菜单
     *
     * @return 返回
     */
    @Override
    public List<ManageMenuAuth> queryUserMenuList() {
        IamUserAuthInfo iamUserAuthInfo = iamUserService.getUserAuthSids(Utils.getUserId(), Utils.getTenantId(), Utils.getUserToken());
        List<Long> authIdList = Lists.newArrayList();
        if (iamUserAuthInfo != null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getOrgSids())) {
            authIdList.addAll(iamUserAuthInfo.getOrgSids());
        }
        // 拼接角色条件
        if (iamUserAuthInfo != null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getRoleSids())) {
            authIdList.addAll(iamUserAuthInfo.getRoleSids());
        }
        // 拼接部门条件
        if (iamUserAuthInfo != null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getDeptSids())) {
            authIdList.addAll(iamUserAuthInfo.getDeptSids());
        }
        // 拼接用户条件
        Long userSid = AppAuthContextHolder.getContext().getAuthoredUser().getSid();
        if (ObjectUtils.isNotEmpty(userSid)) {
            authIdList.add(userSid);
        }
        List<ManageMenuAuth> menuAuthList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(authIdList)) {
            Collections.sort(authIdList);
            // 拼接组织条件
            StringBuilder authCondition = new StringBuilder("(auth_id in(" + Joiner.on(",").join(authIdList) + "))");
            QueryWrapper<ManageMenuAuth> condition = new QueryWrapper<>();
            condition.apply("(" + authCondition + ")");
            condition.select("menu_key", "all_menu_auth_flag");
            menuAuthList = manageMenuAuthMapper.selectList(condition);
        }
        return menuAuthList;
    }
}