package com.digiwin.athena.semc.service.menu.impl;

import com.digiwin.athena.semc.dto.config.TenantAttributeDto;
import com.digiwin.athena.semc.service.tenant.TenantAppService;
import com.digiwin.athena.semc.service.tenant.TenantAttributeService;
import com.digiwin.athena.semc.util.MenuUtils;
import com.digiwin.athena.semc.vo.menu.UserManageMenulVO;
import com.digiwin.athena.semc.vo.tenant.TenantSalesPlanVO;
import com.google.common.collect.Lists;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.menu.manage.SaveSpecialTenantIdReq;
import com.digiwin.athena.semc.entity.menu.manage.ManageMenu;
import com.digiwin.athena.semc.entity.menu.manage.ManageMenuAuth;
import com.digiwin.athena.semc.mapper.menu.manage.ManageMenuAuthMapper;
import com.digiwin.athena.semc.mapper.menu.manage.ManageMenuMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.RoleDTO;
import com.digiwin.athena.semc.service.common.EnvService;
import com.digiwin.athena.semc.proxy.iam.service.model.TenantSimpleDTO;
import com.digiwin.athena.semc.service.menu.ManageMenuAuthService;
import com.digiwin.athena.semc.service.menu.ManageMenuService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.Utils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * 管理后台基础菜单配置表(TManageMenu)表服务实现类
 *
 * @author sungqz
 * @since 2024-09-30
 */
@Slf4j
@Service
public class ManageMenuServiceImpl extends ServiceImpl<ManageMenuMapper, ManageMenu> implements ManageMenuService {

    @Resource
    private ManageMenuMapper manageMenuMapper;

    @Resource
    private ManageMenuAuthMapper manageMenuAuthMapper;

    @Resource
    private IamService iamService;

    @Resource
    private ManageMenuAuthService manageMenuAuthService;

    @Resource
    private EnvService envService;

    @Resource
    private TenantAppService tenantAppService;

    @Resource
    private TenantAttributeService tenantAttributeService;

    /**
     * 菜单访问区域，0全部区域可访问，1只有预区可访问
     */
    public static final int ONLY_PRE_AREA_YES = 1;

    /**
     * 菜单访问区域，0全部区域可访问，1只有预区可访问
     */
    public static final int ONLY_PRE_AREA_NO = 0;

    /**
     * 查询基础菜单配置列表数据
     *
     * @return 返回
     */
    @Override
    public List<ManageMenu> queryManageMenuConfigList() {
        List<ManageMenu> manageMenuListResp = Lists.newArrayList();
        List<ManageMenu> manageMenuList = manageMenuMapper.selectList(Wrappers.emptyWrapper());
        if (CollectionUtils.isEmpty(manageMenuList)) {
            return manageMenuListResp;
        }
        manageMenuList = manageMenuList.stream().sorted(Comparator.comparing(ManageMenu::getMenuSort, Comparator.nullsLast(Comparator.naturalOrder()))).collect(Collectors.toList());
        // 转换为树结构
        for (ManageMenu menu : manageMenuList) {
            // 找到根节点
            if (ObjectUtils.isEmpty(menu.getParentMenuKey())) {
                manageMenuListResp.add(findChildById(menu, manageMenuList));
            }
        }
        return manageMenuListResp;
    }

    /**
     * 根据menuKey构建菜单树
     *
     * @param menu           菜单对象
     * @param manageMenuList 菜单列表
     * @return 返回对象
     */
    private static ManageMenu findChildById(ManageMenu menu, List<ManageMenu> manageMenuList) {
        // 子节点集合
        List<ManageMenu> childList = Lists.newArrayList();
        for (ManageMenu node : manageMenuList) {
            if (Objects.equals(node.getParentMenuKey(), menu.getMenuKey())) {
                // 递归调用
                childList.add(findChildById(node, manageMenuList));
            }
        }
        menu.setChildrenList(childList);
        return menu;
    }

    /**
     * 校验menuKey是否存在
     *
     * @param manageMenu 菜单
     * @return 返回
     */
    @Override
    public List<ManageMenu> existMenuKey(ManageMenu manageMenu) {
        QueryWrapper<ManageMenu> queryWrapper = new QueryWrapper<>();
        // 修改时排除当前配置
        if (ObjectUtils.isNotEmpty(manageMenu.getId())) {
            queryWrapper.ne("id", manageMenu.getId());
        }
        queryWrapper.eq("menu_key", manageMenu.getMenuKey());
        return manageMenuMapper.selectList(queryWrapper);
    }

    /**
     * 新增数据
     *
     * @param manageMenu 实例对象
     * @return 实例对象
     */
    @Override
    public ManageMenu insertManageMenu(ManageMenu manageMenu) {
        String nowTime = DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER);
        manageMenu.setCreateTime(nowTime);
        manageMenu.setCreateUserId(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
        manageMenu.setModifyTime(nowTime);
        manageMenu.setModifyUserId(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
        this.manageMenuMapper.insert(manageMenu);
        return manageMenu;
    }

    /**
     * 修改数据
     *
     * @param manageMenu 实例对象
     * @return 实例对象
     */
    @Override
    public ManageMenu updateManageMenu(ManageMenu manageMenu) {
        manageMenu.setModifyTime(DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER));
        manageMenu.setModifyUserId(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
        this.manageMenuMapper.update(manageMenu);
        return manageMenu;
    }

    /**
     * 通过菜单标识删除菜单
     *
     * @param menuKeyList    菜单标识
     * @param manageMenuList 菜单列表
     */
    @Override
    @Transactional
    public void deleteByMenuKeyList(List<String> menuKeyList, List<ManageMenu> manageMenuList) {
        // 遍历出当前菜单的所有子菜单
        List<String> allKeyList = Lists.newArrayList();
        for (String menuKey : menuKeyList) {
            allKeyList.add(menuKey);
            List<String> childrenIds = getAllChildrenIds(menuKey, manageMenuList);
            allKeyList.addAll(childrenIds);
        }
        // 删除菜单列表
        manageMenuMapper.deleteByMenuKeyList(allKeyList);
        // 删除菜单权限列表
        manageMenuAuthMapper.deleteByMenuKeyList(allKeyList);
    }

    /**
     * 获取所有子菜单key
     *
     * @param parentMenuKey  父菜单menuKey
     * @param manageMenuList 菜单列表
     * @return 返回
     */
    private static List<String> getAllChildrenIds(String parentMenuKey, List<ManageMenu> manageMenuList) {
        List<String> childrenIds = Lists.newArrayList();
        for (ManageMenu node : manageMenuList) {
            if (ObjectUtils.isNotEmpty(node.getParentMenuKey()) && node.getParentMenuKey().equals(parentMenuKey)) {
                childrenIds.add(node.getMenuKey());
                childrenIds.addAll(getAllChildrenIds(node.getMenuKey(), manageMenuList));
            }
        }
        return childrenIds;
    }

    /**
     * 查询权限设置页面的菜单列表数据
     *
     * @param environmentFlag 所属环境标识 0:全部 1:云上 2:下地
     * @return 返回
     */
    @Override
    public List<ManageMenu> queryManageMenuList(Integer environmentFlag) {
        List<ManageMenu> manageMenuListResp = Lists.newArrayList();
        // 获取当前租户的版本号
        String version = getTenantVersion();
        List<ManageMenu> manageMenuList = queryBaseManageMenu(environmentFlag, true, version);
        if (CollectionUtils.isEmpty(manageMenuList)) {
            return manageMenuListResp;
        }

        // 过滤掉属于特殊租户的菜单，即空租户的菜单为通用配置菜单
        manageMenuList = manageMenuList.stream().filter(x -> StringUtils.isBlank(x.getTenantId())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(manageMenuList)) {
            return manageMenuListResp;
        }
        manageMenuList = manageMenuList.stream().sorted(Comparator.comparing(ManageMenu::getMenuSort, Comparator.nullsLast(Comparator.naturalOrder()))).collect(Collectors.toList());
        // 根据id转换为树结构
        for (ManageMenu menu : manageMenuList) {
            // 找到根节点
            if (ObjectUtils.isEmpty(menu.getParentMenuKey())) {
                manageMenuListResp.add(findChildById(menu, manageMenuList));
            }
        }
        //过滤子集为空的父级菜单
        manageMenuListResp= filterParentMenu(manageMenuListResp);
        //娜娜销售包重置菜单
        manageMenuListResp= resetNanaMenu(manageMenuListResp);
        return manageMenuListResp;
    }


    /**
     * 过滤子集为空的父级菜单
     * @param menuList
     * @return
     */
    public List<ManageMenu>  filterParentMenu(List<ManageMenu> menuList) {
        List<ManageMenu> menuNewList = new ArrayList<>();
        for(ManageMenu menu:menuList){
            //是否需要过滤，为true 表示子集下没有末级菜单
            boolean isFilter=isNeedFilter(menu);
            //菜单有末级菜单
            if(!isFilter){
                menuNewList.add(menu);
            }
        }
        return menuNewList;
    }

    /**
     * 判断是否是需要过滤
     * @param menu
     * @return
     */
    public boolean isNeedFilter(ManageMenu menu){
        //是链接  或者 有路由，说明是末级,不需要过滤上级
        if(menu.getMenuType().equals(Constants.ManageMenuTypeEnum.LINK.getType()) ||
                StringUtils.isNotEmpty(menu.getRoutePath())){
            return  false;
        }
         //属于菜单，且路由为空，且子集为空，需要过滤上级
         if(menu.getMenuType().equals(Constants.ManageMenuTypeEnum.MENU.getType())
           && StringUtils.isEmpty(menu.getRoutePath())
                 && CollectionUtils.isEmpty(menu.getChildrenList())){
              return  true;
         }
        List<ManageMenu> childrenNewList=new ArrayList<>();
         //看子集是否都是需要过滤
        boolean isNeedFilteChild=true;
        for(ManageMenu childrenMenu:menu.getChildrenList()){
            boolean isNeedChild=isNeedFilter(childrenMenu);
            //子集都需要过滤，则不返回子集
            if(!isNeedChild){
                childrenNewList.add(childrenMenu);
            }
            //如果为true
            if(isNeedFilteChild){
                isNeedFilteChild=isNeedChild;
            }
        }
         menu.setChildrenList(childrenNewList);
         return isNeedFilteChild;
    }


    /**
     * 根据条件查询基础菜单列表
     *
     * @param environmentFlag 所属环境标识 0:全部 1:云上 2:下地
     * @param filterPre       是否要走仅预区可见菜单逻辑
     * @param version         菜单版本标识
     * @return 返回
     */
    @Override
    public List<ManageMenu> queryBaseManageMenu(Integer environmentFlag, boolean filterPre, String version) {
        QueryWrapper<ManageMenu> condition = new QueryWrapper<>();
        // 查询全部类型 + 所属环境类型的菜单
        condition.and(x -> x.eq("belonging_environment_flag", Constants.MenuEnvironmentFlag.ALL.getFlag())
                .or().eq("belonging_environment_flag", environmentFlag));
        // 查询版本为空 + 所属版本的菜单
        if (StringUtils.isNotEmpty(version)) {
            condition.and(x -> x.isNull("version").or().eq("version", version));
        }
        // 查询出可见的菜单 0:否 1:是
        condition.eq("visible_flag", Constants.VALID_STATUS_ENABLE);
        List<ManageMenu> list = manageMenuMapper.selectList(condition);

        // 下地客户不涉及预区，直接返回
        if (Constants.MenuEnvironmentFlag.CUSTOM.getFlag().equals(environmentFlag)) {
            return list;
        }
        if (!filterPre) {
            return list;
        }
        boolean anyMatch = list.stream().anyMatch(m -> Integer.valueOf(ONLY_PRE_AREA_YES).equals(m.getOnlyPreArea()));
        if (!anyMatch) {
            return list;
        }
        // 获取当前环境是否是预区
        boolean preFlag = envService.tenantIsPre();
        return list.stream().filter(m -> {
            if (Integer.valueOf(ONLY_PRE_AREA_NO).equals(m.getOnlyPreArea())) {
                return true;
            }
            return preFlag;
        }).collect(Collectors.toList());
    }

    /**
     * 管理后台-查询用户的左侧菜单树
     *
     * @param environmentFlag 所属环境标识 0:全部 1:云上 2:下地
     * @return 查询结果
     */
    @Override
    public List<ManageMenu> queryManageMenuTree(Integer environmentFlag) {
        List<ManageMenu> manageMenuListResp = Lists.newArrayList();
        // 获取当前租户的版本号
        String version = getTenantVersion();
        // 查询菜单数据
        List<ManageMenu> manageMenuList = queryFinalManageMenu(environmentFlag, version);
        if (CollectionUtils.isEmpty(manageMenuList)) {
            return manageMenuListResp;
        }
        manageMenuList = manageMenuList.stream().sorted(Comparator.comparing(ManageMenu::getMenuSort, Comparator.nullsLast(Comparator.naturalOrder()))).collect(Collectors.toList());
        // 根据id转换为树结构
        for (ManageMenu menu : manageMenuList) {
            // 找到根节点
            if (ObjectUtils.isEmpty(menu.getParentMenuKey())) {
                manageMenuListResp.add(findChildById(menu, manageMenuList));
            }
        }
        //过滤子集为空的父级菜单
        manageMenuListResp= filterParentMenu(manageMenuListResp);
        //娜娜销售包重置菜单
        manageMenuListResp= resetNanaMenu(manageMenuListResp);
        return manageMenuListResp;
    }

    /**
     * 查询当前用户最终菜单列表
     *
     * @param environmentFlag 所属环境标识 0:全部 1:云上 2:下地
     * @return 返回
     */
    private List<ManageMenu> queryFinalManageMenu(Integer environmentFlag, String version) {
        // 查询基础菜单数据
        List<ManageMenu> manageMenuList = queryBaseManageMenu(environmentFlag, true, version);
        if (CollectionUtils.isEmpty(manageMenuList)) {
            return Lists.newArrayList();
        }

        // 默认菜单
        List<ManageMenu> commonMenuList = manageMenuList.stream().filter(x -> StringUtils.isBlank(x.getTenantId())).collect(Collectors.toList());
        // 特殊账号的菜单，即ISV菜单
        List<ManageMenu> specialMenuList = manageMenuList.stream().filter(x -> StringUtils.isNotBlank(x.getTenantId())
                && x.getTenantId().equals(Utils.getTenantId()) && x.getBelongingUsers().contains(Utils.getUserId())).collect(Collectors.toList());

        // 判断当前用户是否是超管
        List<RoleDTO> roleDTOList = iamService.queryUserRoles(Utils.getUserId(), Utils.getUserToken());
        boolean flag = roleDTOList.stream().anyMatch(x -> x.getId().equals(Constants.SUPER_ADMIN));
        // 如果是超管，则返回默认的菜单 + 特殊账号的菜单
        if (flag) {
            commonMenuList.addAll(specialMenuList);
            return commonMenuList;
        }
        // 不是超管，则查询当前用户配置的菜单权限
        List<ManageMenuAuth> menuAuthList = manageMenuAuthService.queryUserMenuList();
        if (CollectionUtils.isEmpty(menuAuthList)) {
            return specialMenuList;
        }

        // 有全部菜单权限，则返回默认的菜单 + 特殊账号的菜单
        boolean allFlag = menuAuthList.stream().anyMatch(x -> Constants.ALL_AUTH_FLAG_YES.equals(x.getAllMenuAuthFlag()));
        if (allFlag) {
            commonMenuList.addAll(specialMenuList);
            return commonMenuList;
        }
        // 有部分菜单权限，则过滤出有权限的菜单
        List<String> menuKeyList = menuAuthList.stream().map(ManageMenuAuth::getMenuKey).collect(Collectors.toList());
        List<String> specialMenuKeyList = specialMenuList.stream().map(ManageMenu::getMenuKey).collect(Collectors.toList());
        menuKeyList.addAll(specialMenuKeyList);

        List<ManageMenu> menuList = manageMenuList.stream().filter(x -> menuKeyList.contains(x.getMenuKey())).collect(Collectors.toList());
        // 如果只维护了子菜单，则需要获取子菜单的所有父菜单
        List<ManageMenu> allMenuList = Lists.newArrayList();
        Map<String, ManageMenu> menuKeyMap = manageMenuList.stream().collect(Collectors.toMap(ManageMenu::getMenuKey, Function.identity(), (a, b) -> a));
        for (ManageMenu menu : menuList) {
            allMenuList.add(menu);
            List<ManageMenu> parentList = queryAllParentList(menu, menuKeyMap);
            allMenuList.addAll(parentList);
        }
        // 菜单去重
        allMenuList = allMenuList.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(ManageMenu::getMenuKey))), ArrayList::new));
        return allMenuList;
    }

    /**
     * 获取当前菜单的所有父菜单
     *
     * @param manageMenu 当前菜单
     * @param menuKeyMap 菜单信息
     * @return 返回
     */
    private List<ManageMenu> queryAllParentList(ManageMenu manageMenu, Map<String, ManageMenu> menuKeyMap) {
        List<ManageMenu> parentList = Lists.newArrayList();

        // 如果当前菜单的父菜单不为空，则添加
        String parentMenuKey = manageMenu.getParentMenuKey();
        if (StringUtils.isNotBlank(parentMenuKey)) {
            ManageMenu parentMenu = menuKeyMap.get(parentMenuKey);
            parentList.add(parentMenu);
            parentList.addAll(queryAllParentList(parentMenu, menuKeyMap));
        }
        return parentList;
    }

    /**
     * 获取当前租户的版本号
     *
     * @return
     */
    @Override
    public String getTenantVersion() {
        String version = "";
        TenantSimpleDTO simpleDTO = iamService.queryTenantSimple();
        if (null == simpleDTO) {
            return version;
        }
        version = StringUtils.isEmpty(simpleDTO.getVersion()) ? Constants.IAM_TENANT_DEF_V : simpleDTO.getVersion();
        return version;
    }


    /**
     * 校验是否有权限打开菜单
     *
     * @param environmentFlag 所属环境标识 0:全部 1:云上 2:下地
     * @param manageMenu      菜单
     * @return 返回
     */
    @Override
    public boolean validMenuAuth(Integer environmentFlag, ManageMenu manageMenu) {
        //获取当前租户的版本号
        String version = getTenantVersion();
        boolean flag = Boolean.FALSE;
        // 查询当前用户具有权限的菜单列表
        List<ManageMenu> manageMenuList = queryFinalManageMenu(environmentFlag, version);
        if (CollectionUtils.isEmpty(manageMenuList)) {
            return flag;
        }
        // 当前菜单配置的版本号和当前租户的版本号不一致时，则无权限
        if (StringUtils.isNotEmpty(manageMenu.getVersion()) && !version.equals(manageMenu.getVersion())) {
            return flag;
        }

        Integer secondPageFlag = manageMenu.getSecondPageFlag();
        String menuKey = manageMenu.getMenuKey();
        // 当前menuKey是普通菜单
        if (Constants.SecondPageFlagEnum.NO.getFlag().equals(secondPageFlag)) {
            flag = manageMenuList.stream().anyMatch(x -> x.getMenuKey().equals(menuKey));
        } else { // 当前menuKey是特殊页面
            String parentMenuKey = manageMenu.getParentMenuKey();
            flag = manageMenuList.stream().anyMatch(x -> x.getMenuKey().equals(parentMenuKey));
        }
        return flag;
    }

    /**
     * 设置特殊租户和账号
     *
     * @param saveSpecialTenantIdReq 请求入参
     */
    public void saveTenantId(SaveSpecialTenantIdReq saveSpecialTenantIdReq) {
        String str = String.join(",", saveSpecialTenantIdReq.getUserIdList());
        manageMenuMapper.updateTenantIdByIdList(saveSpecialTenantIdReq.getIdList(), saveSpecialTenantIdReq.getTenantId(), str);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean handMenuVersion(ManageMenu manageMenu) {
        //如果有上级 或者是末级菜单，需要将所有上级版本号置空
        handParentMenuVersion(manageMenu);
        //如果有子集，需要把子集中为空的版本号初始化为 当前的父级版本号，且 把当前父级以及 所有上级版本号置空
        List<ManageMenu> childMenuList = selectManageMenu("", manageMenu.getMenuKey());
        //如果是设置的菜单不是末级，则需要将所有子集且版本号为空的初始化为当前菜单的版本号
        if(CollectionUtils.isNotEmpty(childMenuList)){
            handChildMenuVersion(childMenuList,manageMenu.getVersion());
            //则需要置空当前菜单版本号
            return true;
        }
        return false;
    }

    /**
     * 处理菜单所有的子集菜单中为空的版本号
     * @param childMenuList 子集菜单
     */
    public void handChildMenuVersion( List<ManageMenu> childMenuList,String version) {
        if(CollectionUtils.isEmpty(childMenuList)){
            return;
        }
        //如果有子集，需要将所有子集且版本号为空的初始化为当前菜单的版本号
        for (ManageMenu childMenu : childMenuList) {
            //子集菜单有版本号，保持当前的版本号
            if (StringUtils.isNotEmpty(childMenu.getVersion())) {
                continue;
            }
            //将上级版本号赋值给子集
            childMenu.setVersion(version);
            //如果有子集，说明非末级，给空版本号
            List<ManageMenu> childList = selectManageMenu("", childMenu.getMenuKey());
            if(CollectionUtils.isNotEmpty(childList)){
                childMenu.setVersion(null);
            }
            manageMenuMapper.updateById(childMenu);
            handChildMenuVersion(childMenu.getChildrenList(), version);
        }
    }

    /**
     * 处理菜单所有的上级菜单版本号
     * @param manageMenu
     */
    public void handParentMenuVersion(ManageMenu manageMenu) {
        //如果有上级，需要将所有上级版本号置空
        if(StringUtils.isNotEmpty(manageMenu.getParentMenuKey())){
            List<ManageMenu>  parentMenuList=selectManageMenu(manageMenu.getParentMenuKey(),"");
            for(ManageMenu parentMenu:parentMenuList){
                //将上级版本号置空
                parentMenu.setVersion(null);
                manageMenuMapper.updateById(parentMenu);
                handParentMenuVersion(parentMenu);
            }
        }
    }
    /**
     * 根据菜单key 查询
     * @param menuKey
     * @return
     */
    public List<ManageMenu>  selectManageMenu(String menuKey,String parentMenuKey){
        QueryWrapper<ManageMenu> queryWrapper = new QueryWrapper<>();
        if (StringUtils.isNotEmpty(parentMenuKey)) {
            queryWrapper.eq("parent_menu_key", parentMenuKey);
        }
        if (StringUtils.isNotEmpty(menuKey)) {
            queryWrapper.eq("menu_key", menuKey);
        }
        return manageMenuMapper.selectList(queryWrapper);
    }



    @Override
    public UserManageMenulVO queryUserMenuFlag(UserManageMenulVO req) {
        UserManageMenulVO manageMenulVO=new UserManageMenulVO();
        manageMenulVO.setIsHaveMenu(Boolean.FALSE);
        // 获取当前租户的版本号
        String version = getTenantVersion();
        // 查询菜单数据
        List<ManageMenu> manageMenuList = queryFinalManageMenu(req.getEnvironmentFlag(), version);
        if(CollectionUtils.isEmpty(manageMenuList)){
            return  manageMenulVO;
        }
        //查询指定菜单是否有权限
        if(StringUtils.isNotEmpty(req.getMenuKey())){
            List<ManageMenu> menuKeyList=manageMenuList.stream().filter(x -> req.getMenuKey().equals(x.getMenuKey())).collect(Collectors.toList());
            if(CollectionUtils.isNotEmpty(menuKeyList)){
                manageMenulVO.setIsHaveMenu(Boolean.TRUE);
            }
            return  manageMenulVO;
        }

        for(ManageMenu manageMenu:manageMenuList){
            //菜单是链接  或者 有路由，说明是末级，应该具备后台管理菜单
            if(manageMenu.getMenuType().equals(Constants.ManageMenuTypeEnum.LINK.getType()) ||
                    StringUtils.isNotEmpty(manageMenu.getRoutePath())){
                manageMenulVO.setIsHaveMenu(Boolean.TRUE);
                return  manageMenulVO;
            }
        }
        return manageMenulVO;
    }

    private List<ManageMenu>  resetNanaMenu(List<ManageMenu> menuTreeList) {
        try{
            TenantSalesPlanVO tenantSalesPlan = tenantAppService.queryTenantSalesPlan(Utils.getTenantId());
            if(Objects.nonNull(tenantSalesPlan)){
                //娜娜销售包模式 菜单重置过滤
                if("ASA".equals(tenantSalesPlan.getSalesPlan())){
                    TenantAttributeDto<Object> nanaSalesPackageMenu = tenantAttributeService.queryTenantAttributeByCode("nanaSalesPackageMenu");
                    if(Objects.nonNull(nanaSalesPackageMenu)
                            && Objects.nonNull(nanaSalesPackageMenu.getAttributeValue())
                            && nanaSalesPackageMenu.getAttributeValue() instanceof String){
                        String menuStr = (String)nanaSalesPackageMenu.getAttributeValue();
                        return StringUtils.isBlank(menuStr)?menuTreeList:MenuUtils.resetNanaMenu(menuTreeList,menuStr);
                    }
                }else if("intelligententry".equals(tenantSalesPlan.getSalesPlan())){
                    //完整娜娜销售包菜单过滤
                    TenantAttributeDto<Object> nanaSalesPackageMenu = tenantAttributeService.queryTenantAttributeByCode("nanaFullSalesPackageMenu");
                    if(Objects.nonNull(nanaSalesPackageMenu)
                            && Objects.nonNull(nanaSalesPackageMenu.getAttributeValue())
                            && nanaSalesPackageMenu.getAttributeValue() instanceof String){
                        String menuStr = (String)nanaSalesPackageMenu.getAttributeValue();
                        return StringUtils.isBlank(menuStr)?menuTreeList:MenuUtils.resetNanaMenu(menuTreeList,menuStr);
                    }
                }
            }
            return menuTreeList;
        } catch (Exception e) {
            log.error("resetNanaMenu is error",e);
            return menuTreeList;
        }
    }

}
