package com.digiwin.athena.semc.service.menu.impl;

import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.entity.news.NewsAnnouncement;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementType;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementMapper;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementTypeService;
import com.digiwin.athena.semc.service.news.NewsDataAssignService;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
@RequiredArgsConstructor
public class NewsDataAssignServiceImpl implements NewsDataAssignService {

    @Qualifier("semcAsyncExecutor")
    private final Executor asyncTaskExecutor;

    private final ICacheService cacheService;

    private final NewsAnnouncementMapper newsAnnouncementMapper;

    private final NewsAnnouncementTypeService newsAnnouncementTypeService;

    private static final String NEWS_DATA_ASSIGN_KEY = "NewsDataAssignKey";

    private final JdbcTemplate jdbcTemplate;

    /**
     * 公告数据初始化
     *
     * @param removeLockKey 是否删除缓存key
     * @return 返回
     */
    @Override
    public String startAssign(Boolean removeLockKey) {
        if (Boolean.TRUE.equals(removeLockKey)) {
            cacheService.delete(NEWS_DATA_ASSIGN_KEY);
        }
        if (StringUtils.isNotBlank(cacheService.getValue(NEWS_DATA_ASSIGN_KEY))) {
            log.info("公告数据正在初始化，稍后进行重试");
            return "公告数据正在初始化，稍后进行重试";
        }
        asyncTaskExecutor.execute(() -> handlerAssign(removeLockKey));
        return "公告数据初始化开始";
    }

    private void handlerAssign(Boolean removeLockKey) {
        try {
            log.info("公告数据初始化开始");
            cacheService.cache(NEWS_DATA_ASSIGN_KEY, JSON.toJSONString(removeLockKey), Duration.ofHours(4));

            // 公告表-公告类型路径
            // update t_news_announcement set news_type_path = news_type_id;
            jdbcTemplate.execute("update t_news_announcement set news_type_path = news_type_id where news_type_path is null or news_type_path = ''");

            // 公告权限表-公告类型id
            // delete from t_news_announcement_auth where news_id not in (select id from t_news_announcement);
            // update t_news_announcement_auth a set a.news_type_id = (select b.news_type_id from t_news_announcement b where b.id = a.news_id);
            jdbcTemplate.execute("delete from t_news_announcement_auth where news_id not in (select id from t_news_announcement)");
            jdbcTemplate.execute("update t_news_announcement_auth a set a.news_type_id = (select b.news_type_id from t_news_announcement b where b.id = a.news_id)");

            // 公告类型表-公告总数/公告已发布总数
            InterceptorIgnoreUtil.handler(() -> {
                List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementMapper.selectList(Wrappers.emptyWrapper());
                Map<Long, List<NewsAnnouncement>> newsTypeIdMap = newsAnnouncementList.stream().collect(Collectors.groupingBy(NewsAnnouncement::getNewsTypeId));
                newsTypeIdMap.forEach((k, v) -> {
                    Integer totalCount = v.size();
                    Long publishedCount = v.stream().filter(x -> Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag().equals(x.getNewsStatus())).count();
                    LambdaUpdateWrapper<NewsAnnouncementType> updateWrapper = new LambdaUpdateWrapper<>();
                    updateWrapper.set(NewsAnnouncementType::getNewsCount, totalCount)
                            .set(NewsAnnouncementType::getPublishedNewsCount, publishedCount);
                    updateWrapper.eq(NewsAnnouncementType::getId, k);
                    newsAnnouncementTypeService.update(updateWrapper);
                });
            });

        } catch (Exception e) {
            log.error("公告数据初始化异常：{}", JSON.toJSONString(removeLockKey), e);
        } finally {
            cacheService.delete(NEWS_DATA_ASSIGN_KEY);
            log.info("公告数据初始化结束");
        }
    }
}
