package com.digiwin.athena.semc.service.menu.impl;

import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.semc.dto.menu.manage.UserMenuDataTransferReqDTO;
import com.digiwin.athena.semc.entity.menu.manage.ManageMenuAuth;
import com.digiwin.athena.semc.mapper.menu.manage.ManageMenuAuthMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.req.UserAllPermissionReqDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.resp.AppTenantRespVO;
import com.digiwin.athena.semc.proxy.iam.service.model.resp.TenantUserRespVO;
import com.digiwin.athena.semc.proxy.iam.service.model.resp.UserAllPermissionBaseRespVO;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.service.menu.UserMenuDataTransferService;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
@RequiredArgsConstructor
public class UserMenuDataTransferServiceImpl implements UserMenuDataTransferService {

    @Qualifier("semcAsyncExecutor")
    private final Executor asyncTaskExecutor;

    private final ICacheService cacheService;

    private final IamService iamService;

    private final ManageMenuAuthMapper manageMenuAuthMapper;

    private static final String USER_MENU_DATA_TRANSFER_KEY = "UserMenuDataTransferKey";

    private static final String IT_MAINTENANCE="it-maintenance";

    private static final String AUTHORIZATION_CODE = "authorization-code";

    private static final String CARD_LOGIN_MANAGE = "card-login-manage";

    private static final String ALLOW="allow";

    //要迁移的菜单：事件记录簿、运营单元一览表、授权码管理、登录卡管理、IT运维模组、 ERP单据集成配置器（不迁移了）
    //通用
    private static final Map<String, String> USER_MENU_DATA_TRANSFER_COMMON_MAP = new HashMap<String, String>() {{
        //管理模组-事件记录簿，上线之后对应的菜单是：系统管理-事件记录簿
        put("event-log", "事件记录簿");
        //管理模组-运营单元代码一览表，上线之后对应的菜单是：应用管理-运营单元一览表
        put("system-parameter", "运营单元代码一览表");
        //管理模组-授权码管理
        put(AUTHORIZATION_CODE, "授权码管理");
        //管理模组-用户登录卡管理
        //其中授权码管理、登录卡管理将与管理后台-用户管理-解绑管理，三合一形成综合功能页面：用户管理-登录设备配置
        put(CARD_LOGIN_MANAGE, "用户登录卡管理");
        //用户工具-交付设计器，上线之后对应的菜单是：用户工具-交付设计器
        //
        //put("athena-deliverydesigner", "交付设计器");
    }};

    //特殊
    /*
    private static final Map<String, String> USER_MENU_DATA_TRANSFER_SPECIAL_MAP = new HashMap<String, String>() {{
        //IT运维模组（存在modules中的），上线之后对应的菜单是：应用管理-运营单元一览表
        put(IT_MAINTENANCE, "IT运维模组");
        //用户工具-ERP单据集成配置器，上线之后对应的菜单是：用户工具-ERP单据集成配置器（不迁移了）
        //put("athena-ddsmdesigner", "ERP单据集成配置器");
    }};*/



    @Override
    public String startTransfer(UserMenuDataTransferReqDTO reqDTO) {
        if (Boolean.TRUE.equals(reqDTO.getRemoveLockKey())) {
            cacheService.delete(USER_MENU_DATA_TRANSFER_KEY);
        }
        if (StringUtils.isNotBlank(cacheService.getValue(USER_MENU_DATA_TRANSFER_KEY))){
            log.info("用户菜单正在迁移，稍后进行重试");
            return "用户菜单正在迁移，稍后进行重试";
        }
        if (StringUtils.isNotBlank(reqDTO.getUserId()) && StringUtils.isBlank(reqDTO.getTenantSid())) {
            log.info("参数有误，指定用户时租户id不能为空");
            return "参数有误，指定用户时租户id不能为空";
        }

        asyncTaskExecutor.execute(() -> handlerTransfer(reqDTO));
        return "用户菜单开始迁移";
    }


    private void handlerTransfer(UserMenuDataTransferReqDTO reqDTO) {
        try {
            log.info("用户菜单迁移开始");
            String appId ="Athena";
            cacheService.cache(USER_MENU_DATA_TRANSFER_KEY, JSON.toJSONString(reqDTO), Duration.ofHours(4));
            //查询购买了指定应用的租户集合
            List<AppTenantRespVO> appTenantRespVOList = iamService.queryBuyTenantListByAppId(appId);
            if (CollectionUtils.isEmpty(appTenantRespVOList)) {
                log.info("查询购买了指定应用的租户id为空：{}", appId);
                return;
            }
            for (AppTenantRespVO appTenantRespVO : appTenantRespVOList) {
                String tenantSid = appTenantRespVO.getSid();
                String tenantId = appTenantRespVO.getId();
                //如果指定了迁移某个租户id，这不是该租户则进行跳过
                if (StringUtils.isNotBlank(reqDTO.getTenantSid()) && !reqDTO.getTenantSid().equals(tenantSid)){
                    continue;
                }
                //获取租户下用户列表
                List<TenantUserRespVO> tenantUserRespVOList = iamService.queryUserListByTenantId(tenantSid);
                if (CollectionUtils.isEmpty(tenantUserRespVOList)) {
                    log.info("获取租户下用户列表为空：{}", tenantSid);
                    continue;
                }
                //迁移指定用户
                if (StringUtils.isNotBlank(reqDTO.getUserId())) {
                    tenantUserRespVOList.stream().filter(m -> m.getId().equals(reqDTO.getUserId()))
                            .findFirst()
                            .ifPresent(tenantUserRespVO -> handlerUserMenuTransfer(tenantUserRespVO, tenantId));
                    break;
                }
                //迁移全量用户
                else {
                    for (TenantUserRespVO tenantUserRespVO : tenantUserRespVOList) {
                        handlerUserMenuTransfer(tenantUserRespVO, tenantId);
                    }
                }
            }
        }
        catch (Exception e) {
            log.error("用户菜单迁移异常：{}", JSON.toJSONString(reqDTO),e);
        }
        finally {
            cacheService.delete(USER_MENU_DATA_TRANSFER_KEY);
            log.info("用户菜单迁移结束");
        }

    }


    private void handlerUserMenuTransfer(TenantUserRespVO tenantUserRespVO, String tenantId) {
        String userId = tenantUserRespVO.getId();
        UserAllPermissionReqDTO userAllPermissionReqDTO = new UserAllPermissionReqDTO();
        userAllPermissionReqDTO.setUserId(userId);
        userAllPermissionReqDTO.setTenantId(tenantId);
        UserAllPermissionBaseRespVO userAllPermissionBaseRespVO = iamService.queryUserAllPermission(userAllPermissionReqDTO);
        if (userAllPermissionBaseRespVO==null || userAllPermissionBaseRespVO.getResult()==null) {
            log.info("获取用户的功能权限为空：{}",userId);
            return;
        }
        List<UserAllPermissionBaseRespVO.Permission> permissionList = userAllPermissionBaseRespVO.getResult().getPermissions();
        List<UserAllPermissionBaseRespVO.Module> modules = userAllPermissionBaseRespVO.getResult().getModules();
        Map<String, UserAllPermissionBaseRespVO.Module> moduleMap = modules.stream().collect(Collectors.toMap(UserAllPermissionBaseRespVO.Module::getId, Function.identity(), (v1, v2) -> v1));
        if (CollectionUtils.isEmpty(permissionList)) {
            log.info("获取用户的功能权限集合为空：{}", userId);
            return;
        }
        //特殊之IT运维模组
        handlerItMainTenance(moduleMap,permissionList,tenantUserRespVO, tenantId);

        //通用菜单开始处理
        for (UserAllPermissionBaseRespVO.Permission permission : permissionList) {
            if (!USER_MENU_DATA_TRANSFER_COMMON_MAP.containsKey(permission.getId())) {
                //该菜单不需要迁移
                continue;
            }
            if (!ALLOW.equals(permission.getEffect())) {
                log.info("用户该菜单没有权限跳过：{}，{}，{}", userId, permission.getId(),permission.getName());
                continue;
            }
            saveManageMenuAuth(tenantUserRespVO, tenantId,permission);
        }
    }


    /**
     * IT运维模组处理
     * @param moduleMap
     * @param permissionList
     * @param tenantUserRespVO
     * @param tenantId
     */
    private void handlerItMainTenance(Map<String, UserAllPermissionBaseRespVO.Module> moduleMap,
                                      List<UserAllPermissionBaseRespVO.Permission> permissionList,
                                      TenantUserRespVO tenantUserRespVO,
                                      String tenantId){
        UserAllPermissionBaseRespVO.Module module = moduleMap.get(IT_MAINTENANCE);
        if (module == null) {
            return;
        }
        if (!ALLOW.equals(module.getEffect())) {
            return;
        }
        boolean anyMatch = permissionList.stream().anyMatch(m -> m.getModuleId().equals(IT_MAINTENANCE) && m.getId().equals("module-enable") && m.getEffect().equals(ALLOW));
        if (!anyMatch) {
            return;
        }
        UserAllPermissionBaseRespVO.Permission permission = new UserAllPermissionBaseRespVO.Permission();
        permission.setModuleId(IT_MAINTENANCE);
        permission.setId(IT_MAINTENANCE);
        saveManageMenuAuth(tenantUserRespVO, tenantId,permission);
    }


    private void saveManageMenuAuth(TenantUserRespVO tenantUserRespVO,String tenantId, UserAllPermissionBaseRespVO.Permission permission){
        ManageMenuAuth manageMenuAuth = new ManageMenuAuth();
        manageMenuAuth.setMenuKey(permission.getId());
        //如果是要合并的菜单
        if (permission.getId().equals(AUTHORIZATION_CODE) || permission.getId().equals(CARD_LOGIN_MANAGE)) {
            manageMenuAuth.setMenuKey("user-card");
        }
        manageMenuAuth.setAuthId(tenantUserRespVO.getSid());
        manageMenuAuth.setAuthName(tenantUserRespVO.getName());
        manageMenuAuth.setAuthType(3);
        manageMenuAuth.setTenantId(tenantId);
        manageMenuAuth.setDelFlag(0);
        LambdaQueryWrapper<ManageMenuAuth> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(ManageMenuAuth::getMenuKey, manageMenuAuth.getMenuKey());
        queryWrapper.eq(ManageMenuAuth::getAuthId, manageMenuAuth.getAuthId());
        queryWrapper.eq(ManageMenuAuth::getTenantId, manageMenuAuth.getTenantId());
        queryWrapper.eq(ManageMenuAuth::getAuthType, manageMenuAuth.getAuthType());
        InterceptorIgnoreUtil.handler(()->{
            Long count = manageMenuAuthMapper.selectCount(queryWrapper);
            if (count != null && count <= 0) {
                manageMenuAuthMapper.insert(manageMenuAuth);
            }
        });
    }

}
