package com.digiwin.athena.semc.service.message.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.dto.message.SaveBlackWhiteReq;
import com.digiwin.athena.semc.entity.message.MessageBlackWhite;
import com.digiwin.athena.semc.mapper.message.MessageBlackWhiteMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.RoleDTO;
import com.digiwin.athena.semc.service.message.MessageBlackWhiteService;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 消息接收人黑白名单(TMessageBlackWhite)表服务实现类
 *
 * @author sungqz
 * @since 2024-01-10
 */
@Service
public class MessageBlackWhiteServiceImpl extends ServiceImpl<MessageBlackWhiteMapper, MessageBlackWhite> implements MessageBlackWhiteService {
    @Resource
    private MessageBlackWhiteMapper messageBlackWhiteMapper;

    @Resource
    private IamService iamService;

    /**
     * 查询黑白名单用户列表
     *
     * @param type     名单类型 0:黑名单 1:白名单
     * @param key      关键字
     * @param tenantId 租户id
     * @return 查询结果
     */
    @Override
    public List<MessageBlackWhite> queryUserList(Integer type, String key, String tenantId) {
        if (StringUtils.isBlank(tenantId)) {
            tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        }
        List<MessageBlackWhite> blackWhiteList = messageBlackWhiteMapper.queryUserListByCondition(tenantId, type, StringUtils.isNotBlank(key) ? key.trim() : key);
        if (CollectionUtils.isEmpty(blackWhiteList)) {
            return blackWhiteList;
        }
        if (type != null) {
            List<RoleDTO> userList = iamService.queryAllUserList();
            Map<String, String> idMap = userList.stream().collect(Collectors.toMap(RoleDTO::getId, RoleDTO::getName));
            blackWhiteList.forEach(x -> {
                if (idMap.get(x.getUserId()) != null) {
                    x.setUserName(idMap.get(x.getUserId()));
                }
            });
            saveOrUpdateBatch(blackWhiteList);
        }
        return blackWhiteList;
    }

    /**
     * 根据主键id删除用户
     *
     * @param id 主键id
     * @return 结果
     */
    @Override
    public void delUser(Long id) {
        QueryWrapper<MessageBlackWhite> wrapper = new QueryWrapper<>();
        wrapper.eq("id", id);
        messageBlackWhiteMapper.delete(wrapper);
    }

    /**
     * 保存用户
     *
     * @param saveBlackWhiteReq 请求对象
     * @return 结果
     */
    @Override
    @Transactional
    public void saveUser(SaveBlackWhiteReq saveBlackWhiteReq) {
        // 集合为空，则不保留任何用户
        QueryWrapper<MessageBlackWhite> wrapper = new QueryWrapper<>();
        wrapper.eq("type", saveBlackWhiteReq.getType());
        if (CollectionUtils.isEmpty(saveBlackWhiteReq.getUserList())) {
            messageBlackWhiteMapper.delete(wrapper);
            return;
        }
        // 存在主键id，则是保留的用户，删除不在这其中的用户
        List<Long> idList = saveBlackWhiteReq.getUserList().stream().map(SaveBlackWhiteReq.UserDTO::getId).filter(Objects::nonNull).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(idList)) {
            wrapper.notIn("id", idList);
        }
        messageBlackWhiteMapper.delete(wrapper);

        // 已存在的用户
        List<String> userIdList = saveBlackWhiteReq.getUserList().stream().filter(x -> x.getId() != null).map(SaveBlackWhiteReq.UserDTO::getUserId).collect(Collectors.toList());
        // 不存在主键id的记录是新增的用户
        List<SaveBlackWhiteReq.UserDTO> userList = saveBlackWhiteReq.getUserList().stream().filter(x -> x.getId() == null).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(userList)) {
            return;
        }
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        List<MessageBlackWhite> blackWhiteList = Lists.newArrayList();
        for (SaveBlackWhiteReq.UserDTO userDTO : userList) {
            // 去重
            if (userIdList.contains(userDTO.getUserId())) {
                continue;
            }
            userIdList.add(userDTO.getUserId());
            MessageBlackWhite blackWhite = new MessageBlackWhite();
            blackWhite.setUserSid(userDTO.getUserSid());
            blackWhite.setUserId(userDTO.getUserId());
            blackWhite.setUserName(userDTO.getUserName());
            blackWhite.setTenantId(tenantId);
            blackWhite.setType(saveBlackWhiteReq.getType());
            blackWhiteList.add(blackWhite);
        }
        if (CollectionUtils.isNotEmpty(blackWhiteList)) {
            saveBatch(blackWhiteList);
        }
    }
}
