package com.digiwin.athena.semc.service.message.impl;

import com.google.common.collect.Lists;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.message.CopyTemplateReq;
import com.digiwin.athena.semc.dto.message.QueryTemplateListReq;
import com.digiwin.athena.semc.dto.message.QueryTemplateListResp;
import com.digiwin.athena.semc.dto.message.QueryTemplateRelResp;
import com.digiwin.athena.semc.dto.message.SaveTemplateReq;
import com.digiwin.athena.semc.dto.message.TemplateRelReq;
import com.digiwin.athena.semc.entity.message.DingtalkTemplateRel;
import com.digiwin.athena.semc.entity.message.EmailTemplateRel;
import com.digiwin.athena.semc.entity.message.MessageChannelTemplate;
import com.digiwin.athena.semc.entity.message.SmsTemplateRel;
import com.digiwin.athena.semc.entity.message.WecomTemplateRel;
import com.digiwin.athena.semc.mapper.message.DingtalkTemplateRelMapper;
import com.digiwin.athena.semc.mapper.message.EmailTemplateRelMapper;
import com.digiwin.athena.semc.mapper.message.MessageChannelTemplateMapper;
import com.digiwin.athena.semc.mapper.message.SmsTemplateRelMapper;
import com.digiwin.athena.semc.mapper.message.WecomTemplateRelMapper;
import com.digiwin.athena.semc.proxy.emc.service.EmcService;
import com.digiwin.athena.semc.proxy.emc.service.model.OmEventResp;
import com.digiwin.athena.semc.proxy.emc.service.model.OmEventTemplateDTO;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.message.DingtalkTemplateRelService;
import com.digiwin.athena.semc.service.message.EmailTemplateRelService;
import com.digiwin.athena.semc.service.message.MessageChannelTemplateService;
import com.digiwin.athena.semc.service.message.SmsTemplateRelService;
import com.digiwin.athena.semc.service.message.WecomTemplateRelService;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * 消息模板配置表服务实现类
 *
 * @author sungqz
 * @since 2024-1-8
 */
@Slf4j
@Service
public class MessageChannelTemplateServiceImpl extends ServiceImpl<MessageChannelTemplateMapper, MessageChannelTemplate> implements MessageChannelTemplateService {

    @Resource
    private MessageChannelTemplateMapper messageChannelTemplateMapper;

    @Resource
    private EmailTemplateRelService emailTemplateRelService;

    @Resource
    private EmailTemplateRelMapper emailTemplateRelMapper;

    @Resource
    private SmsTemplateRelService smsTemplateRelService;

    @Resource
    private SmsTemplateRelMapper smsTemplateRelMapper;

    @Resource
    private WecomTemplateRelService wecomTemplateRelService;

    @Resource
    private WecomTemplateRelMapper wecomTemplateRelMapper;

    @Resource
    private DingtalkTemplateRelService dingtalkTemplateRelService;

    @Resource
    private DingtalkTemplateRelMapper dingtalkTemplateRelMapper;

    @Resource
    private EmcService emcService;

    @Resource
    private IamService iamService;

    /**
     * 事件列表长度
     */
    public static final int EVENT_LIST_LENGTH_ONE = 1;

    /**
     * 查询各个渠道模板配置列表
     *
     * @param queryTemplateListReq 请求对象
     * @return 返回对象
     */
    @Override
    public List<QueryTemplateListResp> queryTemplateList(QueryTemplateListReq queryTemplateListReq) {
        List<QueryTemplateListResp> templateListRespList = Lists.newArrayList();
        List<OmEventResp> omEventList = emcService.queryOmTemplateList(queryTemplateListReq.getChannelFlag(), queryTemplateListReq.getNameCondition());
        // 过滤事件
        omEventList = filterEvent(queryTemplateListReq, omEventList);
        if (CollectionUtils.isEmpty(omEventList)) {
            return templateListRespList;
        }

        String tenantId = StringUtils.isBlank(queryTemplateListReq.getTenantId()) ? AppAuthContextHolder.getContext().getAuthoredUser().getTenantId() : queryTemplateListReq.getTenantId();
        // 获取模板在数据库中配置的信息
        List<String> eventIdList = omEventList.stream().map(OmEventResp::getId).collect(Collectors.toList());
        List<MessageChannelTemplate> templateList = messageChannelTemplateMapper.queryTemplateByCondition(tenantId, null, queryTemplateListReq.getChannelFlag(), eventIdList);

        Integer previewType = Constants.PreviewTypeEnum.parseTypeByFlag(queryTemplateListReq.getChannelFlag());
        if (CollectionUtils.isEmpty(templateList)) {
            templateListRespList = omEventList.stream().map(x -> {
                QueryTemplateListResp resp = new QueryTemplateListResp();
                resp.setEventId(x.getId());
                resp.setName(x.getName());
                resp.setRemark(x.getDescription());
                resp.setChannelFlag(queryTemplateListReq.getChannelFlag());
                // 默认是启用状态
                resp.setValidStatus(Constants.VALID_STATUS_ENABLE);
                resp.setModifyUserName(x.getModifyBy());
                resp.setModifyTime(x.getModifyDate());
                x.getTemplates().forEach(y -> y.setPreviewType(previewType));
                resp.setTemplateList(x.getTemplates());
                return resp;
            }).collect(Collectors.toList());
            // 更新时间倒叙排序
            templateListRespList = templateListRespList.stream().sorted(Comparator.comparing(QueryTemplateListResp::getModifyTime, Comparator.nullsLast(Comparator.reverseOrder()))).collect(Collectors.toList());
            // 状态条件只针对父级生效
            if (queryTemplateListReq.getStatus() != null) {
                templateListRespList = templateListRespList.stream().filter(x -> x.getValidStatus().equals(queryTemplateListReq.getStatus())).collect(Collectors.toList());
            }
        } else {
            // 父级模板列表
            List<MessageChannelTemplate> parentList = templateList.stream().filter(x -> x.getParentId() == null).collect(Collectors.toList());
            Map<String, MessageChannelTemplate> eventIdMap = parentList.stream().collect(Collectors.toMap(MessageChannelTemplate::getEventId, Function.identity(), (a, b) -> a));
            templateListRespList = omEventList.stream().map(x -> {
                QueryTemplateListResp resp = new QueryTemplateListResp();
                resp.setEventId(x.getId());
                resp.setName(x.getName());
                resp.setRemark(x.getDescription());
                resp.setChannelFlag(queryTemplateListReq.getChannelFlag());
                resp.setValidStatus(Constants.VALID_STATUS_ENABLE);
                resp.setModifyUserName(x.getModifyBy());
                resp.setModifyTime(x.getModifyDate());
                MessageChannelTemplate template = eventIdMap.get(x.getId());
                if (template != null) {
                    resp.setId(template.getId());
                    resp.setParentId(template.getParentId());
                    resp.setValidStatus(template.getValidStatus());
                    resp.setRemark(template.getRemark());
                    // 展示模板的最近修改时间
                    if (StringUtils.isBlank(x.getModifyDate()) || template.getModifyTime().compareTo(x.getModifyDate()) > 0) {
                        resp.setModifyUserId(template.getModifyUserId());
                        resp.setModifyUserName(template.getModifyUserName());
                        resp.setModifyTime(template.getModifyTime());
                    }
                }
                x.getTemplates().forEach(y -> y.setPreviewType(previewType));
                resp.setTemplateList(x.getTemplates());
                return resp;
            }).collect(Collectors.toList());
            // 更新时间倒叙排序
            templateListRespList = templateListRespList.stream().sorted(Comparator.comparing(QueryTemplateListResp::getModifyTime, Comparator.nullsLast(Comparator.reverseOrder()))).collect(Collectors.toList());
            // 状态条件只针对父级生效
            if (queryTemplateListReq.getStatus() != null) {
                templateListRespList = templateListRespList.stream().filter(x -> x.getValidStatus().equals(queryTemplateListReq.getStatus())).collect(Collectors.toList());
            }
            if (CollectionUtils.isEmpty(templateListRespList)) {
                return templateListRespList;
            }

            List<Long> parentIdList = templateListRespList.stream().map(QueryTemplateListResp::getId).filter(Objects::nonNull).collect(Collectors.toList());
            // 子模版
            Map<String, OmEventResp> omEventIdMap = omEventList.stream().collect(Collectors.toMap(OmEventResp::getId, Function.identity(), (a, b) -> a));
            List<MessageChannelTemplate> childList = templateList.stream().filter(x -> x.getParentId() != null).collect(Collectors.toList());
            List<QueryTemplateListResp> childRespList = Lists.newArrayList();
            for (MessageChannelTemplate x : childList) {
                if (!parentIdList.contains(x.getParentId())) {
                    continue;
                }
                QueryTemplateListResp resp = new QueryTemplateListResp();
                resp.setId(x.getId());
                resp.setEventId(x.getEventId());
                resp.setRemark(x.getRemark());
                resp.setValidStatus(x.getValidStatus());
                resp.setChannelFlag(queryTemplateListReq.getChannelFlag());
                resp.setParentId(x.getParentId());
                resp.setModifyUserId(x.getModifyUserId());
                resp.setModifyUserName(x.getModifyUserName());
                resp.setModifyTime(x.getModifyTime());

                // 获取事件名称及渠道列表
                OmEventResp eventResp = omEventIdMap.get(x.getEventId());
                eventResp.getTemplates().forEach(y -> y.setPreviewType(previewType));
                resp.setTemplateList(eventResp.getTemplates());
                resp.setName(eventResp.getName());
                childRespList.add(resp);
            }
            // 按id升序排序
            childRespList = childRespList.stream().sorted(Comparator.comparing(QueryTemplateListResp::getId)).collect(Collectors.toList());
            templateListRespList.addAll(childRespList);
        }
        if (StringUtils.isNotBlank(queryTemplateListReq.getRemark())) {
            templateListRespList = templateListRespList.stream().filter(x -> queryTemplateListReq.getRemark().equals(x.getRemark())).collect(Collectors.toList());
        }
        return templateListRespList;
    }

    /**
     * 过滤事件
     *
     * @param queryTemplateListReq 请求对象
     * @param omEventList          om事件列表
     * @return 返回
     */
    private List<OmEventResp> filterEvent(QueryTemplateListReq queryTemplateListReq, List<OmEventResp> omEventList) {
        if (CollectionUtils.isEmpty(omEventList)) {
            return omEventList;
        }
        // 过滤掉默认事件
        omEventList.removeIf(x -> Constants.SEMC_MESSAGE_DEFAULT_EVENT_ID.equals(x.getId()));
        // 过滤掉没配置模板的事件
        omEventList.removeIf(x -> CollectionUtils.isEmpty(x.getTemplates()));
        // 邮箱过滤掉不包含html的模板
        if (Constants.MessageChannelEnum.EMAIL.getFlag().equals(queryTemplateListReq.getChannelFlag())) {
            for (OmEventResp resp : omEventList) {
                List<OmEventTemplateDTO> templateList = resp.getTemplates().stream().filter(x -> StringUtils.isNotBlank(x.getTemplateUrl())).collect(Collectors.toList());
                resp.setTemplates(templateList);
            }
            omEventList.removeIf(x -> CollectionUtils.isEmpty(x.getTemplates()));
        }
        if (CollectionUtils.isEmpty(omEventList)) {
            return omEventList;
        }
        // 过滤出指定的事件id
        if (StringUtils.isNotBlank(queryTemplateListReq.getEventId())) {
            omEventList = omEventList.stream().filter(x -> queryTemplateListReq.getEventId().equals(x.getId())).collect(Collectors.toList());
            return omEventList;
        }
        return omEventList;
    }

    /**
     * 修改模板状态
     *
     * @param saveTemplateReq 请求对象
     * @return 返回对象
     */
    @Override
    public Long updateStatus(SaveTemplateReq saveTemplateReq) {
        MessageChannelTemplate template;
        String userName = AppAuthContextHolder.getContext().getAuthoredUser().getUserName();
        // 不存在则新增
        if (saveTemplateReq.getTemplateId() == null) {
            template = buildTemplate(saveTemplateReq, userName);
            template.setValidStatus(saveTemplateReq.getStatus());
            messageChannelTemplateMapper.insert(template);
        } else { // 存在则修改状态
            template = queryMessageChannelTemplate(saveTemplateReq);
            template.setModifyUserName(userName);
            template.setValidStatus(saveTemplateReq.getStatus());
            messageChannelTemplateMapper.updateById(template);
        }
        return template.getId();
    }

    /**
     * 查询模板
     *
     * @param saveTemplateReq 请求入参
     * @return 返回
     */
    private MessageChannelTemplate queryMessageChannelTemplate(SaveTemplateReq saveTemplateReq) {
        QueryWrapper<MessageChannelTemplate> wrapper = new QueryWrapper<>();
        wrapper.eq("id", saveTemplateReq.getTemplateId());
        return messageChannelTemplateMapper.selectOne(wrapper);
    }

    /**
     * 构建模板参数
     *
     * @param saveTemplateReq 请求入参
     * @param userName        用户名
     * @return 返回
     */
    private MessageChannelTemplate buildTemplate(SaveTemplateReq saveTemplateReq, String userName) {
        MessageChannelTemplate template = new MessageChannelTemplate();
        template.setEventId(saveTemplateReq.getEventId());
        template.setRemark(saveTemplateReq.getRemark());
        template.setChannelFlag(saveTemplateReq.getChannelFlag());
        template.setModifyUserName(userName);
        return template;
    }

    /**
     * 更新模板
     *
     * @param saveTemplateReq 请求对象
     * @return 返回对象
     */
    @Override
    public Long updateTemplate(SaveTemplateReq saveTemplateReq) {
        // 不存在则新增
        String userName = AppAuthContextHolder.getContext().getAuthoredUser().getUserName();
        MessageChannelTemplate template;
        if (saveTemplateReq.getTemplateId() == null) {
            template = buildTemplate(saveTemplateReq, userName);
            messageChannelTemplateMapper.insert(template);
        } else { // 存在则修改状态
            template = queryMessageChannelTemplate(saveTemplateReq);
            template.setModifyUserName(userName);
            template.setRemark(saveTemplateReq.getRemark());
            messageChannelTemplateMapper.updateById(template);
        }
        return template.getId();
    }

    /**
     * 查看模板接收人
     *
     * @param templateRelReq 请求对象
     * @return 返回对象
     */
    @Override
    public QueryTemplateRelResp queryRelPerson(TemplateRelReq templateRelReq) {
        QueryTemplateRelResp relResp = new QueryTemplateRelResp();
        String tenantId = StringUtils.isBlank(templateRelReq.getTenantId()) ? AppAuthContextHolder.getContext().getAuthoredUser().getTenantId() : templateRelReq.getTenantId();
        // 邮箱
        if (Constants.MessageChannelEnum.EMAIL.getFlag().equals(templateRelReq.getChannelFlag())) {
            List<EmailTemplateRel> emailTemplateRelList = emailTemplateRelMapper.queryEmailByCondition(tenantId, templateRelReq);
            if (CollectionUtils.isEmpty(emailTemplateRelList)) {
                return relResp;
            }
            Map<Integer, List<EmailTemplateRel>> contactList = emailTemplateRelList.stream().collect(Collectors.groupingBy(EmailTemplateRel::getEmailType));
            contactList.forEach((k, v) -> {
                // 收件人
                if (Constants.EmailTypeEnum.CONTACT.getFlag().equals(k)) {
                    List<QueryTemplateRelResp.RelPersonDTO> personDTOList = buildRelPersonList(v);
                    relResp.setContactList(personDTOList);
                }
                // 抄送人
                else if (Constants.EmailTypeEnum.CCCONTACT.getFlag().equals(k)) {
                    List<QueryTemplateRelResp.RelPersonDTO> personDTOList = buildRelPersonList(v);
                    relResp.setCcContactList(personDTOList);
                }
                // 密送人
                else {
                    List<QueryTemplateRelResp.RelPersonDTO> personDTOList = buildRelPersonList(v);
                    relResp.setBccContactList(personDTOList);
                }
            });
        }
        // 短信
        else if (Constants.MessageChannelEnum.SMS.getFlag().equals(templateRelReq.getChannelFlag())) {
            List<SmsTemplateRel> smsTemplateRelList = smsTemplateRelMapper.queryTelephoneByCondition(tenantId, templateRelReq.getEventId(), templateRelReq.getLanguage());
            if (CollectionUtils.isEmpty(smsTemplateRelList)) {
                return relResp;
            }
            List<QueryTemplateRelResp.RelPersonDTO> telephoneList = smsTemplateRelList.stream().map(x -> {
                QueryTemplateRelResp.RelPersonDTO personDTO = new QueryTemplateRelResp.RelPersonDTO();
                personDTO.setId(x.getId());
                personDTO.setValue(x.getTelephone());
                personDTO.setLanguage(x.getLanguage());
                return personDTO;
            }).collect(Collectors.toList());
            relResp.setContactList(telephoneList);
        }
        // 企微
        else if (Constants.MessageChannelEnum.WECOM.getFlag().equals(templateRelReq.getChannelFlag())) {
            List<WecomTemplateRel> wecomTemplateRelList = wecomTemplateRelMapper.queryWeComUserByCondition(tenantId, templateRelReq.getEventId(), templateRelReq.getLanguage());
            if (CollectionUtils.isEmpty(wecomTemplateRelList)) {
                return relResp;
            }
            List<QueryTemplateRelResp.RelPersonDTO> wecomUserIdList = wecomTemplateRelList.stream().map(x -> {
                QueryTemplateRelResp.RelPersonDTO personDTO = new QueryTemplateRelResp.RelPersonDTO();
                personDTO.setId(x.getId());
                personDTO.setValue(x.getUserId());
                personDTO.setLanguage(x.getLanguage());
                return personDTO;
            }).collect(Collectors.toList());
            relResp.setContactList(wecomUserIdList);
        }
        // 钉钉
        else if (Constants.MessageChannelEnum.DINGTALK.getFlag().equals(templateRelReq.getChannelFlag())) {
            List<DingtalkTemplateRel> dingtalkTemplateRelList = dingtalkTemplateRelMapper.queryDingTalkUserByCondition(tenantId, templateRelReq.getEventId(), templateRelReq.getLanguage());
            if (CollectionUtils.isEmpty(dingtalkTemplateRelList)) {
                return relResp;
            }
            List<QueryTemplateRelResp.RelPersonDTO> wecomUserIdList = dingtalkTemplateRelList.stream().map(x -> {
                QueryTemplateRelResp.RelPersonDTO personDTO = new QueryTemplateRelResp.RelPersonDTO();
                personDTO.setId(x.getId());
                personDTO.setValue(x.getUserId());
                personDTO.setLanguage(x.getLanguage());
                return personDTO;
            }).collect(Collectors.toList());
            relResp.setContactList(wecomUserIdList);
        }
        return relResp;
    }

    /**
     * 构建返回邮箱模板收件人
     *
     * @param emailTemplateRelList 模板收件人对象
     * @return 返回对象
     */
    private List<QueryTemplateRelResp.RelPersonDTO> buildRelPersonList(List<EmailTemplateRel> emailTemplateRelList) {
        return emailTemplateRelList.stream().map(x -> {
            QueryTemplateRelResp.RelPersonDTO personDTO = new QueryTemplateRelResp.RelPersonDTO();
            personDTO.setId(x.getId());
            personDTO.setValue(x.getContact());
            personDTO.setLanguage(x.getLanguage());
            return personDTO;
        }).collect(Collectors.toList());
    }

    /**
     * 添加模板接收人
     *
     * @param templateRelReq 请求对象
     * @return 返回结果
     */
    @Override
    @Transactional
    public void addRelPerson(TemplateRelReq templateRelReq) {
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        TreeSet<String> noRepeatList = new TreeSet<>(templateRelReq.getContactList());
        QueryTemplateRelResp relResp = queryRelPerson(templateRelReq);
        // 邮箱
        if (Constants.MessageChannelEnum.EMAIL.getFlag().equals(templateRelReq.getChannelFlag())) {
            // 先去除表里已经存在的邮箱
            if (Constants.EmailTypeEnum.CONTACT.getFlag().equals(templateRelReq.getEmailType()) && CollectionUtils.isNotEmpty(relResp.getContactList())) {
                removeDuplicate(noRepeatList, relResp.getContactList());
            } else if (Constants.EmailTypeEnum.CCCONTACT.getFlag().equals(templateRelReq.getEmailType()) && CollectionUtils.isNotEmpty(relResp.getCcContactList())) {
                removeDuplicate(noRepeatList, relResp.getCcContactList());
            } else if (Constants.EmailTypeEnum.BCCCONTACT.getFlag().equals(templateRelReq.getEmailType()) && CollectionUtils.isNotEmpty(relResp.getBccContactList())) {
                removeDuplicate(noRepeatList, relResp.getBccContactList());
            }
            // 表里没有模板，则先新增模板，为了获取模板主键id
            MessageChannelTemplate template = new MessageChannelTemplate();
            if (templateRelReq.getTemplateId() == null) {
                List<OmEventResp> omEventList = emcService.queryOmTemplateList(Constants.MessageChannelEnum.EMAIL.getFlag(), templateRelReq.getEventId());
                omEventList = omEventList.stream().filter(x -> templateRelReq.getEventId().equals(x.getId())).collect(Collectors.toList());
                template.setEventId(templateRelReq.getEventId());
                template.setRemark(CollectionUtils.isNotEmpty(omEventList) ? omEventList.get(0).getDescription() : "");
                template.setValidStatus(Constants.VALID_STATUS_ENABLE);
                template.setModifyUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
                template.setChannelFlag(Constants.MessageChannelEnum.EMAIL.getFlag());
                messageChannelTemplateMapper.insert(template);
                templateRelReq.setTemplateId(template.getId());
            }

            List<EmailTemplateRel> emailTemplateRelList = noRepeatList.stream().map(x -> {
                EmailTemplateRel emailTemplateRel = new EmailTemplateRel();
                emailTemplateRel.setTemplateId(template.getId() != null ? template.getId() : templateRelReq.getTemplateId());
                emailTemplateRel.setEventId(templateRelReq.getEventId());
                emailTemplateRel.setLanguage(templateRelReq.getLanguage());
                emailTemplateRel.setEmailType(templateRelReq.getEmailType());
                emailTemplateRel.setContact(x);
                return emailTemplateRel;
            }).collect(Collectors.toList());
            // 500个一组分批存储，防止一次数据过大
            List<List<EmailTemplateRel>> partList = Lists.partition(emailTemplateRelList, 500);
            for (List<EmailTemplateRel> tempList : partList) {
                emailTemplateRelService.saveBatch(tempList);
            }
        }
        // 短信
        else if (Constants.MessageChannelEnum.SMS.getFlag().equals(templateRelReq.getChannelFlag())) {
            // 先去除表里已经存在的手机号
            removeDuplicate(noRepeatList, relResp.getContactList());
            // 新增
            List<SmsTemplateRel> emailTemplateRelList = noRepeatList.stream().map(x -> {
                SmsTemplateRel smsTemplateRel = new SmsTemplateRel();
                smsTemplateRel.setEventId(templateRelReq.getEventId());
                smsTemplateRel.setLanguage(templateRelReq.getLanguage());
                smsTemplateRel.setTelephone(x);
                smsTemplateRel.setTenantId(tenantId);
                return smsTemplateRel;
            }).collect(Collectors.toList());
            List<List<SmsTemplateRel>> partList = Lists.partition(emailTemplateRelList, 500);
            for (List<SmsTemplateRel> tempList : partList) {
                smsTemplateRelService.saveBatch(tempList);
            }
        }
        // 企微
        else if (Constants.MessageChannelEnum.WECOM.getFlag().equals(templateRelReq.getChannelFlag())) {
            // 先去除表里已经存在的用户id
            removeDuplicate(noRepeatList, relResp.getContactList());
            // 新增
            List<WecomTemplateRel> wecomTemplateRelList = noRepeatList.stream().map(x -> {
                WecomTemplateRel wecomTemplateRel = new WecomTemplateRel();
                wecomTemplateRel.setEventId(templateRelReq.getEventId());
                wecomTemplateRel.setLanguage(templateRelReq.getLanguage());
                wecomTemplateRel.setUserId(x);
                wecomTemplateRel.setTenantId(tenantId);
                return wecomTemplateRel;
            }).collect(Collectors.toList());
            List<List<WecomTemplateRel>> partList = Lists.partition(wecomTemplateRelList, 500);
            for (List<WecomTemplateRel> tempList : partList) {
                wecomTemplateRelService.saveBatch(tempList);
            }
        }
        // 钉钉
        else if (Constants.MessageChannelEnum.DINGTALK.getFlag().equals(templateRelReq.getChannelFlag())) {
            // 先去除表里已经存在的用户id
            removeDuplicate(noRepeatList, relResp.getContactList());
            // 新增
            List<DingtalkTemplateRel> dingtalkTemplateRelList = noRepeatList.stream().map(x -> {
                DingtalkTemplateRel dingtalkTemplateRel = new DingtalkTemplateRel();
                dingtalkTemplateRel.setEventId(templateRelReq.getEventId());
                dingtalkTemplateRel.setLanguage(templateRelReq.getLanguage());
                dingtalkTemplateRel.setUserId(x);
                dingtalkTemplateRel.setTenantId(tenantId);
                return dingtalkTemplateRel;
            }).collect(Collectors.toList());
            List<List<DingtalkTemplateRel>> partList = Lists.partition(dingtalkTemplateRelList, 500);
            for (List<DingtalkTemplateRel> tempList : partList) {
                dingtalkTemplateRelService.saveBatch(tempList);
            }
        }
        // 维护模板的最近修改人和修改时间
        updateTemplateTime(templateRelReq);
    }

    /**
     * 过滤掉相同的元素
     *
     * @param noRepeatList 源数据
     * @param contactList  新数据
     */
    private void removeDuplicate(TreeSet<String> noRepeatList, List<QueryTemplateRelResp.RelPersonDTO> contactList) {
        if (CollectionUtils.isEmpty(noRepeatList) || CollectionUtils.isEmpty(contactList)) {
            return;
        }
        List<String> existList = contactList.stream().map(QueryTemplateRelResp.RelPersonDTO::getValue).collect(Collectors.toList());
        existList.forEach(noRepeatList::remove);
    }

    /**
     * 维护模板的最近修改人和修改时间
     *
     * @param templateRelReq 请求对象
     */
    private void updateTemplateTime(TemplateRelReq templateRelReq) {
        String userId = AppAuthContextHolder.getContext().getAuthoredUser().getUserId();
        String userName = AppAuthContextHolder.getContext().getAuthoredUser().getUserName();
        messageChannelTemplateMapper.updateTemplateByCondition(userId, userName, LocalDateTime.now(), templateRelReq.getTemplateId());
    }

    /**
     * 删除模板收件人
     *
     * @param templateRelReq 请求对象
     * @return 返回结果
     */
    @Override
    @Transactional
    public void delRelPerson(TemplateRelReq templateRelReq) {
        // 邮箱
        if (Constants.MessageChannelEnum.EMAIL.getFlag().equals(templateRelReq.getChannelFlag())) {
            QueryWrapper<EmailTemplateRel> emailWrapper = new QueryWrapper<>();
            emailWrapper.eq("id", templateRelReq.getId());
            emailTemplateRelMapper.delete(emailWrapper);
        }
        // 短信
        else if (Constants.MessageChannelEnum.SMS.getFlag().equals(templateRelReq.getChannelFlag())) {
            QueryWrapper<SmsTemplateRel> smsWrapper = new QueryWrapper<>();
            smsWrapper.eq("id", templateRelReq.getId());
            smsTemplateRelMapper.delete(smsWrapper);
        }
        // 企微
        else if (Constants.MessageChannelEnum.WECOM.getFlag().equals(templateRelReq.getChannelFlag())) {
            QueryWrapper<WecomTemplateRel> wecomWrapper = new QueryWrapper<>();
            wecomWrapper.eq("id", templateRelReq.getId());
            wecomTemplateRelMapper.delete(wecomWrapper);
        }
        // 钉钉
        else if (Constants.MessageChannelEnum.DINGTALK.getFlag().equals(templateRelReq.getChannelFlag())) {
            QueryWrapper<DingtalkTemplateRel> dingTalkWrapper = new QueryWrapper<>();
            dingTalkWrapper.eq("id", templateRelReq.getId());
            dingtalkTemplateRelMapper.delete(dingTalkWrapper);
        }
        // 维护模板的最近修改人和修改时间
        updateTemplateTime(templateRelReq);
    }

    /**
     * 查询租户下备注对应的事件id
     *
     * @param tenantId 租户id
     * @param remark   备注
     * @return 返回对象
     */
    @Override
    public List<String> queryEventIdList(String tenantId, String remark) {
        List<MessageChannelTemplate> templateList = messageChannelTemplateMapper.queryTemplateByCondition(tenantId, remark, Constants.MessageChannelEnum.EMAIL.getFlag(), null);
        log.info("MessageChannelTemplateServiceImpl | query event id list result:{}. tenantId:{}, remark:{}", templateList, tenantId, remark);
        // 如果没有编辑过，则查询om平台事件列表
        if (CollectionUtils.isEmpty(templateList)) {
            List<OmEventResp> omEventList = emcService.queryOmTemplateList(Constants.MessageChannelEnum.EMAIL.getFlag(), "");
            return omEventList.stream().filter(x -> remark.equals(x.getDescription())).map(OmEventResp::getId).limit(1).collect(Collectors.toList());
        }

        // 如果查询出来的事件id有多个，则返回对应语系的事件
        List<String> eventIdList = templateList.stream().map(MessageChannelTemplate::getEventId).distinct().collect(Collectors.toList());
        if (eventIdList.size() > EVENT_LIST_LENGTH_ONE) {
            List<String> respEventIdList = Lists.newArrayList();
            String locale = iamService.queryUserLangMetadata();
            log.info("MessageChannelTemplateServiceImpl | query event id list locale:{}. tenantId:{}, remark:{}", locale, tenantId, remark);
            locale = locale.replaceAll("_", "-");

            List<OmEventResp> omEventList = emcService.queryOmTemplateList(Constants.MessageChannelEnum.EMAIL.getFlag(), "");
            List<OmEventResp> filterList = omEventList.stream().filter(x -> eventIdList.contains(x.getId()) && CollectionUtils.isNotEmpty(x.getTemplates())).collect(Collectors.toList());
            log.info("MessageChannelTemplateServiceImpl | filter event template list result:{}. tenantId:{}, remark:{}", filterList, tenantId, remark);
            for (OmEventResp resp : filterList) {
                List<String> languageList = resp.getTemplates().stream().map(OmEventTemplateDTO::getLanguage).collect(Collectors.toList());
                if (languageList.contains(locale)) {
                    respEventIdList.add(resp.getId());
                    break;
                }
            }
            return respEventIdList;
        } else {
            return eventIdList;
        }
    }

    /**
     * 根据父级id查询子模板数量
     *
     * @param parentId 父级id
     * @return 返回
     */
    public Long queryChildTemplateCount(Long parentId) {
        QueryWrapper<MessageChannelTemplate> emailWrapper = new QueryWrapper<>();
        emailWrapper.eq("parent_id", parentId);
        Long count = messageChannelTemplateMapper.selectCount(emailWrapper);
        return count;
    }

    /**
     * 复制模板
     *
     * @param copyTemplateReq 请求参数
     */
    @Override
    @Transactional
    public void copyTemplate(CopyTemplateReq copyTemplateReq) {
        String userName = AppAuthContextHolder.getContext().getAuthoredUser().getUserName();
        // 如果模板主键id存在，说明表里存在这条记录，则直接插入子模板
        if (copyTemplateReq.getId() != null) {
            MessageChannelTemplate template = new MessageChannelTemplate();
            template.setEventId(copyTemplateReq.getEventId());
            template.setValidStatus(copyTemplateReq.getStatus());
            template.setChannelFlag(Constants.MessageChannelEnum.EMAIL.getFlag());
            template.setParentId(copyTemplateReq.getId());
            template.setModifyUserName(userName);
            messageChannelTemplateMapper.insert(template);

            // 更新最近修改人和修改时间
            TemplateRelReq templateRelReq = new TemplateRelReq();
            templateRelReq.setTemplateId(copyTemplateReq.getId());
            updateTemplateTime(templateRelReq);
        }
        // 如果模板主键id不存在
        else {
            // 先记录这条模板
            MessageChannelTemplate template = new MessageChannelTemplate();
            template.setEventId(copyTemplateReq.getEventId());
            template.setRemark(copyTemplateReq.getRemark());
            template.setValidStatus(copyTemplateReq.getStatus());
            template.setChannelFlag(Constants.MessageChannelEnum.EMAIL.getFlag());
            template.setModifyUserName(userName);
            messageChannelTemplateMapper.insert(template);

            // 然后插入子模板
            MessageChannelTemplate childTemplate = new MessageChannelTemplate();
            childTemplate.setEventId(copyTemplateReq.getEventId());
            childTemplate.setValidStatus(copyTemplateReq.getStatus());
            childTemplate.setChannelFlag(Constants.MessageChannelEnum.EMAIL.getFlag());
            childTemplate.setParentId(template.getId());
            childTemplate.setModifyUserName(userName);
            messageChannelTemplateMapper.insert(childTemplate);

            // 更新最近修改人和修改时间
            TemplateRelReq templateRelReq = new TemplateRelReq();
            templateRelReq.setTemplateId(template.getId());
            updateTemplateTime(templateRelReq);
        }

    }
}
