package com.digiwin.athena.semc.service.message.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.entity.message.ThirdMessageInfo;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.event.ThirdMessageWecomEvent;
import com.digiwin.athena.semc.mapper.message.ThirdMessageInfoMapper;
import com.digiwin.athena.semc.proxy.emc.service.EmcService;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.AppUserDTO;
import com.digiwin.athena.semc.service.message.ThirdMessageInfoService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import static com.digiwin.athena.semc.common.Constants.MESSAGE_LINK_URL;

/**
 * 异构系统消息表(TThirdMessageInfo)表服务实现类
 *
 * @author sungqz
 * @since 2024-03-05
 */
@Service
@Slf4j
public class ThirdMessageInfoServiceImpl extends ServiceImpl<ThirdMessageInfoMapper, ThirdMessageInfo> implements ThirdMessageInfoService {

    @Resource
    private IamService iamService;

    @Resource
    private EmcService emcService;

    @Resource
    private EnvProperties envProperties;

    /**
     * 发送企微
     *
     * @param event 消息列表
     */
    @Override
    public void sendWecom(ThirdMessageWecomEvent event) {
        String userToken = event.getThirdMessageWecomEventDTO().getUserToken();
        // 查询企微配置
        JSONObject jsonObject = emcService.queryWeComConfig(userToken);
        // 下地客户无法通过上面接口获取企微配置，所以从配置中读取
        if(ObjectUtils.isEmpty(jsonObject)){
            jsonObject = new JSONObject();
            jsonObject.put("appId", envProperties.getWeChatAppId());
            jsonObject.put("corpId", envProperties.getWeChatCorpId());
        }

        String tenantId = event.getThirdMessageWecomEventDTO().getTenantId();
        List<ThirdMessageInfo> newMessageInfoList = event.getThirdMessageWecomEventDTO().getNewMessageInfoList();
        for (ThirdMessageInfo thirdMessageInfo : newMessageInfoList) {
            // 查询所有归户的用户
            List<AppUserDTO> appUserList = iamService.queryAllVerifyMappingUser(userToken, thirdMessageInfo.getMessageAppCode());
            if (CollectionUtils.isEmpty(appUserList)) {
                return;
            }

            // 如果消息是全员的，则全员推送企微
            if (thirdMessageInfo.getMessageOwner().equals(Constants.ALL_STAFF_CAN_SEE_FLAG)) {
                List<String> userIdList = appUserList.stream().map(AppUserDTO::getUserId).collect(Collectors.toList());
                sendWecomMessage(jsonObject, thirdMessageInfo, userIdList, tenantId, userToken);
            } else { // 先查询所有归户的用户，匹配出用户推送企微
                List<String> verifyIdList = Arrays.asList(thirdMessageInfo.getMessageOwner().split(","));
                List<String> userIdList = appUserList.stream().filter(x -> verifyIdList.contains(x.getVerifyUserId())).map(AppUserDTO::getUserId).collect(Collectors.toList());
                sendWecomMessage(jsonObject, thirdMessageInfo, userIdList, tenantId, userToken);
            }
        }
    }

    /**
     * 发送企微消息
     *
     * @param jsonObject       企微配置
     * @param thirdMessageInfo 消息
     * @param userIdList       用户列表
     * @param tenantId         租户id
     * @param userToken        userToken
     */
    private void sendWecomMessage(JSONObject jsonObject, ThirdMessageInfo thirdMessageInfo, List<String> userIdList, String tenantId, String userToken) {
        // 如果没有链接，则消息无需跳转
        if (StringUtils.isBlank(thirdMessageInfo.getMessageUrl()) && StringUtils.isBlank(thirdMessageInfo.getMobileMessageUrl())) {
            emcService.sendWeChat(userIdList, thirdMessageInfo.getMessageTitle(), userToken);
        }
        // 如果没有取到企微配置，则消息不能跳转
        else if (ObjectUtils.isEmpty(jsonObject) || StringUtils.isBlank(jsonObject.getString("appId")) || StringUtils.isBlank(jsonObject.getString("corpId"))) {
            emcService.sendWeChat(userIdList, thirdMessageInfo.getMessageTitle(), userToken);
        }
        else {
            String agentId = jsonObject.getString("appId");
            String corpId = jsonObject.getString("corpId");

            // 回调地址
            String redirectUri = envProperties.getSemcUrl() + "/semc/router/thirdMessageRouting?appId=" + agentId + "&tenantId=" + tenantId +
                    "&messageAppCode=" + thirdMessageInfo.getMessageAppCode() + "&messageUrl=" + thirdMessageInfo.getMessageUrl() + "&mobileMessageUrl=" + thirdMessageInfo.getMobileMessageUrl();
            log.info("third message send wecom. redirectUri:{}", redirectUri);
            String encodeRedirectUri = null;
            try {
                encodeRedirectUri = URLEncoder.encode(redirectUri, "utf-8");
            } catch (UnsupportedEncodingException e) {
                log.error("URLEncoder.encode(uri) occur error. uri:{}", redirectUri, e);
            }

            String weChatUrl = "https://open.weixin.qq.com/connect/oauth2/authorize?appid=" + corpId + "&redirect_uri=" + encodeRedirectUri + "&response_type=code&scope=snsapi_base&state=STATE&agentid=" + agentId + "#wechat_redirect";
            String text = String.format(MESSAGE_LINK_URL, weChatUrl, thirdMessageInfo.getMessageTitle());
            emcService.sendWeChat(userIdList, text, userToken);
        }
    }
}
