package com.digiwin.athena.semc.service.message.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.semc.entity.message.*;
import com.digiwin.athena.semc.event.ThirdMessageWecomEvent;
import com.digiwin.athena.semc.event.dto.ThirdMessageWecomEventDTO;
import com.digiwin.athena.semc.mapper.message.ThirdMessageAppSortMapper;
import com.digiwin.athena.semc.mapper.message.ThirdMessageTypeMapper;
import com.digiwin.athena.semc.vo.message.MessageAppNumResp;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.semc.common.BizException;
import com.digiwin.athena.semc.dto.message.*;
import com.digiwin.athena.semc.entity.message.*;
import com.digiwin.athena.semc.mapper.message.ThirdMessageAppSortMapper;
import com.digiwin.athena.semc.mapper.message.ThirdMessageTypeMapper;
import com.digiwin.athena.semc.vo.message.MessageAppNumResp;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.EAIServiceNameEnum;
import com.digiwin.athena.semc.dto.erpsso.QuerySsoListReq;
import com.digiwin.athena.semc.dto.mq.MessageDO;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.entity.sso.ErpSsoInfo;
import com.digiwin.athena.semc.entity.sso.ThirdSsoInfo;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.mapper.message.ThirdSystemMessageMapper;
import com.digiwin.athena.semc.mapper.portal.PreinstalledApplicationMapper;
import com.digiwin.athena.semc.mapper.sso.ErpSsoInfoMapper;
import com.digiwin.athena.semc.mapper.sso.ThirdSsoInfoMapper;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.AccountDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.AppUserDTO;
import com.digiwin.athena.semc.proxy.tripartite.service.TripartiteService;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.cache.RedisLock;
import com.digiwin.athena.semc.service.message.ThirdMessageInfoService;
import com.digiwin.athena.semc.service.message.ThirdMessageReadInfoService;
import com.digiwin.athena.semc.service.message.ThirdSystemMessageService;
import com.digiwin.athena.semc.service.mq.MessageSendService;
import com.digiwin.athena.semc.service.news.ThirdNewsAnnouncementService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import org.springframework.context.i18n.LocaleContextHolder;

import org.springframework.context.ApplicationEventPublisher;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;
import io.vavr.Tuple3;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;

/**
 * @description: 三方异构系统消息接入
 * @createDate: 2024/3/4
 * @author: sungqz
 */
@Slf4j
@Service
public class ThirdSystemMessageServiceImpl extends ServiceImpl<ThirdSystemMessageMapper, ThirdMessageConfig> implements ThirdSystemMessageService {

    @Resource
    private EnvProperties envProperties;

    @Resource
    ThirdSystemMessageMapper thirdSystemMessageMapper;

    @Autowired
    ThirdMessageInfoService thirdMessageInfoService;

    @Autowired
    ThirdMessageReadInfoService thirdMessageReadInfoService;

    @Autowired
    private TripartiteService tripartiteService;

    @Autowired
    private ESPService espService;

    @Resource
    private IamService iamService;

    @Autowired
    private AppLinkService appLinkService;

    @Resource
    private MessageSendService messageSendService;

    @Resource
    private ThirdSsoInfoMapper thirdSsoInfoMapper;

    @Resource
    private ErpSsoInfoMapper erpSsoInfoMapper;

    @Autowired
    ThirdNewsAnnouncementService thirdNewsAnnouncementService;

    @Resource
    PreinstalledApplicationMapper preinstalledApplicationMapper;

    @Resource
    private MessageUtils messageUtils;

    @Resource
    private ThirdMessageAppSortMapper thirdMessageAppSortMapper;

    @Resource
    private ApplicationEventPublisher eventPublisher;

    @Resource
    private Converter converter;

    @Resource
    private ThirdMessageTypeMapper thirdMessageTypeMapper;

    /**
     * 全员可见标识
     */
    public static final String ALL_STAFF_CAN_SEE_FLAG = "all";


    /**
     * 读取状态 0：全部 1：已读 2：未读
     */
    public static final int MESSAGE_READ_FLAG_ALL = 0;

    /**
     * 读取状态 0：全部 1：已读 2：未读
     */
    public static final int MESSAGE_READ_FLAG_READ = 1;

    /**
     * 读取状态 0：全部 1：已读 2：未读
     */
    public static final int MESSAGE_READ_FLAG_UNREAD = 2;

    /**
     * 根据appId查询当前应用的配置
     *
     * @param appCode 应用注册时生成的appId
     * @return
     */
    @Override
    public List<ThirdMessageConfig> queryConfigByAppCode(String appCode, String appId) {
        return thirdSystemMessageMapper.queryConfigByAppCode(appCode, appId);
    }

    /**
     * 查询异构系统的消息配置
     *
     * @param queryThirdMessageConfigReq 查询参数
     * @return 返回
     */
    @Override
    public PageInfoResp<ThirdMessageConfig> queryMessageConfigPage(QueryThirdMessageConfigReq queryThirdMessageConfigReq) {
        QueryWrapper<ThirdMessageConfig> condition = new QueryWrapper<>();
        if (StringUtils.isNotBlank(queryThirdMessageConfigReq.getAppPrimaryId())) {
            condition.eq("app_primary_id", queryThirdMessageConfigReq.getAppPrimaryId());
        }
        if (queryThirdMessageConfigReq.getValidStatus() != null) {
            condition.eq("valid_status", queryThirdMessageConfigReq.getValidStatus());
        }
        if (queryThirdMessageConfigReq.getAppSource() != null) {
            condition.eq("app_source", queryThirdMessageConfigReq.getAppSource());
        }
        if (queryThirdMessageConfigReq.getDataGetType() != null) {
            condition.eq("data_get_type", queryThirdMessageConfigReq.getDataGetType());
        }
        condition.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId()).orderByDesc("modify_time");

        // 查询
        PageInfoResp<ThirdMessageConfig> pageInfoResp = new PageInfoResp<>();
        Page<ThirdMessageConfig> pageResult = thirdSystemMessageMapper.selectPage(new Page<>(queryThirdMessageConfigReq.getPageNum(), queryThirdMessageConfigReq.getPageSize()), condition);
        if (pageResult == null || CollectionUtils.isEmpty(pageResult.getRecords())) {
            return pageInfoResp;
        }

        // 获取应用名称
        List<ThirdMessageConfig> messageConfigList = pageResult.getRecords();
        // 组装应用名称
        packageAppCodeName(messageConfigList);

        pageInfoResp.setTotalPages(Integer.parseInt(String.valueOf(pageResult.getPages())));
        pageInfoResp.setPageNo(queryThirdMessageConfigReq.getPageNum());
        pageInfoResp.setPageSize(queryThirdMessageConfigReq.getPageSize());
        pageInfoResp.setTotalRecords(Integer.parseInt(String.valueOf(pageResult.getTotal())));
        pageInfoResp.setList(messageConfigList);
        return pageInfoResp;
    }

    /**
     * 组装应用名称
     *
     * @param messageConfigList 应用配置列表
     */
    public void packageAppCodeName(List<ThirdMessageConfig> messageConfigList) {
        // 根据源应用来源（ 0：CS、1：BS、2：平台预设、3：雅典娜应用）拆分数据
        // CS应用
        List<ThirdMessageConfig> csList = messageConfigList.stream().filter(item -> ApplicationTypeEnum.CS_APPLICATION.getType().equals(item.getAppSource())).collect(Collectors.toList());
        Map<Long, ErpSsoInfo> erpInfoMap = getErpInfoMap(csList);
        // BS应用
        List<ThirdMessageConfig> bsList = messageConfigList.stream().filter(item -> ApplicationTypeEnum.BS_APPLICATION.getType().equals(item.getAppSource())).collect(Collectors.toList());
        Map<Long, ThirdSsoInfo> thirdSsoInfoMap = getSsoInfoMap(bsList);
        // 预设应用
        Map<Long, PreinstalledApplication> preInfoMap = getPreInfoMap();
        for (ThirdMessageConfig messageConfig : messageConfigList) {
            String appName = "";
            // cs
            if (ApplicationTypeEnum.CS_APPLICATION.getType().equals(messageConfig.getAppSource())) {
                ErpSsoInfo erpSsoInfo = erpInfoMap.get(Long.parseLong(messageConfig.getAppPrimaryId()));
                if (null != erpSsoInfo) {
                    appName = erpSsoInfo.getName();
                }
            }
            // bs
            if (ApplicationTypeEnum.BS_APPLICATION.getType().equals(messageConfig.getAppSource())) {
                ThirdSsoInfo thirdSsoInfo = thirdSsoInfoMap.get(Long.parseLong(messageConfig.getAppPrimaryId()));
                if (thirdSsoInfo != null) {
                    appName = thirdSsoInfo.getAppName();
                }
            }
            // 预设
            if (ApplicationTypeEnum.PRESET_APPLICATION.getType().equals(messageConfig.getAppSource())) {
                PreinstalledApplication preinstalledApplication = preInfoMap.get(Long.parseLong(messageConfig.getAppPrimaryId()));
                if (null != preinstalledApplication) {
                    appName = preinstalledApplication.getApplicationName();
                }
            }
            messageConfig.setAppName(appName);
        }
    }

    /**
     * 查询平台预设应用
     *
     * @param
     * @return
     */
    public Map<Long, PreinstalledApplication> getPreInfoMap() {
        List<PreinstalledApplication> preinstalledAppList = preinstalledApplicationMapper.selectList(new QueryWrapper<PreinstalledApplication>());
        Map<Long, PreinstalledApplication> preInfoMap = preinstalledAppList.stream().collect(Collectors.toMap(PreinstalledApplication::getId, Function.identity(), (a, b) -> a));

        return preInfoMap;
    }

    /**
     * 查询CS应用
     *
     * @param csList
     * @return
     */
    public Map<Long, ErpSsoInfo> getErpInfoMap(List<ThirdMessageConfig> csList) {
        Map<Long, ErpSsoInfo> erpInfoMap = new HashMap<>();
        if (CollectionUtils.isEmpty(csList)) {
            return erpInfoMap;
        }
        List<String> thirdFolderIdList = csList.stream().map(ThirdMessageConfig::getAppPrimaryId).collect(Collectors.toList());
        QuerySsoListReq querySsoListReq = new QuerySsoListReq();
        List<Long> idList = thirdFolderIdList.stream().map(Long::parseLong).collect(Collectors.toList());
        querySsoListReq.setIdList(idList);
        List<ErpSsoInfo> erpInfoList = erpSsoInfoMapper.querySsoListByScope(querySsoListReq);
        erpInfoMap = erpInfoList.stream().collect(Collectors.toMap(ErpSsoInfo::getId, Function.identity(), (a, b) -> a));
        return erpInfoMap;
    }

    /**
     * 查询BS应用
     *
     * @param bsList
     * @return
     */
    public Map<Long, ThirdSsoInfo> getSsoInfoMap(List<ThirdMessageConfig> bsList) {
        Map<Long, ThirdSsoInfo> thirdSsoInfoMap = new HashMap<>();
        if (CollectionUtils.isEmpty(bsList)) {
            return thirdSsoInfoMap;
        }
        List<String> thirdFolderIdList = bsList.stream().map(ThirdMessageConfig::getAppPrimaryId).collect(Collectors.toList());
        QuerySsoListReq querySsoListReq = new QuerySsoListReq();
        List<Long> idList = thirdFolderIdList.stream().map(Long::parseLong).collect(Collectors.toList());
        querySsoListReq.setIdList(idList);
        List<ThirdSsoInfo> thirdSsoInfoList = thirdSsoInfoMapper.queryThirdListByScope(querySsoListReq);
        thirdSsoInfoMap = thirdSsoInfoList.stream().collect(Collectors.toMap(ThirdSsoInfo::getId, Function.identity(), (a, b) -> a));
        return thirdSsoInfoMap;
    }

    /**
     * 校验应用是否已经存在
     *
     * @param saveThirdMessageConfigReq 请求入参
     * @return
     */
    @Override
    public boolean validMessageConfigExist(SaveThirdMessageConfigReq saveThirdMessageConfigReq) {
        QueryWrapper<ThirdMessageConfig> condition = new QueryWrapper<>();
        condition.eq("app_source", saveThirdMessageConfigReq.getAppSource());
        condition.eq("app_primary_id", saveThirdMessageConfigReq.getAppPrimaryId());
        if (saveThirdMessageConfigReq.getId() != null) {
            condition.ne("id", saveThirdMessageConfigReq.getId());
        }
        ThirdMessageConfig thirdMessageConfig = thirdSystemMessageMapper.selectOne(condition);
        return thirdMessageConfig != null;
    }

    /**
     * 新增异构系统的消息配置
     *
     * @param saveThirdMessageConfigReq 请求参数
     */
    @Override
    public ThirdMessageConfig saveApplication(SaveThirdMessageConfigReq saveThirdMessageConfigReq) {
        if (saveThirdMessageConfigReq.getId() == null) {
            ThirdMessageConfig thirdMessageConfig = converter.convert(saveThirdMessageConfigReq, ThirdMessageConfig.class);
            thirdMessageConfig.setValidStatus(saveThirdMessageConfigReq.getValidStatus());
            // 新增系统的消息配置
            thirdSystemMessageMapper.insert(thirdMessageConfig);
            return thirdMessageConfig;
        } else {
            ThirdMessageConfig thirdMessageConfig = converter.convert(saveThirdMessageConfigReq, ThirdMessageConfig.class);
            thirdMessageConfig.setValidStatus(saveThirdMessageConfigReq.getValidStatus());
            thirdSystemMessageMapper.updateById(thirdMessageConfig);
            return thirdMessageConfig;
        }
    }

    /**
     * 接收异构系统的消息推送 restful接口
     *
     * @param receiveThirdMessageReq 请求入参
     */
    @Override
    public void receiveMessage(ReceiveThirdMessageReq receiveThirdMessageReq) {
        Tuple3<String, String, ApplicationTypeEnum> tuple3 = thirdNewsAnnouncementService.checkAppCode(receiveThirdMessageReq.getAppId(), receiveThirdMessageReq.getAppCode());
        String tenantId = tuple3._2;
        if (StringUtils.isBlank(tenantId)) {
            log.warn("接收异构系统的消息推送租户id为空：{}", tuple3._1);
            return;
        }
        QueryWrapper<ThirdMessageInfo> condition = new QueryWrapper<ThirdMessageInfo>()
                .eq("message_app_id", receiveThirdMessageReq.getAppId()).eq("tenant_id", tenantId);
        List<ThirdMessageInfo> thirdMessageInfoList = thirdMessageInfoService.list(condition);
        Map<String, ThirdMessageInfo> existMessageMap = CollectionUtils.isNotEmpty(thirdMessageInfoList) ?
                thirdMessageInfoList.stream().collect(Collectors.toMap(ThirdMessageInfo::getMessageId, a -> a, (k1, k2) -> k1)) : Maps.newHashMap();

        // 是否全员可见标识
        boolean allSeeFlag = Boolean.FALSE;
        // 三方应用用户列表
        List<String> verifyUserIdList = Lists.newArrayList();
        List<ThirdMessageInfo> messageList = Lists.newArrayList();
        List<ThirdMessageInfo> newMessageInfoList = new ArrayList<>();
        for (ThirdMessageInfo info : receiveThirdMessageReq.getMessageList()) {
            // 更新的消息,更新消息的可查看用户列表
            if (existMessageMap.containsKey(info.getMessageId())) {
                ThirdMessageInfo existInfo = existMessageMap.get(info.getMessageId());
                info.setId(existInfo.getId());
                info.setModifyTime(DateUtils.getNowTime(null));
                // 全员可见存储为"all"
                if (Constants.MESSAGE_ALL_STAFF_FLAG.equals(info.getMessageAllStaffRead())) {
                    info.setMessageOwner(Constants.ALL_STAFF_CAN_SEE_FLAG);
                    allSeeFlag = Boolean.TRUE;
                } else {
                    info.setMessageOwner(existInfo.getMessageOwner().equals(Constants.ALL_STAFF_CAN_SEE_FLAG) ? info.getMessageOwner() : existInfo.getMessageOwner() + "," + info.getMessageOwner());
                    verifyUserIdList.addAll(Arrays.asList(info.getMessageOwner().split(",")));
                }
            } else { // 新增的消息
                info.setMessageAppCode(receiveThirdMessageReq.getAppCode());
                info.setMessageAppId(receiveThirdMessageReq.getAppId());
                info.setCreateTime(DateUtils.getNowTime(null));
                info.setModifyTime(DateUtils.getNowTime(null));
                if (Constants.MESSAGE_ALL_STAFF_FLAG.equals(info.getMessageAllStaffRead())) {
                    info.setMessageOwner(Constants.ALL_STAFF_CAN_SEE_FLAG);
                    allSeeFlag = Boolean.TRUE;
                } else {
                    verifyUserIdList.addAll(Arrays.asList(info.getMessageOwner().split(",")));
                    newMessageInfoList.add(info);
                }
            }
            info.setMessageTitle(info.getMessageTitle());
            info.setMessageContent(info.getMessageContent());
            info.setMessageUrl(info.getMessageUrl());
            info.setTenantId(tenantId);
            messageList.add(info);
        }
        if (CollectionUtils.isNotEmpty(messageList)) {
            thirdMessageInfoService.saveOrUpdateBatch(messageList);
        }

        // mqtt通知前端
        Integer sideType = messageList.get(0).getSideType();
        if (Boolean.TRUE.equals(allSeeFlag)) {
            MessageDO payload = new MessageDO();
            payload.setTenantId(tenantId);
            payload.setAllStaffNoticeFlag(Boolean.TRUE);
            payload.setType(2);
            payload.setAppId(receiveThirdMessageReq.getAppCode());
            if (Objects.equals(0,sideType)) {
                messageSendService.sendToClient(null, payload, Constants.THIRD_MESSAGE_NOTICE_USER_TOPIC);
            }
            if (Objects.equals(1,sideType)){
                messageSendService.sendToClient(null, payload, Constants.THIRD_MESSAGE_MOBILE_NOTICE_USER_TOPIC);
            }
            if (Objects.equals(2,sideType)){
                messageSendService.sendToClient(null, payload, Constants.THIRD_MESSAGE_NOTICE_USER_TOPIC);
                messageSendService.sendToClient(null, payload, Constants.THIRD_MESSAGE_MOBILE_NOTICE_USER_TOPIC);
            }
        } else {
            List<String> distinctList = verifyUserIdList.stream().distinct().collect(Collectors.toList());
            for (String verifyUserId : distinctList) {
                MessageDO payload = new MessageDO();
                payload.setTenantId(tenantId);
                payload.setUserId(verifyUserId);
                payload.setType(2);
                payload.setAppId(receiveThirdMessageReq.getAppCode());
                if (Objects.equals(0,sideType)) {
                    messageSendService.sendToClient(verifyUserId, payload, Constants.THIRD_MESSAGE_NOTICE_USER_TOPIC);
                }
                if (Objects.equals(1,sideType)) {
                    messageSendService.sendToClient(verifyUserId, payload, Constants.THIRD_MESSAGE_MOBILE_NOTICE_USER_TOPIC);
                }
                if (Objects.equals(2,sideType)){
                    messageSendService.sendToClient(verifyUserId, payload, Constants.THIRD_MESSAGE_NOTICE_USER_TOPIC);
                    messageSendService.sendToClient(verifyUserId, payload, Constants.THIRD_MESSAGE_MOBILE_NOTICE_USER_TOPIC);
                }
            }
        }

        // 发送企微事件
        String sendChannel = receiveThirdMessageReq.getSendChannel();
        if (StringUtils.isNotBlank(sendChannel) && sendChannel.contains(Constants.MessageChannelEnum.WECOM.getFlag()) && CollectionUtils.isNotEmpty(newMessageInfoList)) {
            // 三方应用调用该接口，无法得知token，所以需要获取集成token
            String userToken = iamService.queryIntegrationUserToken(tenantId);
            if (StringUtils.isNotBlank(userToken)) {
                ThirdMessageWecomEventDTO thirdMessageWecomEventDTO = new ThirdMessageWecomEventDTO();
                thirdMessageWecomEventDTO.setNewMessageInfoList(newMessageInfoList);
                thirdMessageWecomEventDTO.setTenantId(tenantId);
                thirdMessageWecomEventDTO.setUserToken(userToken);
                ThirdMessageWecomEvent thirdMessageWecomEvent = new ThirdMessageWecomEvent(this, thirdMessageWecomEventDTO);
                eventPublisher.publishEvent(thirdMessageWecomEvent);

            }
        }
    }

    /**
     * 查询消息的未读数量，在首页小铃铛展示
     */
    @Override
    public Long queryUnreadCount(Integer clientType) {
        QueryThirdMessageListReq queryThirdMessageListReq = new QueryThirdMessageListReq();
        queryThirdMessageListReq.setClientType(clientType);
        List<ThirdMessageInfo> messageInfoList = queryMessageList(queryThirdMessageListReq, null);
        if (CollectionUtils.isEmpty(messageInfoList)) {
            return 0L;
        }
        return messageInfoList.stream().filter(x -> x.getIsRed().equals(true)).count();
    }

    /**
     * 查询租户下的账套信息
     *
     * @param verifyUserList 参数
     */
    private void queryAccountList(List<AppUserDTO> verifyUserList) {
        // 查询当前租户下的账套信息
        List<AccountDTO> accountDTOList = iamService.queryTenantAccountMappingList(null, null, null);
        if (CollectionUtils.isEmpty(accountDTOList)) {
            return;
        }
        Map<String, List<AccountDTO>> appCodeMap = accountDTOList.stream().collect(Collectors.groupingBy(AccountDTO::getAppId));
        for (AppUserDTO appUserDTO : verifyUserList) {
            if (CollectionUtils.isNotEmpty(appCodeMap.get(appUserDTO.getAppCode()))) {
                List<AccountDTO> accountList = appCodeMap.get(appUserDTO.getAppCode());
                List<String> accountIdList = Lists.newArrayList();
                accountList.forEach(x -> {
                    if (StringUtils.isNotBlank(x.getAccount())) {
                        accountIdList.add(x.getAccount());
                    } else if (StringUtils.isNotBlank(x.getCompanyId())) {
                        accountIdList.add(x.getCompanyId());
                    }
                });
                appUserDTO.setAccountIdList(accountIdList);
            }
        }
    }

    /**
     * 查询三方用户id
     *
     * @param thirdMessageInfoList 消息列表
     * @return
     */
    private List<AppUserDTO> queryVerifyUseridList(List<ThirdMessageInfo> thirdMessageInfoList) {
        thirdMessageInfoList = thirdMessageInfoList.stream()
                .collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(ThirdMessageInfo::getMessageAppCode))), ArrayList::new));
        List<AppUserDTO> appVerifyUserIdList = Lists.newArrayList();
        for (ThirdMessageInfo thirdMessageInfo : thirdMessageInfoList) {
            String verifyUserId = iamService.queryMappingEmpId(thirdMessageInfo.getMessageAppCode());
            if (StringUtils.isNotBlank(verifyUserId)) {
                AppUserDTO appUserDTO = new AppUserDTO();
                appUserDTO.setAppCode(thirdMessageInfo.getMessageAppCode());
                appUserDTO.setAppId(thirdMessageInfo.getMessageAppId());
                appUserDTO.setVerifyUserId(verifyUserId);
                appVerifyUserIdList.add(appUserDTO);
            }
        }
        return appVerifyUserIdList;
    }

    /**
     * 查询消息列表
     *
     * @param queryThirdMessageListReq 请求入参
     * @return 返回对象
     */
    @Override
    public PageInfoResp<ThirdMessageInfo> queryMessageListPage(QueryThirdMessageListReq queryThirdMessageListReq) {
        PageInfoResp<ThirdMessageInfo> pageInfoResp = new PageInfoResp<>();
        // 查询消息列表
        // 查询当前租户的sso配置应用，过滤出当前租户下的应用
        List<AppLinkDTO> appLinkDTOList = appLinkService.queryManageListSync(AppAuthContextHolder.getContext().getAuthoredUser());
        List<ThirdMessageInfo> messageInfoList = queryMessageList(queryThirdMessageListReq, appLinkDTOList);
        if (CollectionUtils.isEmpty(messageInfoList)) {
            return pageInfoResp;
        }
        // 根据消息状态过滤
        if (ObjectUtils.isNotEmpty(queryThirdMessageListReq.getReadFlag()) && !queryThirdMessageListReq.getReadFlag().equals(MESSAGE_READ_FLAG_ALL)) {
            if (queryThirdMessageListReq.getReadFlag().equals(MESSAGE_READ_FLAG_READ)) {
                messageInfoList = messageInfoList.stream().filter(x -> x.getIsRed().equals(false)).collect(Collectors.toList());
            }
            if (queryThirdMessageListReq.getReadFlag().equals(MESSAGE_READ_FLAG_UNREAD)) {
                messageInfoList = messageInfoList.stream().filter(x -> x.getIsRed().equals(true)).collect(Collectors.toList());
            }
        }

        // 排序
        messageInfoList = messageInfoList.stream().sorted(Comparator.comparing(ThirdMessageInfo::getMessagePublishTime).reversed()).collect(Collectors.toList());
        // 计算分页并返回
        int fromIndex = (queryThirdMessageListReq.getPageNum() - 1) * queryThirdMessageListReq.getPageSize();
        if (fromIndex >= messageInfoList.size() || fromIndex < 0) {
            return pageInfoResp;
        }
        int toIndex = fromIndex + queryThirdMessageListReq.getPageSize();
        if (toIndex > messageInfoList.size()) {
            toIndex = messageInfoList.size();
        }
        List<ThirdMessageInfo> messageInfoRespList = messageInfoList.subList(fromIndex, toIndex);

        // 消息类型
        List<ThirdMessageType> messageTypeList = thirdMessageTypeMapper.selectList(Wrappers.emptyWrapper());
        Map<String, String> messageTypeMap = messageTypeList.stream().collect(Collectors.toMap(ThirdMessageType::getMessageTypeCode, ThirdMessageType::getMessageTypeName, (a, b) -> a));

        // 获取应用appId，callBackUrl，协议类型，用于前端获取授权码
        for (ThirdMessageInfo messageInfo : messageInfoRespList) {
            messageInfo.setMessageTypeName(messageTypeMap.get(messageInfo.getMessageTypeCode()));
            // 点击查看详情SSO参数
            for (AppLinkDTO appLinkDTO : appLinkDTOList) {
                if (messageInfo.getMessageAppCode().equals(appLinkDTO.getCode())) {
                    messageInfo.setMessageAppName(appLinkDTO.getName());
                    messageInfo.setCallBackUrl(appLinkDTO.getCallBackUrl());
                    messageInfo.setApplicationAppId(appLinkDTO.getApplicationAppId());
                    messageInfo.setProtocolType(appLinkDTO.getProtocolType());
                    messageInfo.setAppToken(appLinkDTO.getAppToken());
                    if (Objects.isNull(messageInfo.getSideType())){
                        messageInfo.setSideType(0);
                    }
                    if (Objects.equals(0,messageInfo.getSideType())) {
                        messageInfo.setMessageUrl(appLinkService.parseJumpUrl(messageInfo.getMessageUrl(), appLinkDTO));
                    }
                    if (Objects.equals(1,messageInfo.getSideType())){
                        messageInfo.setMobileMessageUrl(appLinkService.parseJumpUrl(messageInfo.getMobileMessageUrl(), appLinkDTO));
                    }
                    if (Objects.equals(2,messageInfo.getSideType())){
                        messageInfo.setMessageUrl(appLinkService.parseJumpUrl(messageInfo.getMessageUrl(), appLinkDTO));
                        messageInfo.setMobileMessageUrl(appLinkService.parseJumpUrl(messageInfo.getMobileMessageUrl(), appLinkDTO));
                    }
                    break;
                }
            }
        }

        int total = messageInfoList.size();
        int totalPages = total % queryThirdMessageListReq.getPageSize() == 0 ? total / queryThirdMessageListReq.getPageSize()
                : total / queryThirdMessageListReq.getPageSize() + 1;
        pageInfoResp.setTotalPages(totalPages);
        pageInfoResp.setPageNo(queryThirdMessageListReq.getPageNum());
        pageInfoResp.setPageSize(queryThirdMessageListReq.getPageSize());
        pageInfoResp.setTotalRecords(total);
        pageInfoResp.setList(messageInfoRespList);
        return pageInfoResp;
    }


    /**
     * 条件查询登录用户的三方消息列表
     *
     * @param queryThirdMessageListReq
     * @return 用户的全量三方消息
     */
    public ThirdMessageListResp listMessage(QueryThirdMessageListReq queryThirdMessageListReq) {
        ThirdMessageListResp listResp = new ThirdMessageListResp();
        List<AppLinkDTO> appLinkDTOList = new ArrayList<>();
        List<AppUserDTO> verifyUserList = new ArrayList<>();
        List<ThirdMessageInfo> messageInfoList = Lists.newArrayList();
        LambdaQueryWrapper<ThirdMessageInfo> condition = new LambdaQueryWrapper<>();
        condition.eq(ThirdMessageInfo::getTenantId, Utils.getTenantId());
        if (StringUtils.isNotEmpty(queryThirdMessageListReq.getStartTimeStr())) {
            condition.ge(ThirdMessageInfo::getMessagePublishTime, DateUtils.StringToDate(queryThirdMessageListReq.getStartTimeStr() + " 00:00:00"));
        }
        if (StringUtils.isNotEmpty(queryThirdMessageListReq.getEndTimeStr())) {
            condition.le(ThirdMessageInfo::getMessagePublishTime, DateUtils.StringToDate(queryThirdMessageListReq.getEndTimeStr() + " 23:59:59"));
        }
        if (0 == queryThirdMessageListReq.getClientType()){
            List list = new ArrayList();
            list.add(0);
            list.add(2);
            condition.in(ThirdMessageInfo::getSideType,list);
        }
        if (1 == queryThirdMessageListReq.getClientType()){
            List list = new ArrayList();
            list.add(1);
            list.add(2);
            condition.in(ThirdMessageInfo::getSideType,list);
        }
        // 查询当前租户消息列表
        List<ThirdMessageInfo> thirdMessageInfoList = thirdMessageInfoService.list(condition);
        if (CollectionUtils.isEmpty(thirdMessageInfoList)) {
            listResp.setAppLinkDTOList(appLinkDTOList);
            listResp.setVerifyUserList(verifyUserList);
            listResp.setMessageInfoList(messageInfoList);
            return listResp;
        }
        // 查询当前租户的sso配置应用，过滤出当前租户下的应用
        appLinkDTOList = appLinkService.queryManageListSync(AppAuthContextHolder.getContext().getAuthoredUser());
        List<String> appIdList = appLinkDTOList.stream().map(AppLinkDTO::getApplicationAppId).filter(StringUtils::isNotBlank).collect(Collectors.toList());
        thirdMessageInfoList = thirdMessageInfoList.stream().filter(x -> appIdList.contains(x.getMessageAppId())).collect(Collectors.toList());
        if (StringUtils.isNotBlank(queryThirdMessageListReq.getAppId())) {
            thirdMessageInfoList = thirdMessageInfoList.stream().filter(x -> queryThirdMessageListReq.getAppId().equals(x.getMessageAppId())).collect(Collectors.toList());
        }

        // 查询当前鼎捷云用户id在各个应用下归户的三方用户id
        verifyUserList = queryVerifyUseridList(thirdMessageInfoList);
        if (CollectionUtils.isEmpty(verifyUserList)) {
            listResp.setAppLinkDTOList(appLinkDTOList);
            listResp.setVerifyUserList(verifyUserList);
            listResp.setMessageInfoList(messageInfoList);
            return listResp;
        }

        /*// 查询租户下映射的账套信息
        queryAccountList(verifyUserList);*/

        // 过滤符合条件的消息列表
        /*for (AppUserDTO appUserDTO : verifyUserList) {
            for (ThirdMessageInfo x : thirdMessageInfoList) {
                // 存在账套
                if (CollectionUtils.isNotEmpty(appUserDTO.getAccountIdList())) {
                    if (appUserDTO.getAppId().equals(x.getMessageAppId())
                            && Arrays.asList(x.getMessageOwner().split(",")).contains(appUserDTO.getVerifyUserId())
                            && appUserDTO.getAccountIdList().contains(x.getAccountId())) {
                        messageInfoList.add(x);
                    }
                } else {
                    if (x.getMessageAppId().equals(appUserDTO.getAppId())
                            && Arrays.asList(x.getMessageOwner().split(",")).contains(appUserDTO.getVerifyUserId())) {
                        messageInfoList.add(x);
                    }
                }
            }
        }*/
        for (AppUserDTO appUserDTO : verifyUserList) {
            for (ThirdMessageInfo x : thirdMessageInfoList) {
                if (x.getMessageAppId().equals(appUserDTO.getAppId())
                        && (Arrays.asList(x.getMessageOwner().split(",")).contains(appUserDTO.getVerifyUserId())
                        || Arrays.asList(x.getMessageOwner().split(",")).contains(ALL_STAFF_CAN_SEE_FLAG))) {
                    messageInfoList.add(x);
                }
            }
        }
        listResp.setAppLinkDTOList(appLinkDTOList);
        listResp.setVerifyUserList(verifyUserList);
        listResp.setMessageInfoList(messageInfoList);
        return listResp;
    }

    /**
     * 读取消息，记录为已读消息
     *
     * @param readThirdMessageReq 请求入参
     */
    public void readMessage(ReadThirdMessageReq readThirdMessageReq) {
        QueryWrapper<ThirdMessageReadInfo> readCondition = new QueryWrapper<>();
        readCondition.eq("message_app_code", readThirdMessageReq.getMessageAppCode());
        readCondition.eq("message_app_id", readThirdMessageReq.getMessageAppId());
        readCondition.eq("message_id", readThirdMessageReq.getMessageId());
        readCondition.eq("read_user", AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
        readCondition.eq("tenant_id", Utils.getTenantId());
        List<ThirdMessageReadInfo> readInfos = thirdMessageReadInfoService.getBaseMapper().selectList(readCondition);
        if (CollectionUtils.isEmpty(readInfos)) {
            String verifyUserId = iamService.queryMappingEmpId(readThirdMessageReq.getMessageAppCode());
            ThirdMessageReadInfo readInfo = new ThirdMessageReadInfo();
            readInfo = new ThirdMessageReadInfo();
            readInfo.setMessageAppCode(readThirdMessageReq.getMessageAppCode());
            readInfo.setMessageAppId(readThirdMessageReq.getMessageAppId());
            readInfo.setMessageId(readThirdMessageReq.getMessageId());
            readInfo.setReadUser(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
            readInfo.setReadSourceUser(verifyUserId);
            readInfo.setCreateTime(DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER));
            readInfo.setTenantId(Utils.getTenantId());
            thirdMessageReadInfoService.save(readInfo);
        }
    }

    /**
     * 删除应用的消息配置
     *
     * @param deleteMessageConfigReq 请求参数
     */
    @Override
    public void deleteBatchApp(DeleteMessageConfigReq deleteMessageConfigReq) {
        if (CollectionUtils.isNotEmpty(deleteMessageConfigReq.getIdList())) {
            thirdSystemMessageMapper.deleteBatchIds(deleteMessageConfigReq.getIdList());
        }
        if (CollectionUtils.isNotEmpty(deleteMessageConfigReq.getPrimaryIdList())) {
            QueryWrapper<ThirdMessageConfig> condition = new QueryWrapper<>();
            condition.in("app_primary_id", deleteMessageConfigReq.getPrimaryIdList());
            thirdSystemMessageMapper.delete(condition);
        }
    }

    @Override
    public ResponseEntity<?> syncMessageData() {

        String tenantsid = Utils.getTenantId();

        // 已启用的主动获取消息的异构消息配置信息
        QueryWrapper<ThirdMessageConfig> condition = new QueryWrapper<>();
        condition.eq("data_get_type", 2);
        condition.eq("valid_status", 1);
        List<ThirdMessageConfig> thirdMessageConfigList = thirdSystemMessageMapper.selectList(condition);

        if (CollectionUtils.isEmpty(thirdMessageConfigList)) {
            return ResponseEntityWrapper.wrapperOk(messageUtils.getMessage("error.message.third.data.updated"));
        }

        RedisLock redisLock = new RedisLock(thirdMessageConfigList.get(thirdMessageConfigList.size() - 1).getAppPrimaryId() + tenantsid);

        try {
            if (!redisLock.isLocked()) {

                redisLock.lock();
                for (ThirdMessageConfig config : thirdMessageConfigList) {

                    List<Map<String, Object>> messageList = new ArrayList<>();
                    Integer appAccessModel = config.getAppAccessModel();
                    String messageAppCode = config.getAppCode();
                    String appId = config.getApplicationAppId();

                    // 混合云
                    if (appAccessModel == 0) {

                        String middleSystemName = config.getMiddleSystemName();
                        String middleSystemUid = config.getMiddleSystemUid();
                        Map<String, String> extHeader = new HashMap<>();
                        extHeader.put("digi-userToken", Utils.getUserToken());
                        extHeader.put("digi-appToken", envProperties.getAppToken());
                        // 通过ESP查询三方系统的消息数据
                        for (int i = 1; ; i++) {
                            Map<String, Object> parameter = new HashMap<>();
                            parameter.put("page_no", i);
                            parameter.put("page_size", 100);
                            parameter.put("tenant_id", tenantsid);
                            try {
                                Map<String, Object> result = espService.queryByEsp(middleSystemName,
                                        middleSystemUid, EAIServiceNameEnum.MESSAGE_PULL.getServiceName(), extHeader, parameter, null,
                                        null);
                                // 没有显示字段时，返回异常
                                if (Objects.isNull(result.get("message_list"))) {
                                    System.out.println("syncMessageData 混合云未查询到消息数据");
                                    break;
                                }
                                messageList = (List<Map<String, Object>>) result.get("message_list");
                                System.out.println("syncMessageData 混合云消息数据" + messageList);
                                // 插入消息数据
                                if (CollectionUtils.isNotEmpty(messageList)) {
                                    handleMessageData(messageAppCode, appId, messageList);
                                }
                                if (!(Boolean) result.get("has_next")) {
                                    break;
                                }
                            } catch (Exception e) {
                                break;
                            }
                        }
                    }
                    // 非混合云
                    if (appAccessModel == 1) {
                        for (int i = 1; ; i++) {
                            String domain = config.getDomain();
                            Map<String, Object> bodyMap = new HashMap<>();
                            bodyMap.put("page_no", i);
                            bodyMap.put("page_size", 100);
                            bodyMap.put("tenant_id", tenantsid);
                            try {
                                Map<String, Object> jobData = tripartiteService.queryThirdData(domain + Constants.QUERY_MESSAGEJOB_LIST_URL, "", bodyMap);
                                if (null == jobData) {
                                    System.out.println("syncMessageData 非混合云未查询到消息数据");
                                    break;
                                }
                                messageList = (List<Map<String, Object>>) jobData.get("message_list");
                                System.out.println("syncMessageData 非混合云消息数据" + messageList);
                                if (CollectionUtils.isNotEmpty(messageList)) {
                                    // 插入消息数据
                                    handleMessageData(messageAppCode, appId, messageList);
                                }
                                if (!(Boolean) jobData.get("has_next")) {
                                    break;
                                }
                            } catch (Exception e) {
                                break;
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            System.out.println(e.getMessage());
        } finally {
            redisLock.unlock();
        }
        return ResponseEntityWrapper.wrapperOk(messageUtils.getMessage("error.message.third.data.updated"));
    }

    @Override
    public ResponseEntity<BaseResultDTO<Integer>> readAllMessage(QueryThirdMessageListReq queryThirdMessageListReq) {
        // 查询当前租户消息列表
        ThirdMessageListResp listResp = listMessage(queryThirdMessageListReq);
        List<ThirdMessageInfo> messageInfoList = listResp.getMessageInfoList();
        List<AppUserDTO> verifyUserList = listResp.getVerifyUserList();

        //没有三方消息，所以也就没有需要标记已读的消息
        if (CollectionUtils.isEmpty(messageInfoList)) {
            return ResponseEntityWrapperUtil.wrapperOk(0);
        }
        Map<String, String> verifyUserMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(verifyUserList)) {
            verifyUserMap = verifyUserList.stream().collect(Collectors.toMap(value -> value.getAppCode() + "-" + value.getAppId(), value -> value.getVerifyUserId(), (key1, key2) -> key1));
        }

        // 取出消息ID
        List<String> messageIdList = messageInfoList.stream().map(ThirdMessageInfo::getMessageId).distinct().collect(Collectors.toList());
        // 保存已读信息
        QueryWrapper<ThirdMessageReadInfo> readCondition = new QueryWrapper<>();
        if (CollectionUtils.isNotEmpty(messageIdList)) {
            readCondition.in("message_id", messageIdList);
        }
        readCondition.eq("read_user", Utils.getUserId());
        readCondition.eq("tenant_id", Utils.getTenantId());
        List<ThirdMessageReadInfo> readInfoList = thirdMessageReadInfoService.getBaseMapper().selectList(readCondition);
        Map<String, ThirdMessageReadInfo> readInfoMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(readInfoList)) {
            readInfoMap = readInfoList.stream().collect(Collectors.toMap(value -> value.getMessageAppCode() + "-" + value.getMessageAppId() + "-" + value.getMessageId() + "-" + value.getReadUser(), value -> value, (key1, key2) -> key1));
        }

        //默认暂无消息需要已读
        int i = 0;
        for (ThirdMessageInfo messageInfo : messageInfoList) {
            String readKey = messageInfo.getMessageAppCode() + "-" + messageInfo.getMessageAppId() + "-" + messageInfo.getMessageId() + "-" + Utils.getUserId();
            ThirdMessageReadInfo readInfo = readInfoMap.get(readKey);
            if (null != readInfo) {
                continue;
            }
            String verifyUserKey = messageInfo.getMessageAppCode() + "-" + messageInfo.getMessageAppId();
            String verifyUserId = verifyUserMap.get(verifyUserKey);
            readInfo = new ThirdMessageReadInfo();
            readInfo.setMessageAppCode(messageInfo.getMessageAppCode());
            readInfo.setMessageAppId(messageInfo.getMessageAppId());
            readInfo.setMessageId(messageInfo.getMessageId());
            readInfo.setReadUser(Utils.getUserId());
            readInfo.setReadSourceUser(verifyUserId);
            readInfo.setCreateTime(DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER));
            readInfo.setTenantId(Utils.getTenantId());
            thirdMessageReadInfoService.save(readInfo);
            i++;
        }
        return ResponseEntityWrapperUtil.wrapperOk(i);

    }

    public void handleMessageData(String messageAppCode, String appId, List<Map<String, Object>> messageList) {
        List<ThirdMessageInfo> messageInfos = new ArrayList<>();

        // 查出该应用下所有消息，已有消息id更新
        QueryWrapper<ThirdMessageInfo> condition = new QueryWrapper<ThirdMessageInfo>()
                .eq("message_app_id", appId);
        List<ThirdMessageInfo> thirdMessageInfoList = thirdMessageInfoService.list(condition);
        Map<String, ThirdMessageInfo> existMessageMap = CollectionUtils.isNotEmpty(thirdMessageInfoList) ?
                thirdMessageInfoList.stream().collect(Collectors.toMap(ThirdMessageInfo::getMessageId, a -> a, (k1, k2) -> k1)) : Maps.newHashMap();

        for (Map<String, Object> map : messageList) {
            ThirdMessageInfo messageInfo = new ThirdMessageInfo();
            messageInfo.setMessageAppCode(messageAppCode);
            messageInfo.setMessageAppId(appId);
            messageInfo.setMessageId(map.get("message_id").toString());
            messageInfo.setMessageTitle(map.get("message_title").toString());
            messageInfo.setMessageUrl(map.get("message_url").toString());
            messageInfo.setMessagePublishTime(map.get("message_publish_time").toString());
            messageInfo.setMessageContent(map.get("message_content").toString());
            messageInfo.setMessageOwner(map.get("message_owner").toString());
            messageInfo.setMessageAllStaffRead(Integer.parseInt(map.get("message_all_staff_read").toString()));
            messageInfo.setTenantId(Utils.getTenantId());
            String accountId = MapUtils.getString(map, "account_id");
            if (StringUtils.isNotBlank(accountId)) {
                messageInfo.setAccountId(accountId);
                messageInfo.setAccountName(MapUtils.getString(map, "account_name"));
            }
            // 更新的消息,更新消息的可查看用户列表
            if (existMessageMap.containsKey(messageInfo.getMessageId())) {
                ThirdMessageInfo existInfo = existMessageMap.get(messageInfo.getMessageId());
                messageInfo.setId(existInfo.getId());
                messageInfo.setModifyTime(DateUtils.getNowTime(null));
                // 全员可见存储为"all"
                if (Constants.MESSAGE_ALL_STAFF_FLAG.equals(messageInfo.getMessageAllStaffRead())) {
                    messageInfo.setMessageOwner(Constants.ALL_STAFF_CAN_SEE_FLAG);
                } else {
                    messageInfo.setMessageOwner(existInfo.getMessageOwner().equals("all") ? messageInfo.getMessageOwner() : existInfo.getMessageOwner() + "," + messageInfo.getMessageOwner());
                }
            } else { // 新增的消息
                messageInfo.setCreateTime(DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER));
                if (Constants.MESSAGE_ALL_STAFF_FLAG.equals(messageInfo.getMessageAllStaffRead())) {
                    messageInfo.setMessageOwner(Constants.ALL_STAFF_CAN_SEE_FLAG);
                }
            }

            messageInfos.add(messageInfo);
        }

        if (CollectionUtils.isNotEmpty(messageInfos)) {
            thirdMessageInfoService.saveOrUpdateBatch(messageInfos);
        }
    }

    /**
     * 查询消息所属应用列表
     *
     * @param queryThirdMessageListReq 请求对象
     * @return 返回
     */
    @Override
    public List<MessageAppNumResp> queryMessageAppList(QueryThirdMessageListReq queryThirdMessageListReq) {
        List<MessageAppNumResp> messageAppNumRespList = Lists.newArrayList();
        // 查询消息列表
        List<ThirdMessageInfo> messageInfoList = queryMessageList(queryThirdMessageListReq, null);
        if (CollectionUtils.isEmpty(messageInfoList)) {
            return messageAppNumRespList;
        }

        // 构建返回集合对象
        Map<String, List<ThirdMessageInfo>> appIdMap = messageInfoList.stream().collect(Collectors.groupingBy(ThirdMessageInfo::getMessageAppId));
        for (Map.Entry<String, List<ThirdMessageInfo>> entry : appIdMap.entrySet()) {
            MessageAppNumResp messageAppNumResp = new MessageAppNumResp();
            // appId
            String appId = entry.getKey();
            messageAppNumResp.setAppId(appId);

            // 应用名称
            List<ThirdMessageInfo> list = entry.getValue();
            messageAppNumResp.setAppName(list.get(0).getMessageAppName());

            // 已读数量
            long readNum = list.stream().filter(x -> x.getIsRed().equals(false)).count();
            messageAppNumResp.setReadNum(readNum);

            // 未读数量
            long unReadNum = list.stream().filter(x -> x.getIsRed().equals(true)).count();
            messageAppNumResp.setUnReadNum(unReadNum);

            // 总数
            messageAppNumResp.setSumNum((long) list.size());
            Optional<ThirdMessageInfo> latest = list.stream()
                    .max(Comparator.comparing(ThirdMessageInfo::getModifyTime));
            if (latest.isPresent()) {
                messageAppNumResp.setLatestMessageContent(latest.get().getMessageContent());
                messageAppNumResp.setLatestMessageModifyTime(latest.get().getModifyTime());
            }
            messageAppNumRespList.add(messageAppNumResp);
        }

        // 获取当前用户的应用展示顺序
        LambdaQueryWrapper<ThirdMessageAppSort> sortCondition = new LambdaQueryWrapper<>();
        sortCondition.eq(ThirdMessageAppSort::getUserId, AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
        sortCondition.eq(ThirdMessageAppSort::getTenantId, Utils.getTenantId());
        sortCondition.orderByAsc(ThirdMessageAppSort::getAppSort);
        List<ThirdMessageAppSort> appSortList = thirdMessageAppSortMapper.selectList(sortCondition);
        if (CollectionUtils.isNotEmpty(appSortList)) {
            Map<String, MessageAppNumResp> appIdRespMap = messageAppNumRespList.stream().collect(Collectors.toMap(MessageAppNumResp::getAppId, Function.identity()));
            appSortList.forEach(x -> {
                String appId = x.getAppId();
                MessageAppNumResp messageAppNumResp = appIdRespMap.get(appId);
                if (ObjectUtils.isNotEmpty(messageAppNumResp)) {
                    messageAppNumResp.setAppSort(x.getAppSort());
                }
            });
        }

        // 应用先按排序正序，再按名称正序
        messageAppNumRespList = messageAppNumRespList.stream()
                .sorted(Comparator.comparing(MessageAppNumResp::getAppSort, Comparator.nullsLast(Comparator.naturalOrder()))
                        .thenComparing(MessageAppNumResp::getAppName, Comparator.nullsLast(Comparator.nullsLast(Comparator.naturalOrder()))))
                .collect(Collectors.toList());
        return messageAppNumRespList;
    }

    /**
     * 查询消息分类列表
     *
     * @return 返回
     */
    @Override
    public List<ThirdMessageType> queryMessageType() {
        return thirdMessageTypeMapper.selectList(Wrappers.emptyWrapper());
    }

    /**
     * 保存用户的消息应用展示顺序
     *
     * @param appIdList 应用id列表
     * @return 返回
     */
    @Override
    @Transactional
    public Boolean saveAppSort(List<String> appIdList) {
        // 删除当前应用顺序
        LambdaQueryWrapper<ThirdMessageAppSort> condition = new LambdaQueryWrapper<>();
        condition.eq(ThirdMessageAppSort::getUserId, Utils.getUserId());
        condition.eq(ThirdMessageAppSort::getTenantId, Utils.getTenantId());
        thirdMessageAppSortMapper.delete(condition);

        // 新增应用顺序
        for (int i = 0; i < appIdList.size(); i++) {
            ThirdMessageAppSort thirdMessageAppSort = new ThirdMessageAppSort();
            thirdMessageAppSort.setAppId(appIdList.get(i));
            thirdMessageAppSort.setAppSort(i);
            thirdMessageAppSort.setUserId(Utils.getUserId());
            thirdMessageAppSort.setTenantId(Utils.getTenantId());
            thirdMessageAppSortMapper.insert(thirdMessageAppSort);
        }
        return true;
    }

    /**
     * 查询三方消息列表
     *
     * @param queryThirdMessageListReq 请求对象
     * @param appLinkDTOList           sso配置应用
     * @return 返回
     */
    private List<ThirdMessageInfo> queryMessageList(QueryThirdMessageListReq queryThirdMessageListReq, List<AppLinkDTO> appLinkDTOList) {
        List<ThirdMessageInfo> messageInfoList = Lists.newArrayList();
        LambdaQueryWrapper<ThirdMessageInfo> condition = new LambdaQueryWrapper<>();
        condition.eq(ThirdMessageInfo::getTenantId, Utils.getTenantId());
        // 关键字
        if (StringUtils.isNotBlank(queryThirdMessageListReq.getKey())) {
            condition.and(x -> x.like(ThirdMessageInfo::getMessageTitle, queryThirdMessageListReq.getKey())
                    .or().like(ThirdMessageInfo::getMessageContent, queryThirdMessageListReq.getKey()));
        }
        // 应用id
        if (StringUtils.isNotBlank(queryThirdMessageListReq.getAppId())) {
            condition.eq(ThirdMessageInfo::getMessageAppId, queryThirdMessageListReq.getAppId());
        }
        // 消息分类
        if (StringUtils.isNotBlank(queryThirdMessageListReq.getMessageTypeCode())) {
            condition.eq(ThirdMessageInfo::getMessageTypeCode, queryThirdMessageListReq.getMessageTypeCode());
        }
        // 开始查询时间
        if (StringUtils.isNotBlank(queryThirdMessageListReq.getStartTimeStr())) {
            condition.ge(ThirdMessageInfo::getMessagePublishTime, queryThirdMessageListReq.getStartTimeStr() + " 00:00:00");
        }
        // 结束查询时间
        if (StringUtils.isNotBlank(queryThirdMessageListReq.getEndTimeStr())) {
            condition.le(ThirdMessageInfo::getMessagePublishTime, queryThirdMessageListReq.getEndTimeStr() + " 23:59:59");
        }
        if (0 == queryThirdMessageListReq.getClientType()){
            List list = new ArrayList();
            list.add(0);
            list.add(2);
            condition.in(ThirdMessageInfo::getSideType,list);
        }
        if (1 == queryThirdMessageListReq.getClientType()){
            List list = new ArrayList();
            list.add(1);
            list.add(2);
            condition.in(ThirdMessageInfo::getSideType,list);
        }
        List<ThirdMessageInfo> thirdMessageInfoList = thirdMessageInfoService.list(condition);
        if (CollectionUtils.isEmpty(thirdMessageInfoList)) {
            return messageInfoList;
        }
        // 查询当前租户的sso配置应用
        if (CollectionUtils.isEmpty(appLinkDTOList)) {
            List<String> appIdList = thirdMessageInfoList.stream().map(ThirdMessageInfo::getMessageAppId).collect(Collectors.toList());
            appLinkDTOList = appLinkService.querySsoConfigListSync(appIdList);
        }
        List<String> appIdList = appLinkDTOList.stream().map(AppLinkDTO::getApplicationAppId).filter(StringUtils::isNotBlank).collect(Collectors.toList());
        thirdMessageInfoList = thirdMessageInfoList.stream().filter(x -> appIdList.contains(x.getMessageAppId())).collect(Collectors.toList());

        // 查询当前鼎捷云用户id在各个应用下归户的三方用户id
        List<AppUserDTO> verifyUserList = queryVerifyUseridList(thirdMessageInfoList);
        if (CollectionUtils.isEmpty(verifyUserList)) {
            return messageInfoList;
        }

        for (AppUserDTO appUserDTO : verifyUserList) {
            for (ThirdMessageInfo x : thirdMessageInfoList) {
                if (x.getMessageAppId().equals(appUserDTO.getAppId())
                        && (Arrays.asList(x.getMessageOwner().split(",")).contains(appUserDTO.getVerifyUserId())
                        || Arrays.asList(x.getMessageOwner().split(",")).contains(Constants.ALL_STAFF_CAN_SEE_FLAG))) {
                    messageInfoList.add(x);
                }
            }
        }
        if (CollectionUtils.isEmpty(messageInfoList)) {
            return messageInfoList;
        }

        // 查询当前登录人已读的消息列表
        List<ThirdMessageReadInfo> readMessageInfoList = queryReadMessageList(verifyUserList);
        if (CollectionUtils.isEmpty(readMessageInfoList)) {
            return messageInfoList;
        }
        Map<String, String> appIdNameMap = appLinkDTOList.stream().filter(x -> StringUtils.isNotBlank(x.getApplicationAppId())).collect(Collectors.toMap(AppLinkDTO::getApplicationAppId, AppLinkDTO::getName, (a, b) -> a));
        for (ThirdMessageInfo messageInfo : messageInfoList) {
            messageInfo.setMessageAppName(appIdNameMap.get(messageInfo.getMessageAppId()));
            for (ThirdMessageReadInfo readInfo : readMessageInfoList) {
                if (readInfo.getMessageAppId().equals(messageInfo.getMessageAppId()) && readInfo.getMessageId().equals(messageInfo.getMessageId())) {
                    messageInfo.setIsRed(false);
                }
            }
        }
        return messageInfoList;
    }

    /**
     * 询当前登录人已读的消息列表
     *
     * @param verifyUserList 应用列表
     * @return 返回
     */
    private List<ThirdMessageReadInfo> queryReadMessageList(List<AppUserDTO> verifyUserList) {
        QueryWrapper<ThirdMessageReadInfo> readCondition = new QueryWrapper<>();
        readCondition.in("message_app_id", verifyUserList.stream().map(AppUserDTO::getAppId).collect(Collectors.toList()));
        readCondition.eq("read_user", Utils.getUserId());
        readCondition.eq("tenant_id", Utils.getTenantId());
        List<ThirdMessageReadInfo> readMessageInfoList = thirdMessageReadInfoService.list(readCondition);
        readMessageInfoList = readMessageInfoList.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(() ->
                new TreeSet<>(Comparator.comparing(o -> o.getMessageAppId() + ";" + o.getMessageId() + ";" + o.getReadUser()))), ArrayList::new));
        return readMessageInfoList;
    }

    /**
     * 查询消息数量，用于页签展示
     *
     * @param queryThirdMessageListReq 请求入参
     * @return 返回对象
     */
    @Override
    public MessageAppNumResp queryMessageCount(QueryThirdMessageListReq queryThirdMessageListReq) {
        MessageAppNumResp numResp = new MessageAppNumResp();
        // 查询消息列表
        List<ThirdMessageInfo> messageInfoList = queryMessageList(queryThirdMessageListReq, null);
        if (CollectionUtils.isEmpty(messageInfoList)) {
            return numResp;
        }
        // 已读数量
        long readCount = messageInfoList.stream().filter(x -> x.getIsRed().equals(false)).count();
        // 未读数量
        long unReadCount = messageInfoList.stream().filter(x -> x.getIsRed().equals(true)).count();
        numResp.setReadNum(readCount);
        numResp.setUnReadNum(unReadCount);
        numResp.setSumNum((long) messageInfoList.size());
        return numResp;
    }
}
